/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <app.h>
#include <efl_extension.h>
#include "$(appName).h"
#include "view_defines.h"
#include "view.h"
#include "types.h"

static struct view_info {
	Evas_Object *win;
	Evas_Object *conform;
	Evas_Object *layout_main;
	Evas_Object *layout_app;
	Evas_Object *layout_paths;
	Evas_Object *app_icon;
	Evas_Object *paths_list;
	Evas_Object *popup_path;
	app_info_query_cb app_info_query_func;
	path_query_cb path_query_func;
	path_files_query_cb path_files_query_func;
} s_info = {
	.win = NULL,
	.conform = NULL,
	.layout_main = NULL,
	.layout_app = NULL,
	.layout_paths = NULL,
	.app_icon = NULL,
	.paths_list = NULL,
	.popup_path = NULL,
	.app_info_query_func = NULL,
	.path_query_func = NULL,
	.path_files_query_func = NULL,
};

static void _delete_win_request_cb(void *data, Evas_Object *obj, void *event_info);
static void _layout_back_cb(void *data, Evas_Object *obj, void *event_info);
static void _toolbar_tab_selected_cb(void *data, Evas_Object *obj, const char *emission, const char *source);
static void _paths_icon_top_clicked_cb(void *data, Evas_Object *obj, const char *emission, const char *source);
static Evas_Object *_get_paths_item_content_cb(void *data, Evas_Object *obj, const char *part);
static void _path_item_selected_cb(void *data, Evas_Object *obj, void *event_info);
static void _path_popup_button_clicked_cb(void *data, Evas_Object *obj, const char *emission, const char *source);
static void _get_app_resource(const char *edj_file_in, char *edj_path_out);
static Eina_Bool _create_main_layout(void);
static Eina_Bool _create_app_view(void);
static Eina_Bool _create_paths_view(void);
static void _create_path_popup(path_type_t *path, int files_count);
static void _delete_path_popup(void);
static bool _is_same_string(const char *s1, const char *s2);
static void _update_application_tab(void);

/**
 * @brief Creates essential objects: window, conformant and layout.
 * @param[in] user_data The user data.
 * @return This function returns EINA_TRUE of the view was created successfully,
 * otherwise EINA_FALSE is returned.
 */
Eina_Bool view_create(void *user_data)
{
	/* Create the window */
	s_info.win = view_create_win(PACKAGE);
	if (s_info.win == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create a window.");
		return EINA_FALSE;
	}

	/* Create the conformant */
	s_info.conform = view_create_conformant(s_info.win);
	if (s_info.conform == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create a conformant");
		return EINA_FALSE;
	}

	if (!_create_main_layout())
		return EINA_FALSE;

	if (!_create_app_view())
		return EINA_FALSE;

	if (!_create_paths_view())
		return EINA_FALSE;

	/* Show the window after main view is set up */
	evas_object_show(s_info.win);

	return EINA_TRUE;
}

/**
 * @brief Creates a basic window named package
 * @param[in] pkg_name Name of the window
 * @return The function returns window object if it was created successfully,
 * otherwise 'NULL' is returned.
 */
Evas_Object *view_create_win(const char *pkg_name)
{
	Evas_Object *win = NULL;

	/*
	 * Window
	 * Create and initialize elm_win.
	 * elm_win is mandatory to manipulate the window.
	 */
	win = elm_win_util_standard_add(pkg_name, pkg_name);
	elm_win_conformant_set(win, EINA_TRUE);
	elm_win_indicator_mode_set(win, ELM_WIN_INDICATOR_SHOW);
	elm_win_indicator_opacity_set(win, ELM_WIN_INDICATOR_OPAQUE);
	elm_win_autodel_set(win, EINA_TRUE);

	evas_object_smart_callback_add(win, "delete,request", _delete_win_request_cb, NULL);

	return win;
}

/**
 * @brief Creates a conformant object for parent object.
 * @param[in] parent The parent object for conformant object.
 * @return The function returns conformant object if it was created successfully,
 * otherwise 'NULL' is returned.
 */
Evas_Object *view_create_conformant(Evas_Object *parent)
{
	Evas_Object *conform = NULL;

	if (!parent) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong input arguments.");
		return NULL;
	}

	conform = elm_conformant_add(parent);
	if (!conform) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_conformant_add() failed.");
		return NULL;
	}

	evas_object_size_hint_weight_set(conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(parent, conform);
	evas_object_show(conform);

	return conform;
}

/**
 * @brief Creates a layout object for parent object based on provided EDJE script.
 * @param[in] parent The parent object for layout object.
 * @param[in] edj_file_name The relative path to the layout EDJE script.
 * @param[in] edj_group The name of the group to be loaded from the EDJE script.
 * @return The function returns layout object if it was created successfully,
 * otherwise 'NULL' is returned.
 */
Evas_Object *view_create_layout(Evas_Object *parent, const char *edj_file_name, const char *edj_group)
{
	char edj_path[PATH_MAX] = {0, };
	Evas_Object *layout = NULL;

	if (!parent || !edj_file_name || !edj_group) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong input arguments.");
		return NULL;
	}

	_get_app_resource(edj_file_name, edj_path);

	layout = elm_layout_add(parent);
	if (!layout) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_layout_add() failed.");
		return NULL;
	}

	if (!elm_layout_file_set(layout, edj_path, edj_group)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_layout_file_set() failed.");
		evas_object_del(layout);
		return NULL;
	}

	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	return layout;
}

/**
 * @brief Creates a layout object and sets it to the parent object based on provided EDJE script.
 * @param[in] parent The parent object for layout object.
 * @param[in] edj_file_name The relative path to the layout EDJE script.
 * @param[in] edj_group The name of the group to be loaded from the EDJE script.
 * @param[in] target_part_name The name of the EDJE part where the layout object has to be set.
 * @return The function returns layout object if it was created successfully,
 * otherwise 'NULL' is returned.
 */
Evas_Object *view_create_set_layout(Evas_Object *parent, const char *edj_file_name, const char *edj_group, const char *target_part_name)
{
	Evas_Object *layout = view_create_layout(parent, edj_file_name, edj_group);
	if (!layout)
		return NULL;

	if (!target_part_name)
		elm_object_content_set(parent, layout);
	else
		elm_object_part_content_set(parent, target_part_name, layout);

	return layout;
}

/**
 * @brief Creates an image object without image file set.
 * In order to set the image file, the elm_image_file_set() function must be used.
 * @param[in] parent The parent object for image object.
 * @param[in] part_name The name of the EDJE part where the image object has to be set.
 * @return The function returns image object if it was created successfully,
 * otherwise 'NULL' is returned.
 */
Evas_Object *view_create_image(Evas_Object *parent, const char *part_name)
{
	Evas_Object *image = elm_image_add(parent);
	if (!image) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_image_add() failed.");
		return NULL;
	}

	elm_image_smooth_set(image, EINA_TRUE);
	elm_image_aspect_fixed_set(image, EINA_TRUE);

	if (part_name)
		elm_object_part_content_set(parent, part_name, image);

	return image;
}

/**
 * @brief Creates a genlist object and sets it to the parent object.
 * @param[in] parent The parent object for genlist object.
 * @param[in] target_part_name The name of the EDJE part where the genlist object has to be set.
 * @return The function returns genlist object if it was created successfully,
 * otherwise 'NULL' is returned.
 */
Evas_Object *view_create_genlist(Evas_Object *parent, const char *target_part_name)
{
	Evas_Object *list = NULL;

	if (!parent || !target_part_name) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong input arguments.");
		return NULL;
	}

	list = elm_genlist_add(parent);
	if (!list) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_genlist_add() failed.");
		return NULL;
	}

	evas_object_size_hint_weight_set(list, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_object_part_content_set(parent, target_part_name, list);

	evas_object_show(list);

	return list;
}

/**
 * @brief Creates an item class for genlist object.
 * @param[in] on_text_get_cb The callback function's handler invoked when the genlist's item has to be redrawn.
 * @return The function returns item class if it was created successfully,
 * otherwise 'NULL' is returned.
 */
Elm_Genlist_Item_Class *view_create_genlist_item_class(Elm_Genlist_Item_Content_Get_Cb on_content_get_cb)
{
	static Elm_Genlist_Item_Class *itc = NULL;

	if (!itc) {
		itc = elm_genlist_item_class_new();
		if (!itc) {
			dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_genlist_item_class_new() failed.");
			return NULL;
		}

		itc->item_style = "full";
		itc->func.text_get = NULL;
		itc->func.content_get = on_content_get_cb;
		itc->func.state_get = NULL;
		itc->func.del = NULL;
	}

	return itc;
}

/**
 * @brief Function sets view callbacks.
 * @param[in] app_info_query_func The callback function to be invoked when the application needs
 * to display its base information (identifier, name, version).
 * @param[in] path_query_func The callback function to be invoked when the application needs
 * to query the supported application paths.
 * @param[in] path_files_query_func The callback function to be invoked when the path information
 * needs to be displayed.
 */
void view_set_callbacks(app_info_query_cb app_info_query_func, path_query_cb path_query_func, path_files_query_cb path_files_query_func)
{
	s_info.app_info_query_func = app_info_query_func;
	s_info.path_query_func = path_query_func;
	s_info.path_files_query_func = path_files_query_func;
}

/**
 * @brief Triggers low battery indicator on and off.
 * @param[in] status The battery status.
 */
void view_update_battery_status(app_event_low_battery_status_e status)
{
	switch ((int)status) {
	case DEVICE_BATTERY_LEVEL_NORMAL:
		edje_object_signal_emit(elm_layout_edje_get(s_info.layout_app), SIGNAL_BATTERY_NORMAL, "");
		break;
	case APP_EVENT_LOW_BATTERY_CRITICAL_LOW:
		edje_object_signal_emit(elm_layout_edje_get(s_info.layout_app), SIGNAL_BATTERY_CRITICAL, "");
		break;
	case APP_EVENT_LOW_BATTERY_POWER_OFF:
		edje_object_signal_emit(elm_layout_edje_get(s_info.layout_app), SIGNAL_BATTERY_OFF, "");
		break;
	default:
		dlog_print(DLOG_ERROR, LOG_TAG, "Function view_update_battery_status() failed. Not handled battery status.");
	}
}

/**
 * @brief Triggers low memory indicator on and off.
 * @param[in] status The memory status.
 */
void view_update_memory_status(app_event_low_memory_status_e status)
{
	switch ((int)status) {
	case APP_EVENT_LOW_MEMORY_NORMAL:
		edje_object_signal_emit(elm_layout_edje_get(s_info.layout_app), SIGNAL_MEMORY_NORMAL, "");
		break;
	case APP_EVENT_LOW_MEMORY_SOFT_WARNING:
		edje_object_signal_emit(elm_layout_edje_get(s_info.layout_app), SIGNAL_MEMORY_SOFT_WARNING, "");
		break;
	case APP_EVENT_LOW_MEMORY_HARD_WARNING:
		edje_object_signal_emit(elm_layout_edje_get(s_info.layout_app), SIGNAL_MEMORY_HARD_WARNING, "");
		break;
	default:
		dlog_print(DLOG_ERROR, LOG_TAG, "Function view_update_memory_status() failed. Not handled memory status.");
	}
}

/**
 * @brief Displays the language setting.
 * @param[in] language The chosen language code.
 */
void view_update_language(const char *language)
{
	if (language)
		elm_object_part_text_set(s_info.layout_app, PART_APP_LANG, language);
}

/**
 * @brief Displays the regional formatting setting code.
 * @param[in] region The chosen regional formatting code.
 */
void view_update_region_format(const char *region)
{
	if (region)
		elm_object_part_text_set(s_info.layout_app, PART_APP_REGION, region);
}

/**
 * @brief Displays the device orientation in textual form and rotation angle.
 * @param[in] status_msg The status message to be displayed.
 */
void view_update_orientation(const char *orientation, int angle)
{
	char angle_str[16] = {0,};

	if (orientation)
		elm_object_part_text_set(s_info.layout_app, PART_APP_ORIENT, orientation);

	snprintf(angle_str, sizeof(angle_str), "%d°", angle);
	elm_object_part_text_set(s_info.layout_app, PART_APP_ANGLE, angle_str);
}

/**
 * @brief Gets the current rotation of the main window.
 * @return This function returns the rotation angle of the main window
 * expressed in degrees.
 */
int view_get_rotation(void)
{
	return elm_win_rotation_get(s_info.win);
}

/**
 * @brief Destroys window and frees its resources.
 */
void view_destroy(void)
{
	if (s_info.win == NULL)
		return;

	_delete_path_popup();

	evas_object_del(s_info.layout_app);
	evas_object_del(s_info.layout_paths);
	evas_object_del(s_info.win);
}

/**
 * @brief Internal callback function invoked when the main window needs to be destroyed.
 * @param[in] data The user data passed to the evas_object_smart_callback_add() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] event_info The structure containing the information on this event.
 */
static void _delete_win_request_cb(void *data, Evas_Object *obj, void *event_info)
{
	ui_app_exit();
}

/**
 * @brief Internal callback function invoked on HW Back button press.
 * @param[in] data The user data passed to the eext_object_event_callback_add() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] event_info The structure containing the information on this event.
 */
static void _layout_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	if (s_info.popup_path)
		_delete_path_popup();
	else
		elm_win_lower(s_info.win);
}

/**
 * @brief Internal callback function invoked on toolbar item click.
 * This callback function responds to the signal emitted from the EDJE script
 * and is responsible for layouts switching when toolbar's tab is changed.
 * @param[in] data The user data passed to the callback attachment function.
 * @param[in] obj The layout object emitting the signal.
 * @param[in] emission The name of the signal emitted.
 * @param[in] source The name of the part emitting the signal.
 */
static void _toolbar_tab_selected_cb(void *data, Evas_Object *obj, const char *emission, const char *source)
{
	Evas_Object *current_page = elm_object_part_content_unset(s_info.layout_main, PART_MAIN_CONTENT);
	if (current_page)
		evas_object_hide(current_page);

	if (_is_same_string(source, PART_MAIN_TOOLBAR_TAB_APPS)) {
		elm_object_part_content_set(s_info.layout_main, PART_MAIN_CONTENT, s_info.layout_app);
		_update_application_tab();
	} else if (_is_same_string(source, PART_MAIN_TOOLBAR_TAB_PATHS)) {
		elm_object_part_content_set(s_info.layout_main, PART_MAIN_CONTENT, s_info.layout_paths);
	}
}

/**
 * @brief Internal callback function invoked on "list top" icon click.
 * This callback function responds to the mouse click signal
 * and is responsible for scrolling the genlist object to the top.
 * @param[in] data The user data passed to the callback attachment function.
 * @param[in] obj The layout object emitting the signal.
 * @param[in] emission The name of the signal emitted.
 * @param[in] source The name of the part emitting the signal.
 */
static void _paths_icon_top_clicked_cb(void *data, Evas_Object *obj, const char *emission, const char *source)
{
	Elm_Object_Item *first = elm_genlist_first_item_get(s_info.paths_list);

	elm_genlist_item_bring_in(first, ELM_GENLIST_ITEM_SCROLLTO_IN);
}

/**
 * @brief Internal callback function invoked on paths list's item selection.
 * This function is responsible for popup window creation with selected path information.
 * @param[in] data The user data passed to the elm_genlist_item_append() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] event_info The structure containing the information on this event.
 */
static void _path_item_selected_cb(void *data, Evas_Object *obj, void *event_info)
{
	int f_count = 0;
	path_type_t *path = (path_type_t *)data;

	if (!path || !path->name || !path->path) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_path_item_selected_cb() failed. Invalid input argument.");
		return;
	}

	if (s_info.path_files_query_func)
		f_count = s_info.path_files_query_func((const char *)path->path);

	_create_path_popup(path, f_count);
}

/**
 * @brief Internal callback function invoked on 'Close' button press from popup window.
 * This function is responsible for closing the popup window.
 * @param[in] data The user data passed to the callback attachment function.
 * @param[in] obj The layout object emitting the signal.
 * @param[in] emission The name of the signal emitted.
 * @param[in] source The name of the part emitting the signal.
 */
static void _path_popup_button_clicked_cb(void *data, Evas_Object *obj, const char *emission, const char *source)
{
	_delete_path_popup();
}

/**
 * @brief Internal callback function invoked on paths list's item display.
 * This function is assigned to the func.content_get handler of genlist item's class structure.
 * It is responsible for items layout creation and its text parts population
 * with provided text content.
 * @param[in] data The user data passed to the elm_genlist_item_append() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] event_info The structure containing the information on this event.
 * @return The layout object to be displayed in item's content area.
 */
static Evas_Object *_get_paths_item_content_cb(void *data, Evas_Object *obj, const char *part)
{
	Evas_Object *layout;
	path_type_t *path = (path_type_t *)data;

	if (_is_same_string(part, "elm.swallow.content")) {
		layout = view_create_layout(obj, EDJ_PATHS, GRP_PATHS_LIST_ITEM);
		if (!layout)
			return NULL;

		evas_object_show(layout);

		if (path && path->name)
			elm_object_part_text_set(layout, PART_PATHS_LIST_ITEM_DESC, path->name);
		else
			elm_object_part_text_set(layout, PART_PATHS_LIST_ITEM_DESC, "Unknown");

		if (path && path->path)
			elm_object_part_text_set(layout, PART_PATHS_LIST_ITEM_PATH, path->path);
		else
			elm_object_part_text_set(layout, PART_PATHS_LIST_ITEM_PATH, "Undefined");

		return layout;
	}

	return NULL;
}

/**
 * @brief Internal function which creates fully qualified path to the provided resource file.
 * @param[in] edj_file_in The file name.
 * @param[out] edj_path_out The fully qualified path to the edj_file_in file.
 */
static void _get_app_resource(const char *edj_file_in, char *edj_path_out)
{
	char *res_path = app_get_resource_path();
	if (res_path) {
		snprintf(edj_path_out, PATH_MAX, "%s%s", res_path, edj_file_in);
		free(res_path);
	}
}

/**
 * @brief Internal function which creates the main panel (toolbar and its tabs without the content).
 * @return The function returns 'EINA_TRUE' if the main layout was created successfully,
 * otherwise 'EINA_FALSE' is returned.
 */
static Eina_Bool _create_main_layout(void)
{
	s_info.layout_main = view_create_set_layout(s_info.conform, EDJ_MAIN, GRP_MAIN, NULL);
	if (!s_info.layout_main)
		return EINA_FALSE;

	eext_object_event_callback_add(s_info.layout_main, EEXT_CALLBACK_BACK, _layout_back_cb, NULL);
	elm_layout_signal_callback_add(s_info.layout_main, SIGNAL_TOOLBAR_TAB_SELECTED, PART_MAIN_TOOLBAR_TABS, _toolbar_tab_selected_cb, NULL);

	edje_object_signal_emit(elm_layout_edje_get(s_info.layout_main), SIGNAL_TOOLBAR_TAB_APPS_SELECT, "");

	return EINA_TRUE;
}

/**
 * @brief Internal function which creates the layout for 'Application' tab.
 * @return The function returns 'EINA_TRUE' if the layout was created successfully,
 * otherwise 'EINA_FALSE' is returned.
 */
static Eina_Bool _create_app_view(void)
{
	s_info.layout_app = view_create_layout(s_info.layout_main, EDJ_APP, GRP_APP);
	if (!s_info.layout_app)
		return EINA_FALSE;

	s_info.app_icon = view_create_image(s_info.layout_app, PART_APP_ICON);
	if (!s_info.app_icon)
		return EINA_FALSE;

	return EINA_TRUE;
}

/**
 * @brief Internal function which creates the layout for 'Paths' tab.
 * During the layout creation, the path_query_cb callback function is invoked
 * to query all the supported resource path information and adds them to the list.
 * @return The function returns 'EINA_TRUE' if the layout was created successfully,
 * otherwise 'EINA_FALSE' is returned.
 */
static Eina_Bool _create_paths_view(void)
{
	path_type_t *path = NULL;
	Elm_Genlist_Item_Class *itc;

	s_info.layout_paths = view_create_layout(s_info.layout_main, EDJ_PATHS, GRP_PATHS);
	if (!s_info.layout_paths)
		return EINA_FALSE;

	elm_layout_signal_callback_add(s_info.layout_paths, "mouse,clicked,1", PART_PATHS_TOP_ICON, _paths_icon_top_clicked_cb, NULL);

	s_info.paths_list = view_create_genlist(s_info.layout_paths, PART_PATHS_CONTENT);
	if (!s_info.paths_list)
		return EINA_FALSE;

	itc = view_create_genlist_item_class(_get_paths_item_content_cb);
	if (!itc)
		return EINA_FALSE;

	if (!s_info.path_query_func) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_create_paths_view() failed. Callback function path_query_cb not assigned.");
		return EINA_FALSE;
	}

	while (s_info.path_query_func(&path))
		if (path)
			elm_genlist_item_append(s_info.paths_list, itc, (void *)path, NULL, ELM_GENLIST_ITEM_NONE, _path_item_selected_cb, (void *)path);

	return EINA_TRUE;
}

/**
 * @brief Internal function which creates a popup window with layout prepared
 * for the path information display.
 * @param[in] path The path information structure.
 * @param[in] files_count The number of files stored in the given directory.
 */
static void _create_path_popup(path_type_t *path, int files_count)
{
	Evas_Object *layout;
	char buff[16] = {0,};

	_delete_path_popup();

	s_info.popup_path = elm_popup_add(s_info.win);
	if (!s_info.popup_path) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_popup_add() failed.");
		return;
	}

	layout = view_create_layout(s_info.popup_path, EDJ_POPUP, GRP_POPUP);
	if (!layout) {
		_delete_path_popup();
		return;
	}

	elm_object_part_content_set(s_info.popup_path, "default", layout);
	eext_object_event_callback_add(layout, EEXT_CALLBACK_BACK, _layout_back_cb, NULL);
	elm_layout_signal_callback_add(layout, SIGNAL_POPUP_BUTTON_PRESSED, PART_POPUP_BUTTON, _path_popup_button_clicked_cb, NULL);

	elm_object_part_text_set(layout, PART_POPUP_TITLE, path->name);
	elm_object_part_text_set(layout, PART_POPUP_PATH, path->path);

	snprintf(buff, sizeof(buff), "%d", files_count);
	elm_object_part_text_set(layout, PART_POPUP_COUNTER, buff);

	evas_object_show(s_info.popup_path);
}

/**
 * @brief Internal function which deletes the popup window created by _create_path_popup() function.
 */
static void _delete_path_popup(void)
{
	evas_object_del(s_info.popup_path);
	s_info.popup_path = NULL;
}

/**
 * @brief Internal function which checks whether two strings are the same.
 * @param[in] s1 The first string.
 * @param[in] s2 The second string.
 * @return This function returns 'true' is both given strings are the same,
 * otherwise 'false' is returned.
 */
static bool _is_same_string(const char *s1, const char *s2)
{
	if (!s1 || !s2)
		return false;

	return (strncmp(s1, s2, strlen(s1)) == 0 && strlen(s1) == strlen(s2));
}

/**
 * @brief Populates the 'Application' tab with application specific information.
 * These information are obtained via the app_info_query_cb callback function,
 * which are: application's name, identifier, version and a path to the icon file.
 */
static void _update_application_tab(void)
{
	app_info_t *app_info = NULL;

	if (!s_info.app_info_query_func) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_update_application_tab() failed. Callback function not assigned.");
		return;
	}

	s_info.app_info_query_func(&app_info);

	if (app_info->name)
		elm_object_part_text_set(s_info.layout_app, PART_APP_NAME, app_info->name);

	if (app_info->id)
		elm_object_part_text_set(s_info.layout_app, PART_APP_ID, app_info->id);

	if (app_info->version)
		elm_object_part_text_set(s_info.layout_app, PART_APP_VER, app_info->version);

	if (app_info->icon_path)
		elm_image_file_set(s_info.app_icon, app_info->icon_path, NULL);
}
