/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <app_manager.h>
#include <notification.h>
#include "$(appName).h"
#include "data.h"

typedef struct _app_info {
	char *app_id;
	app_control_h app_control;
} app_info_t;

static struct data_info {
	Eina_List *apps_list;
	app_info_t *app;
} s_info = {
	.apps_list = NULL,
	.app = NULL,
};

static bool _get_application(int index, app_info_t **app_info);
static bool _create_application(const char *appid, const char *operation, app_info_t **app_info);
static void _free_application(app_info_t **app_info);
static void _free_applications_list(void);
static bool _create_app_control(app_control_h *app_control);
static void _destroy_app_control(app_control_h app_control);
static bool _set_app_control_app_id(app_control_h app_control, const char *app_id);
static bool _set_app_control_operation(app_control_h app_control, const char *operation);

/**
 * @brief Finalization function for data module.
 * This function releases all the memory and resources allocated during the application's run time.
 */
void data_finalize(void)
{
	_free_applications_list();
	_free_application(&s_info.app);
}

/**
 * @brief Creates new application information structure (appId and app_control handle).
 * After successful creation, the structure is added to the list.
 * @param[in] appid The application identifier.
 * @param[in] operation The operation to be performed over an application.
 * @return Function returns 'true' if the application was successfully created
 * and added to the list, otherwise 'false' is returned.
 */
bool data_add_application(const char *appid, const char *operation)
{
	app_info_t *app_info = NULL;

	if (!_create_application(appid, operation, &app_info))
		return false;

	s_info.apps_list = eina_list_append(s_info.apps_list, (void *)app_info);

	return true;
}

/**
 * @brief Creates new application information structure (appId and app_control handle).
 * After successful creation, the structure is stored internally without adding it to the list.
 * @param[in] appid The application identifier.
 * @param[in] operation The operation to be performed over an application.
 * @return Function returns 'true' if the application was successfully created,
 * otherwise 'false' is returned.
 */
bool data_set_application(const char *appid, const char *operation)
{
	_free_application(&s_info.app);

	return _create_application(appid, operation, &s_info.app);
}

/**
 * @brief Sets the extra data field in the app_control handle referenced
 * by the provided index.
 * @param[index] index The list index of application which app_control handle
 * is to be altered with new extra data. If the index is APP_INDEX_EMAIL,
 * then the stand alone application information structure is taken into account
 * instead of the one stored in a list.
 * @param[in] key The key name of the data field to be altered.
 * @param[in] value The value to be assigned to the provided key.
 * @return This function returns 'true' if the extra data was successfully set
 * to the app_control referenced by the provided index. Otherwise 'false' is returned.
 */
bool data_set_application_data(int index, const char *key, const char *value)
{
	int ret;
	app_info_t *app_info = NULL;

	if (!_get_application(index, &app_info))
		return false;

	if (!app_info->app_control) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_control handle not created.");
		return false;
	}

	ret = app_control_add_extra_data(app_info->app_control, key, value);
	if (ret != APP_CONTROL_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "app_control_add_extra_data() failed. Err = %d", ret);

	return (ret == APP_CONTROL_ERROR_NONE);
}

/**
 * @brief Enumerates over all the applications supporting provided operation.
 * @param[in] operation The operation to be matched.
 * @param[in] func_cb The callback function handler which will be invoked
 * for each application matching the app_control prior created.
 * @return Function returns 'EINA_TRUE' on successful enumeration,
 * otherwise 'EINA_FALSE' is returned.
 */
bool data_search_application(const char *operation, app_control_app_matched_cb func_cb)
{
	int ret;
	app_control_h app_control;

	_free_applications_list();

	if (!_create_app_control(&app_control) ||
		!_set_app_control_operation(app_control, operation))
		return false;

	ret = app_control_foreach_app_matched(app_control, func_cb, (void *)operation);
	if (ret != APP_CONTROL_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "app_control_foreach_app_matched() failed. Err = %d", ret);

	_destroy_app_control(app_control);

	return (ret == APP_CONTROL_ERROR_NONE);
}

/**
 * @brief Sends the launch request with prior created app_control handle.
 * If the app_control handle was not prior created with data_app_control_create() function,
 * then this function fails.
 * @param[in] index The index of application to be launched.
 * If the index is APP_INDEX_EMAIL, then the stand alone application information structure
 * is taken into account instead of the one stored in a list.
 * @param[in] func_cb The callback function handler which will be invoked
 * in a response to launch request sent to the external application.
 * This callback function reports the result of performed operation over an external application.
 * @return Function returns 'true' on successful request sent,
 * otherwise 'false' is returned.
 */
bool data_launch_application(int index, app_control_reply_cb func_cb)
{
	int ret;
	app_info_t *app_info = NULL;

	if (!_get_application(index, &app_info))
		return false;

	if (!app_info->app_control) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_control handle not created.");
		return false;
	}

	ret = app_control_set_launch_mode(app_info->app_control, APP_CONTROL_LAUNCH_MODE_GROUP);
	if (ret != APP_CONTROL_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_control_set_launch_mode() failed. Err = %d", ret);
		return false;
	}

	ret = app_control_send_launch_request(app_info->app_control, func_cb, NULL);
	if (ret != APP_CONTROL_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_control_send_launch_request() failed. Err = %d", ret);

		if (ret == APP_CONTROL_ERROR_APP_NOT_FOUND) {
			char *app_id;
			int ret_get_app_id = app_control_get_app_id(app_info->app_control, &app_id);
			if (ret_get_app_id == APP_CONTROL_ERROR_NONE) {
				char err_msg[128] = "App not found: ";
				strncat(err_msg, app_id, strlen(app_id));
				free(app_id);

				int noti_ret = notification_status_message_post(err_msg);
				if (noti_ret != NOTIFICATION_ERROR_NONE) {
					dlog_print(DLOG_ERROR, LOG_TAG, "notification_status_message_post() failed. Err = %d", noti_ret);
				}
			} else {
				dlog_print(DLOG_ERROR, LOG_TAG, "app_control_get_app_id() failed. Err = %d", ret_get_app_id);
			}
		}
	} else {
		dlog_print(DLOG_INFO, LOG_TAG, "Launch request sent to the %s application.", app_info->app_id);
	}

	return (ret == APP_CONTROL_ERROR_NONE);
}

/**
 * @brief Sends the terminate request with prior created app_control handle.
 * If the app_control handle was not prior created with data_app_control_create() function,
 * then this function fails. Only those applications can be terminated, which were
 * launched with the same app_control handle. For this reason, the app_control handle
 * shall not be destroyed after an external application is launched if it is assumed
 * to be terminated later.
 * @param[in] index The index of application to be terminated.
 * @return Function returns 'true' on successful request sent,
 * otherwise 'false' is returned.
 */
bool data_terminate_application(int index)
{
	int ret;
	app_info_t *app_info = NULL;

	if (!_get_application(index, &app_info))
		return false;

	if (!app_info->app_control) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_control handle not created.");
		return false;
	}

	ret = app_control_send_terminate_request(app_info->app_control);
	if (ret != APP_CONTROL_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "app_control_send_terminate_request() failed. Err = %d", ret);
	else
		dlog_print(DLOG_INFO, LOG_TAG, "Terminate request sent to the %s application.", app_info->app_id);

	return (ret == APP_CONTROL_ERROR_NONE);
}

/**
 * @brief Function obtains the absolute path to the application's icon file.
 * @param[out] appid The appid of the application which icon's path has to be obtained.
 * @return The path to the application's icon file if the function succeeds,
 * or NULL if the function fails.
 */
char *data_get_icon_path(const char *appid)
{
	int ret;
	app_info_h app_info = NULL;
	char *icon_path = NULL;

	ret = app_info_create(appid, &app_info);
	if (ret != APP_MANAGER_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_info_create() failed. Err = %d.", ret);
		return NULL;
	}

	ret = app_info_get_icon(app_info, &icon_path);
	if (ret != APP_MANAGER_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "app_info_get_icon() failed. Err = %d.", ret);

	app_info_destroy(app_info);

	return (ret == APP_MANAGER_ERROR_NONE ? icon_path : NULL);
}

/**
 * @brief Internal function which gets the application information structure referenced by its index.
 * @param[in] index The index of the application structure to be returned.
 * If the index is APP_INDEX_EMAIL, then the stand alone application information structure
 * is taken into account instead of the one stored in a list.
 * @param[out] app_info The application information structure referenced by the index provided.
 * @return This function returns 'true' if the application information structure
 * was successfully obtained, otherwise 'false' is returned.
 */
static bool _get_application(int index, app_info_t **app_info)
{
	if (index == APP_INDEX_EMAIL) {
		*app_info = s_info.app;
		return (*app_info != NULL);
	}

	if (!s_info.apps_list) {
		dlog_print(DLOG_ERROR, LOG_TAG, "data_get_application() failed. Applications list is empty.");
		return false;
	}

	*app_info = (app_info_t *)eina_list_nth((const Eina_List *)s_info.apps_list, (unsigned int)index);
	if (!(*app_info))
		dlog_print(DLOG_ERROR, LOG_TAG, "eina_list_nth() failed.");

	return (*app_info != NULL);
}

/**
 * @brief Creates the application structure.
 * @param[in] appid The application identifier.
 * @param[in] operation The operation to be performed over an application.
 * @param[out] app_info The application's information structure.
 * @return This function returns 'true' if the application information structure
 * was successfully created, otherwise 'false' is returned.
 */
static bool _create_application(const char *appid, const char *operation, app_info_t **app_info)
{
	*app_info = (app_info_t *)calloc(1, sizeof(app_info_t));
	if (!*app_info) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_create_application() failed. Memory allocation error.");
		return false;
	}

	if (!_create_app_control(&(*app_info)->app_control)) {
		free(*app_info);
		*app_info = NULL;
		return false;
	}

	if (!_set_app_control_app_id((*app_info)->app_control, appid) ||
		!_set_app_control_operation((*app_info)->app_control, operation)) {
		_destroy_app_control((*app_info)->app_control);
		free(*app_info);
		*app_info = NULL;
		return false;
	}

	(*app_info)->app_id = strdup(appid);

	return true;
}

/**
 * @brief Release the application structure.
 */
static void _free_application(app_info_t **app_info)
{
	if (!*app_info)
		return;

	free((*app_info)->app_id);
	_destroy_app_control((*app_info)->app_control);
	free(*app_info);

	*app_info = NULL;
}

/**
 * @brief Release list of stored application details.
 */
static void _free_applications_list(void)
{
	app_info_t *app_info = NULL;

	EINA_LIST_FREE(s_info.apps_list, app_info)
		_free_application(&app_info);

	s_info.apps_list = NULL;
}

/**
 * @brief Internal function which creates new app_control handle.
 * @param[out] app_control The newly created app_control handle.
 * @return This function returns 'true' is the app_control handle was created successfully,
 * otherwise 'false' is returned.
 */
static bool _create_app_control(app_control_h *app_control)
{
	int ret = app_control_create(app_control);
	if (ret != APP_CONTROL_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "app_control_create() failed. Err = %d", ret);

	return (ret == APP_CONTROL_ERROR_NONE);
}

/**
 * @brief Internal function which destroys app_control handle.
 * If the app_control handle is already destroyed then nothing happens.
 * @param[in] app_control The app_control handle to be destroyed.
 */
static void _destroy_app_control(app_control_h app_control)
{
	int ret;

	if (!app_control)
		return;

	ret = app_control_destroy(app_control);
	if (ret != APP_CONTROL_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "app_control_destroy() failed. Err = %d", ret);
}

/**
 * @brief Internal function which sets the appId for prior created app_control handle.
 * If the app_control handle was not prior created with data_app_control_create() function,
 * then this function fails.
 * @param[in] app_control The app_control handle for which the appId is to be set.
 * @param[in] app_id The appId to be set to the app_control handle for later operation execution.
 * @return Function returns 'true' on successful appId setting,
 * otherwise 'false' is returned.
 */
static bool _set_app_control_app_id(app_control_h app_control, const char *app_id)
{
	int ret;

	if (!app_control) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_control handle not created.");
		return false;
	}

	ret = app_control_set_app_id(app_control, app_id);
	if (ret != APP_CONTROL_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "app_control_set_app_id() failed. Err = %d", ret);

	return (ret == APP_CONTROL_ERROR_NONE);
}

/**
 * @brief Internal function which sets the operation for prior created app_control handle.
 * If the app_control handle was not prior created with data_app_control_create() function,
 * then this function fails.
 * @param[in] app_control The app_control handle for which the operation is to be set.
 * @param[in] operation The operation name to be performed with over an external application.
 * @return Function returns 'true' on successful operation setting,
 * otherwise 'false' is returned.
 */
static bool _set_app_control_operation(app_control_h app_control, const char *operation)
{
	int ret;

	if (!app_control) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_control handle not created.");
		return false;
	}

	ret = app_control_set_operation(app_control, operation);
	if (ret != APP_CONTROL_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "app_control_set_operation() failed. Err = %d", ret);

	return (ret == APP_CONTROL_ERROR_NONE);
}
