/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <tizen.h>
#include <Elementary.h>
#include "$(appName).h"
#include "data.h"
#include "view.h"

#define COMPOSE_OP_RECIPIENT "example@tizen.org"
#define COMPOSE_OP_RECIPIENT_CC "cc_example@tizen.org"
#define COMPOSE_OP_RECIPIENT_BCC "bcc_example@tizen.org"
#define COMPOSE_OP_SUBJECT "Example message subject"
#define COMPOSE_OP_MESSAGE "Example message text"

#define EMAIL_DEFAULT_MESSAGE_SUBJECT "Message from appcontrol"
#define EMAIL_APP_ID "org.tizen.email-composer"

static void _search_application_cb(const char *operation);
static void _execute_application_cb(int app_index, const char *operation);
static void _terminate_application_cb(int app_index);
static void _send_message_cb(const char *address_to, const char *message);

/**
 * @brief Hook to take necessary actions before main event loop starts.
 * Initialize UI resources and application's data.
 * If this function returns true, the main loop of application starts.
 * If this function returns false, the application is terminated.
 */
static bool app_create(void *user_data)
{
	view_set_callbacks(_search_application_cb,
						_execute_application_cb,
						_terminate_application_cb,
						_send_message_cb);

	return view_create(NULL);
}

/**
 * @brief This callback function is called when another application
 * sends a launch request to the application.
 */
static void app_control(app_control_h app_control, void *user_data)
{
	/* Handle the launch request. */
}

/**
 * @brief This callback function is called each time
 * the application is completely obscured by another application
 * and becomes invisible to the user.
 */
static void app_pause(void *user_data)
{
	/* Take necessary actions when application becomes invisible. */
}

/**
 * @brief This callback function is called each time
 * the application becomes visible to the user.
 */
static void app_resume(void *user_data)
{
	/* Take necessary actions when application becomes visible. */
}

/**
 * @brief This callback function is called once after the main loop of the application exits.
 */
static void app_terminate(void *user_data)
{
	/* Release all resources. */
	data_finalize();
	view_destroy();
}

/**
 * @brief This function will be called when the language is changed.
 */
static void ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/* APP_EVENT_LANGUAGE_CHANGED */
	char *locale = NULL;

	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);

	if (locale != NULL) {
		elm_language_set(locale);
		free(locale);
	}
	return;
}

/**
 * @brief Main function of the application.
 */
int main(int argc, char *argv[])
{
	int ret;

	ui_app_lifecycle_callback_s event_callback = {0, };
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	/*
	 * If you want to handle more events,
	 * please check the application life cycle guide.
	 */
	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, NULL);

	ret = ui_app_main(argc, argv, &event_callback, NULL);
	if (ret != APP_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "ui_app_main() failed. err = %d", ret);

	return ret;
}

/**
 * @brief Callback function invoked for each application matching the created app_control.
 * Each found application's identifier is added to list view.
 * @param[in] app_control The application control handle.
 * @param[in] appid The identifier of matched application.
 * @param[in] user_data The data passed to the app_control_foreach_app_matched() function.
 * @return The function returns 'true' to continue matching applications enumeration,
 * otherwise 'false' is returned.
 */
static bool _application_matched_cb(app_control_h app_control, const char *appid, void *user_data)
{
	char *operation = (char *)user_data;
	char *icon_path = data_get_icon_path(appid);

	if (data_add_application(appid, operation))
		view_add_application(appid, (const char *)icon_path);

	free(icon_path);

	return true;
}

/**
 * @brief Callback function invoked in reply to launch request sent via app_control.
 * @param[in] request The application control handle of a request.
 * @param[in] reply The application control handle of a reply.
 * @param[in] result The launch result.
 * @param[in] user_data The data passed to the app_control_send_launch_request() function.
 */
static void _app_control_launch_reply_cb(app_control_h request, app_control_h reply, app_control_result_e result, void *user_data)
{
	switch (result) {
	case APP_CONTROL_RESULT_APP_STARTED:
		dlog_print(DLOG_INFO, LOG_TAG, "Launch request result: Application started");
		break;
	case APP_CONTROL_RESULT_SUCCEEDED:
		dlog_print(DLOG_INFO, LOG_TAG, "Launch request result: Operation succeeded");
		break;
	case APP_CONTROL_RESULT_FAILED:
		dlog_print(DLOG_INFO, LOG_TAG, "Launch request result: Operation failed by the caller");
		break;
	case APP_CONTROL_RESULT_CANCELED:
		dlog_print(DLOG_INFO, LOG_TAG, "Launch request result: Operation canceled by the framework");
		break;
	default:
		dlog_print(DLOG_INFO, LOG_TAG, "Launch request result: unknown result type");
	}
}

/**
 * @brief Callback function invoked on operation radio button click from the view module.
 * This function performs searching for all the applications which can handle provided operation.
 * @param[in] operation The name of the selected operation.
 */
static void _search_application_cb(const char *operation)
{
	data_search_application(operation, _application_matched_cb);
}

/**
 * @brief Callback function invoked on "Execute" button click from the view module.
 * This function launches the application specified by index with provided operation.
 * @param[in] app_index The index of prior found application to execute.
 * @param[in] operation The name of the selected operation.
 */
static void _execute_application_cb(int app_index, const char *operation)
{
	data_set_application_data(app_index, APP_CONTROL_DATA_TO, COMPOSE_OP_RECIPIENT);
	data_set_application_data(app_index, APP_CONTROL_DATA_CC, COMPOSE_OP_RECIPIENT_CC);
	data_set_application_data(app_index, APP_CONTROL_DATA_BCC, COMPOSE_OP_RECIPIENT_BCC);
	data_set_application_data(app_index, APP_CONTROL_DATA_SUBJECT, COMPOSE_OP_SUBJECT);
	data_set_application_data(app_index, APP_CONTROL_DATA_TEXT, COMPOSE_OP_MESSAGE);

	data_launch_application(app_index, _app_control_launch_reply_cb);
}

/**
 * @brief Callback function invoked on "Kill" button click from the view module.
 * This function terminates the application prior launched.
 */
static void _terminate_application_cb(int app_index)
{
	data_terminate_application(app_index);
}

/**
 * @brief Callback function invoked on "Send" button click from the view module.
 * This function sends an default email to the provided email address.
 * The email's subject is defined internally.
 * After the email is sent, the _app_control_launch_reply_cb() callback function will be called
 * to notify the result of send operation.
 * @param[in] address_to The email's destination address.
 * @param[in] message The email's message body to be send.
 */
static void _send_message_cb(const char *address_to, const char *message)
{
	if (!data_set_application(EMAIL_APP_ID, APP_CONTROL_OPERATION_COMPOSE))
		return;

	data_set_application_data(APP_INDEX_EMAIL, APP_CONTROL_DATA_TO, address_to);
	data_set_application_data(APP_INDEX_EMAIL, APP_CONTROL_DATA_SUBJECT, EMAIL_DEFAULT_MESSAGE_SUBJECT);
	data_set_application_data(APP_INDEX_EMAIL, APP_CONTROL_DATA_TEXT, message);

	data_launch_application(APP_INDEX_EMAIL, _app_control_launch_reply_cb);
}
