/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "$(appName).h"
#include "view_defines.h"
#include "view.h"

#define APPS_LIST_ITEM_WIDTH 360
#define APPS_LIST_ITEM_HEIGHT 219

#define DONE_BT_CODE 36

#define DEFAULT_TEXT_MESSAGE "Dear Developer,<br><br>This is the default message sent from<br>"\
								"appcontrol sample application.<br>"\
								"Feel free to modify this text message in email composer.<br><br>"\
								"Best Regards."
#define DEFAULT_TEXT_MESSAGE_EMAIL "Dear Developer,\n\nThis is the default message sent from\n"\
								"appcontrol sample application.\n"\
								"Feel free to modify this text message in email composer.\n\n"\
								"Best Regards."

typedef enum {OP_1 = 0, OP_2, OP_3, OP_MAX} operation_e;

typedef struct _operation {
	const char *caption;
	const char *operation;
} operation_t;

typedef struct _list_item {
	char *caption;
	char *icon_path;
	bool gray_scale;
	bool padding;
	bool enabled;
	Elm_Object_Item *item;
} list_item_t;

static struct view_info {
	Evas_Object *win;
	Evas_Object *conform;
	Evas_Object *main_layout;
	Evas_Object *apps_grid;
	Elm_Gengrid_Item_Class *itc;
	Elm_Object_Item *fake_item;
	Elm_Theme *grid_theme;
	operation_t operations[OP_MAX];
	operation_e selected_op;
	search_cb search_func_cb;
	execute_cb execute_func_cb;
	terminate_cb terminate_func_cb;
	send_cb send_func_cb;
} s_info = {
	.win = NULL,
	.conform = NULL,
	.main_layout = NULL,
	.apps_grid = NULL,
	.itc = NULL,
	.fake_item = NULL,
	.grid_theme = NULL,
	.operations = { [OP_1] = { .caption = "View operation", .operation = APP_CONTROL_OPERATION_VIEW, },
					[OP_2] = { .caption = "Pick operation", .operation = APP_CONTROL_OPERATION_PICK, },
					[OP_3] = { .caption = "Compose operation", .operation = APP_CONTROL_OPERATION_COMPOSE, }, },
	.selected_op = OP_MAX,
	.search_func_cb = NULL,
	.execute_func_cb = NULL,
	.terminate_func_cb = NULL,
	.send_func_cb = NULL,
};

static void _window_delete_request_cb(void *data, Evas_Object *obj, void *event_info);
static void _window_back_cb(void *data, Evas_Object *obj, void *event_info);
static void _operation_radio_selected_cb(void *data, Evas_Object *obj, const char *emission, const char *source);
static void _execute_button_click_cb(void *data, Evas_Object *obj, const char *emission, const char *source);
static void _kill_button_click_cb(void *data, Evas_Object *obj, const char *emission, const char *source);
static void _send_button_click_cb(void *data, Evas_Object *obj, const char *emission, const char *source);
static void _address_focused_cb(void *data, Evas_Object *obj, void *event_info);
static void _address_unfocused_cb(void *data, Evas_Object *obj, void *event_info);
static void _input_panel_key_down_cb(void *data, Evas *e, Evas_Object *obj, void *event_info);
static Evas_Object *_get_app_grid_content_cb(void *data, Evas_Object *obj, const char *part);
static char *_get_app_grid_text_cb(void *data, Evas_Object *obj, const char *part);
static void _del_app_grid_item_cb(void *data, Evas_Object *obj);
static void _get_app_resource(const char *edj_file_in, char *edj_path_out);
static Eina_Bool _create_main_layout(Evas_Object *parent, const char *part_name);
static Elm_Gengrid_Item_Class *_create_gengrid_item_class(void);
static void _set_image_grayscale(Evas_Object *image);
static bool _is_same_string(const char *s1, const char *s2);
static bool _create_list_item(const char *caption, const char *icon_path, bool enabled, list_item_t **li);
static bool _set_list_item_enabled(Elm_Object_Item *item, bool state);

/**
 * @brief Creates essential objects: window, conformant and layout.
 * @return Function returns EINA_TRUE on successful view creation,
 * otherwise EINA_FALSE is returned.
 */
Eina_Bool view_create(void *user_data)
{
	Evas_Object *scroller;

	/* Create the window */
	s_info.win = view_create_win(PACKAGE);
	if (s_info.win == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create a window.");
		return EINA_FALSE;
	}

	/* Create the conformant */
	s_info.conform = view_create_conformant(s_info.win);
	if (s_info.conform == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create a conformant");
		return EINA_FALSE;
	}

	scroller = view_create_scroller(s_info.conform, "default");
	if (!scroller)
		return EINA_FALSE;

	if (!_create_main_layout(scroller, "default"))
		return EINA_FALSE;

	/* Show the window after main view is set up */
	evas_object_show(s_info.win);

	return EINA_TRUE;
}

/**
 * @brief Creates a basic window named package
 * @param[in] pkg_name Name of the window
 * @return Function returns window object on successful creation,
 * otherwise NULL is returned.
 */
Evas_Object *view_create_win(const char *pkg_name)
{
	Evas_Object *win = NULL;

	/*
	 * Window
	 * Create and initialize elm_win.
	 * elm_win is mandatory to manipulate the window.
	 */
	win = elm_win_util_standard_add(pkg_name, pkg_name);
	elm_win_conformant_set(win, EINA_TRUE);
	elm_win_autodel_set(win, EINA_TRUE);
	elm_win_indicator_mode_set(win, ELM_WIN_INDICATOR_SHOW);

	evas_object_smart_callback_add(win, "delete,request", _window_delete_request_cb, NULL);
	eext_object_event_callback_add(win, EEXT_CALLBACK_BACK, _window_back_cb, NULL);

	return win;
}

/**
 * @brief Creates a conformant without indicator for mobile app.
 * @param[in] win The object to which you want to set this conformant. Conformant is mandatory for base GUI to have proper size.
 * @return Function returns conformant object on successful creation,
 * otherwise NULL is returned.
 */
Evas_Object *view_create_conformant(Evas_Object *win)
{
	/*
	 * Conformant
	 * Create and initialize elm_conformant.
	 * elm_conformant is mandatory for base GUI to have proper size
	 * when indicator or virtual keypad is visible.
	 */
	Evas_Object *conform = NULL;

	if (win == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "window is NULL.");
		return NULL;
	}

	conform = elm_conformant_add(win);
	evas_object_size_hint_weight_set(conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(win, conform);

	evas_object_show(conform);
	elm_object_signal_emit(conform, "elm,state,indicator,overlap", "elm");

	return conform;
}

/**
 * @brief Creates a layout to target parent object with edje file.
 * @param[in] parent The object to which you want to add this layout.
 * @param[in] file_path File path of EDJ file will be used.
 * @param[in] group_name Name of group in EDJ you want to set to.
 * @param[in] part_name Name of the part in EDJ you want to set to.
 * @return Function returns layout object on successful creation,
 * otherwise NULL is returned.
 */
Evas_Object *view_create_layout(Evas_Object *parent, const char *file_path, const char *group_name, const char *part_name)
{
	char edj_path[PATH_MAX] = {0, };
	Evas_Object *layout = NULL;

	if (!group_name) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Cannot create layout, the group name is empty.");
		return NULL;
	}

	_get_app_resource(file_path, edj_path);

	layout = elm_layout_add(parent);
	if (!layout) {
		dlog_print(DLOG_ERROR, LOG_TAG, "elm_layout_add() failed.");
		return NULL;
	}

	if (!elm_layout_file_set(layout, edj_path, group_name)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "elm_layout_file_set() failed.");
		return NULL;
	}

	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	if (part_name)
		elm_object_part_content_set(parent, part_name, layout);

	return layout;
}

/**
 * @brief Creates a scroller object and sets it to the parent object.
 * @param[in] parent The parent object for scroller object.
 * @param[in] part_name The name of the EDJE part where the scroller object has to be set.
 * @return The function returns scroller object if it was created successfully,
 * otherwise 'NULL' is returned.
 */
Evas_Object *view_create_scroller(Evas_Object *parent, const char *part_name)
{
	Evas_Object *scroller = elm_scroller_add(parent);
	if (!scroller) {
		dlog_print(DLOG_ERROR, LOG_TAG, "view_create_scroller() failed.");
		return NULL;
	}

	evas_object_size_hint_weight_set(scroller, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_scroller_bounce_set(scroller, EINA_FALSE, EINA_TRUE);
	elm_scroller_policy_set(scroller, ELM_SCROLLER_POLICY_OFF, ELM_SCROLLER_POLICY_OFF);
	elm_object_part_content_set(parent, part_name, scroller);
	evas_object_show(scroller);

	return scroller;
}

/**
 * @brief Creates a gengrid object and sets it to the parent object.
 * @param[in] parent The parent object for gengrid object.
 * @param[in] part_name The name of the EDJE part where the gengrid object has to be set.
 * @return The function returns gengrid object if it was created successfully,
 * otherwise 'NULL' is returned.
 */
Evas_Object *view_create_gengrid(Evas_Object *parent, const char *part_name)
{
	Evas_Object *grid = NULL;
	char theme_path[PATH_MAX] = {0, };

	grid = elm_gengrid_add(parent);
	if (!grid) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_gengrid_add() failed.");
		return NULL;
	}

	evas_object_size_hint_weight_set(grid, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_gengrid_align_set(grid, 0.01, 0.5);
	elm_gengrid_item_size_set(grid, APPS_LIST_ITEM_WIDTH, APPS_LIST_ITEM_HEIGHT);
	elm_gengrid_horizontal_set(grid, EINA_TRUE);
	elm_scroller_page_relative_set(grid, 0.5, 0.0);
	elm_scroller_content_min_limit(grid, EINA_FALSE, EINA_FALSE);
	elm_scroller_page_scroll_limit_set(grid, 1, 0);
	elm_scroller_policy_set(grid, ELM_SCROLLER_POLICY_OFF, ELM_SCROLLER_POLICY_OFF);
	elm_object_part_content_set(parent, part_name, grid);

	evas_object_show(grid);

	if (!s_info.grid_theme) {
		_get_app_resource(EDJ_APPS_ITEM_THEME, theme_path);

		s_info.grid_theme = elm_theme_new();
		elm_theme_ref_set(s_info.grid_theme, NULL);
		elm_theme_overlay_add(NULL, theme_path);
	}

	return grid;
}

/**
 * @brief Creates an image object without image file set.
 * In order to set the image file, the elm_image_file_set() function must be used.
 * @param[in] parent The parent object for image object.
 * @param[in] file_path The absolute path to the image file to be displayed.
 * @return The function returns image object if it was created successfully,
 * otherwise 'NULL' is returned.
 */
Evas_Object *view_create_image(Evas_Object *parent, const char *file_path)
{
	Evas_Object *image = elm_image_add(parent);
	if (!image) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_image_add() failed.");
		return NULL;
	}

	elm_image_smooth_set(image, EINA_TRUE);
	elm_image_aspect_fixed_set(image, EINA_TRUE);

	if (file_path)
		elm_image_file_set(image, file_path, NULL);

	return image;
}

/**
 * @brief Creates entry object.
 * @param[in] parent The parent object for the entry.
 * @param[in] edj_part_name The name of the part where the entry is to be embedded.
 * @return Function returns entry object on successful creation,
 * otherwise NULL is returned.
 */
Evas_Object *view_create_entry(Evas_Object *parent, const char *edj_part_name)
{
	Evas_Object *entry = elm_entry_add(parent);
	if (!entry) {
		dlog_print(DLOG_ERROR, LOG_TAG, "elm_entry_add() failed.");
		return NULL;
	}

	evas_object_size_hint_weight_set(entry, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_entry_line_wrap_set(entry, ELM_WRAP_NONE);
	elm_entry_single_line_set(entry, EINA_TRUE);
	elm_entry_scrollable_set(entry, EINA_TRUE);
	elm_entry_input_panel_layout_set(entry, ELM_INPUT_PANEL_LAYOUT_EMAIL);
	elm_entry_input_panel_return_key_type_set(entry, ELM_INPUT_PANEL_RETURN_KEY_TYPE_DONE);
	elm_object_part_content_set(parent, edj_part_name, entry);
	evas_object_show(entry);

	return entry;
}

/**
 * @brief Sets the callbacks invoked from the view module.
 * @param[in] search_func_cb The callback function invoked on operation radio button click.
 * @param[in] execute_func_cb The callback function invoked on "Execute" button clicked.
 * @param[in] terminate_func_cb The callback function invoked on "Kill" button clicked.
 * @param[in] send_func_cb The callback function invoked on "Send" button clicked.
 */
void view_set_callbacks(search_cb search_func_cb, execute_cb execute_func_cb, terminate_cb terminate_func_cb, send_cb send_func_cb)
{
	s_info.search_func_cb = search_func_cb;
	s_info.execute_func_cb = execute_func_cb;
	s_info.terminate_func_cb = terminate_func_cb;
	s_info.send_func_cb = send_func_cb;
}

/**
 * @brief Adds an application to the grid.
 * @param[in] caption The text to be displayed aside the application's icon.
 * @param[in] icon_path The path to the icon to be displayed aside the caption.
 * @return Function returns 'EINA_TRUE' if an application was added successfully,
 * otherwise 'EINA_FALSE' is returned.
 */
Eina_Bool view_add_application(const char *caption, const char *icon_path)
{
	int items_count;
	list_item_t *list_item;

	if (!caption || !icon_path) {
		dlog_print(DLOG_ERROR, LOG_TAG, "view_add_application() failed. Invalid input argument.");
		return EINA_FALSE;
	}

	items_count = elm_gengrid_items_count(s_info.apps_grid);

	if (items_count == 0)
		if (_create_list_item(NULL, NULL, false, &list_item))
			s_info.fake_item = elm_gengrid_item_append(s_info.apps_grid, s_info.itc, (void *)list_item, NULL, NULL);

	if (_create_list_item(caption, icon_path, (bool)(items_count == 0), &list_item))
		list_item->item = elm_gengrid_item_insert_before(s_info.apps_grid, s_info.itc, (void *)list_item, s_info.fake_item, NULL, NULL);

	elm_gengrid_item_show(elm_gengrid_first_item_get(s_info.apps_grid), ELM_GENGRID_ITEM_SCROLLTO_TOP);

	return EINA_TRUE;
}

/**
 * @brief Destroys window and frees its resources.
 */
void view_destroy(void)
{
	if (s_info.win == NULL)
		return;

	elm_gengrid_item_class_free(s_info.itc);
	elm_theme_free(s_info.grid_theme);
	evas_object_del(s_info.win);
}

/**
 * @brief Callback function to be invoked on main window close.
 * @param[in] data Custom data passed to the callback attachment function.
 * @param[in] obj Object to which the handler is attached.
 * @param[in] event_info The information on the event occurred.
 */
static void _window_delete_request_cb(void *data, Evas_Object *obj, void *event_info)
{
	ui_app_exit();
}

/**
 * @brief Callback function to be invoked on HW Back button press. It lowers the main application's window.
 * @param[in] data Custom data passed to the callback attachment function.
 * @param[in] obj Object to which the handler is attached.
 * @param[in] event_info The information on the event occurred.
 */
static void _window_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	elm_win_lower(s_info.win);
}

/**
 * @brief Internal callback function invoked on operation radio button selection.
 * This callback function responds to the signal emitted from the EDJE script.
 * Internally, it searches for applications matching the selected operation.
 * @param[in] data The user data passed to the callback attachment function.
 * @param[in] obj The layout object emitting the signal.
 * @param[in] emission The name of the signal emitted.
 * @param[in] source The name of the part emitting the signal.
 */
static void _operation_radio_selected_cb(void *data, Evas_Object *obj, const char *emission, const char *source)
{
	if (!s_info.search_func_cb) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_operation_radio_selected_cb() function failed: search_func_cb callback function is not assigned");
		return;
	}

	elm_gengrid_clear(s_info.apps_grid);

	if (_is_same_string(source, PART_OPERATION_OPER_1)) {
		s_info.search_func_cb(s_info.operations[OP_1].operation);
		s_info.selected_op = OP_1;
	} else if (_is_same_string(source, PART_OPERATION_OPER_2)) {
		s_info.search_func_cb(s_info.operations[OP_2].operation);
		s_info.selected_op = OP_2;
	} else { if (_is_same_string(source, PART_OPERATION_OPER_3))
		s_info.search_func_cb(s_info.operations[OP_3].operation);
		s_info.selected_op = OP_3;
	}
}

/**
 * @brief Callback function to be invoked on "Execute" button click.
 * It executes selected application taking into account selected operation.
 * @param[in] data The user data passed to the callback attachment function.
 * @param[in] obj The layout object emitting the signal.
 * @param[in] emission The name of the signal emitted.
 * @param[in] source The name of the part emitting the signal.
 */
static void _execute_button_click_cb(void *data, Evas_Object *obj, const char *emission, const char *source)
{
	int selected_app = -1;

	if (s_info.selected_op == OP_MAX)
		return;

	if (!s_info.execute_func_cb)
		return;

	elm_scroller_current_page_get(s_info.apps_grid, &selected_app, NULL);

	s_info.execute_func_cb(selected_app, s_info.operations[s_info.selected_op].operation);
}

/**
 * @brief Callback function to be invoked on "Kill" button click.
 * It terminates recently executed application.
 * @param[in] data The user data passed to the callback attachment function.
 * @param[in] obj The layout object emitting the signal.
 * @param[in] emission The name of the signal emitted.
 * @param[in] source The name of the part emitting the signal.
 */
static void _kill_button_click_cb(void *data, Evas_Object *obj, const char *emission, const char *source)
{
	int selected_app = -1;

	if (!s_info.terminate_func_cb)
		return;

	elm_scroller_current_page_get(s_info.apps_grid, &selected_app, NULL);

	s_info.terminate_func_cb(selected_app);
}

/**
 * @brief Callback function to be invoked on "Send" button click.
 * It sends prior created message to the recipient.
 * @param[in] data The user data passed to the callback attachment function.
 * @param[in] obj The layout object emitting the signal.
 * @param[in] emission The name of the signal emitted.
 * @param[in] source The name of the part emitting the signal.
 */
static void _send_button_click_cb(void *data, Evas_Object *obj, const char *emission, const char *source)
{
	const char *address = elm_entry_entry_get((Evas_Object *)data);

	if (!address || strlen(address) == 0)
		return;

	if (!s_info.send_func_cb)
		return;

	s_info.send_func_cb(address, DEFAULT_TEXT_MESSAGE_EMAIL);
}

/**
 * @brief Callback function to be invoked when the address field gains focus.
 * @param[in] data Custom data passed to the callback attachment function.
 * @param[in] obj Object to which the handler is attached.
 * @param[in] event_info The information on the event occurred.
 */
static void _address_focused_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *layout = (Evas_Object *)data;

	elm_layout_signal_emit(layout, SIGNAL_MAIL_TO_HINT_HIDE, "");
}

/**
 * @brief Callback function to be invoked when the address field looses focus.
 * @param[in] data Custom data passed to the callback attachment function.
 * @param[in] obj Object to which the handler is attached.
 * @param[in] event_info The information on the event occurred.
 */
static void _address_unfocused_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *layout = (Evas_Object *)data;
	const char *input = elm_entry_entry_get(obj);

	if (!input || strlen(input) == 0)
		elm_layout_signal_emit(layout, SIGNAL_MAIL_TO_HINT_SHOW, "");
}

/**
 * @brief Callback function to be invoked when any key from the software keyboard is pressed.
 * Internally, this function hooks the "Done" button in order to close the input panel.
 * @param[in] data Custom data passed to the callback attachment function.
 * @param[in] e The evas surface of the object.
 * @param[in] obj Object to which the handler is attached.
 * @param[in] event_info The information on the event occurred.
 */
static void _input_panel_key_down_cb(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	Evas_Event_Key_Down *ev = (Evas_Event_Key_Down *)event_info;

	if (ev->keycode == DONE_BT_CODE) {
		elm_entry_input_panel_hide(obj);
		elm_object_focus_set(obj, EINA_FALSE);
	}
}

/**
 * @brief Internal callback function invoked on applications list page change.
 * This function keeps track of displayed items and makes grayed application's
 * icon from the list when it is displayed in other place than the first one.
 * @param[in] data Custom data passed to the callback attachment function.
 * @param[in] obj Object to which the handler is attached.
 * @param[in] event_info The information on the event occurred.
 */
static void _grid_page_changed(void *data, Evas_Object *obj, void *event_info)
{
	static int prev_page_no = -1;
	Elm_Object_Item *item;
	int item_idx = 0;
	int page_no = -1;

	elm_scroller_current_page_get(s_info.apps_grid, &page_no, NULL);

	if (page_no != prev_page_no)
		prev_page_no = page_no;
	else
		return;

	item = elm_gengrid_first_item_get(s_info.apps_grid);
	if (!item) {
		dlog_print(DLOG_ERROR, LOG_TAG, "elm_gengrid_first_item_get() failed.");
		return;
	}

	do {
		if (!_set_list_item_enabled(item, (bool)(item_idx == page_no)))
			continue;

		item = elm_gengrid_item_next_get(item);
		item_idx++;
	} while (item != NULL);
}

/**
 * @brief Internal callback function invoked on applications grid item display.
 * This function is assigned to the func.content_get handler of gengrid item's class structure.
 * It is responsible for items layout creation and its text parts population
 * with relevant content.
 * @param[in] data The user data passed to the elm_gengrid_item_append() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] part The name of the target part.
 * @return The layout object to be displayed in item's content area.
 */
static Evas_Object *_get_app_grid_content_cb(void *data, Evas_Object *obj, const char *part)
{
	Evas_Object *image;
	list_item_t *list_item = (list_item_t *)data;

	if (!list_item)
		return NULL;

	if (strncmp(part, "elm.swallow.icon", strlen(part)))
		return NULL;

	if (list_item->padding)
		return NULL;

	image = view_create_image(obj, list_item->icon_path);
	if (!image)
		return NULL;

	if (!list_item->enabled)
		_set_image_grayscale(image);

	return image;
}

/**
 * @brief Internal callback function invoked on applications grid item display.
 * This function is assigned to the func.text_get handler of gengrid item's class structure.
 * It is responsible for items text display.
 * @param[in] data The user data passed to the elm_gengrid_item_append() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] part The name of the target part.
 * @return The layout object to be displayed in item's content area.
 */
static char *_get_app_grid_text_cb(void *data, Evas_Object *obj, const char *part)
{
	list_item_t *list_item = (list_item_t *)data;

	if (!list_item)
		return NULL;

	if (strncmp(part, "elm.text", strlen(part)))
		return NULL;

	if (list_item->padding || !list_item->caption)
		return NULL;

	if (list_item->enabled)
		elm_object_item_signal_emit(list_item->item, SIGNAL_APPS_ITEM_ENABLE, "");
	else
		elm_object_item_signal_emit(list_item->item, SIGNAL_APPS_ITEM_DISABLE, "");

	return strdup(list_item->caption);
}

/**
 * @brief Internal callback function invoked on applications grid item removal.
 * This function is assigned to the func.del handler of gengrid item's class structure.
 * @param[in] data The user data passed to the elm_gengrid_item_append() function.
 * @param[in] obj The object invoking this callback function.
 */
static void _del_app_grid_item_cb(void *data, Evas_Object *obj)
{
	list_item_t *list_item = (list_item_t *)data;

	if (!list_item)
		return;

	free(list_item->caption);
	free(list_item->icon_path);
	free(list_item);
}

/**
 * @brief Internal function which creates fully qualified path to the provided resource file.
 * @param[in] edj_file_in The file name.
 * @param[out] edj_path_out The fully qualified path to the edj_file_in file.
 */
static void _get_app_resource(const char *edj_file_in, char *edj_path_out)
{
	char *res_path = app_get_resource_path();
	if (res_path) {
		snprintf(edj_path_out, PATH_MAX, "%s%s", res_path, edj_file_in);
		free(res_path);
	}
}

/**
 * @brief Creates the application's main layout.
 * @param[in] parent The parent object for the main layout.
 * @param[in] part_name The name of the EDJE part where the main layout has to be set.
 * @return Function returns EINA_TRUE on successful layout creation,
 * otherwise EINA_FALSE is returned.
 */
static Eina_Bool _create_main_layout(Evas_Object *parent, const char *part_name)
{
	Evas_Object *layout;
	Evas_Object *entry;

	s_info.main_layout = view_create_layout(parent, EDJ_MAIN, GRP_MAIN, part_name);
	if (!s_info.main_layout)
		return EINA_FALSE;

	layout = view_create_layout(s_info.main_layout, EDJ_OPERATIONS, GRP_OPERATIONS, PART_MAIN_OPERATIONS);
	if (!layout)
		return EINA_FALSE;

	elm_object_part_text_set(layout, PART_OPERATION_OPER_1_CAPTION, s_info.operations[OP_1].caption);
	elm_object_part_text_set(layout, PART_OPERATION_OPER_2_CAPTION, s_info.operations[OP_2].caption);
	elm_object_part_text_set(layout, PART_OPERATION_OPER_3_CAPTION, s_info.operations[OP_3].caption);

	elm_layout_signal_callback_add(layout, SIGNAL_OPERATION_RADIO_SELECTED, PART_OPERATION_OPERATIONS, _operation_radio_selected_cb, NULL);

	layout = view_create_layout(s_info.main_layout, EDJ_APPS_LIST, GRP_APPS_LIST, PART_MAIN_APPS_LIST);
	if (!layout)
		return EINA_FALSE;

	s_info.apps_grid = view_create_gengrid(layout, PART_APPS_LIST_APPS_LIST);
	if (!s_info.apps_grid)
		return EINA_FALSE;

	evas_object_smart_callback_add(s_info.apps_grid, "scroll,page,changed", _grid_page_changed, NULL);

	s_info.itc = _create_gengrid_item_class();
	if (!s_info.itc)
		return EINA_FALSE;

	elm_layout_signal_callback_add(layout, SIGNAL_BUTTON_EXECUTE_CLICKED, PART_APPS_LIST_BUTTON_EXECUTE, _execute_button_click_cb, NULL);
	elm_layout_signal_callback_add(layout, SIGNAL_BUTTON_KILL_CLICKED, PART_APPS_LIST_BUTTON_KILL, _kill_button_click_cb, NULL);

	layout = view_create_layout(s_info.main_layout, EDJ_COMPOSE, GRP_COMPOSE, PART_MAIN_COMPOSE);
	if (!layout)
		return EINA_FALSE;

	entry = view_create_entry(layout, PART_COMPOSE_ENTRY_MAIL_TO);
	if (!entry)
		return EINA_FALSE;

	evas_object_event_callback_add(entry, EVAS_CALLBACK_KEY_DOWN, _input_panel_key_down_cb, NULL);
	evas_object_smart_callback_add(entry, "focused", _address_focused_cb, (void *)layout);
	evas_object_smart_callback_add(entry, "unfocused", _address_unfocused_cb, (void *)layout);

	elm_layout_signal_callback_add(layout, SIGNAL_BUTTON_SEND_CLICKED, PART_COMPOSE_BUTTON_SEND, _send_button_click_cb, (void *)entry);
	elm_object_part_text_set(layout, PART_COMPOSE_MESSAGE, DEFAULT_TEXT_MESSAGE);

	return EINA_TRUE;
}

/**
 * @brief Internal function which creates an item class for gengrid object.
 * @return The function returns item class if it was created successfully,
 * otherwise 'NULL' is returned.
 */
static Elm_Gengrid_Item_Class *_create_gengrid_item_class(void)
{
	static Elm_Gengrid_Item_Class *itc = NULL;

	if (!itc) {
		itc = elm_gengrid_item_class_new();
		if (!itc) {
			dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_gengrid_item_class_new() failed.");
			return NULL;
		}

		itc->item_style = "full";
		itc->func.text_get = _get_app_grid_text_cb;
		itc->func.content_get = _get_app_grid_content_cb;
		itc->func.state_get = NULL;
		itc->func.del = _del_app_grid_item_cb;
	}

	return itc;
}

/**
 * @brief Internal function which changes the image color to gray scale.
 * @param[in] image The target image to be updated.
 */
static void _set_image_grayscale(Evas_Object *image)
{
	int i;
	int width = 0;
	int height = 0;
	unsigned int *buff;
	unsigned int mean;

	elm_image_object_size_get(image, &width, &height);
	buff = evas_object_image_data_get(elm_image_object_get(image), EINA_TRUE);
	if (!buff) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function evas_object_image_data_get() failed.");
		return;
	}

	for (i = 0; i < width * height; i++) {
		if ((buff[i] & 0x00ffffff) == 0x00)
			continue;

		mean = (((buff[i] & 0x00ff0000) >> 16) + ((buff[i] & 0x0000ff00) >> 8) + (buff[i] & 0x000000ff)) / 3;
		buff[i] = (buff[i] & 0xff000000) + (mean << 16) + (mean << 8) + mean;
	}

	evas_object_image_data_set(image, (void *)buff);
}

/**
 * @brief Internal function which checks whether two strings are the same.
 * @param[in] s1 The first string.
 * @param[in] s2 The second string.
 * @return This function returns 'true' is both given strings are the same,
 * otherwise 'false' is returned.
 */
static bool _is_same_string(const char *s1, const char *s2)
{
	if (!s1 || !s2)
		return false;

	return (strncmp(s1, s2, strlen(s1)) == 0 && strlen(s1) == strlen(s2));
}

/**
 * @brief Internal function which creates a list item data structure.
 * @param[in] caption The text caption of a list item.
 * @param[in] icon_path The path to the icon file.
 * @param[in] enabled The state of an item.
 * @return This function returns 'true' if the list item data structure was
 * created and initialized successfully, otherwise 'false' is returned.
 */
static bool _create_list_item(const char *caption, const char *icon_path, bool enabled, list_item_t **li)
{
	*li = (list_item_t *)calloc(1, sizeof(list_item_t));
	if (!*li) {
		dlog_print(DLOG_ERROR, LOG_TAG, "calloc() failed. Failed to allocate memory for list item.");
		return false;
	}

	if (caption)
		(*li)->caption = strdup(caption);

	if (icon_path)
		(*li)->icon_path = strdup(icon_path);

	(*li)->enabled = enabled;
	(*li)->gray_scale = !enabled;
	(*li)->padding = (!caption && !icon_path);
	(*li)->item = NULL;

	return true;
}

/**
 * @brief Internal function which changes the enabled state of an item.
 * @param[in] item The target item which enabled state is to be updated.
 * @param[in] state The enabled state to be set.
 * @return This function returns 'true' is the target item's enabled state
 * was set successfully, otherwise 'false' is returned.
 */
static bool _set_list_item_enabled(Elm_Object_Item *item, bool state)
{
	list_item_t *item_data;
	Evas_Object *image;

	if (!item) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_set_list_item_enabled() failed. Invalid input argument");
		return false;
	}

	item_data = (list_item_t *)elm_object_item_data_get(item);
	if (!item_data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "elm_object_item_data_get() failed.");
		return false;
	}

	item_data->enabled = state;
	item_data->gray_scale = !item_data->enabled;

	image = elm_object_item_part_content_get(item, "elm.swallow.icon");
	if (image && !item_data->enabled)
		_set_image_grayscale(image);

	elm_gengrid_item_update(item);

	if (item_data->enabled)
		elm_object_item_signal_emit(item, SIGNAL_APPS_ITEM_ENABLE, "");
	else
		elm_object_item_signal_emit(item, SIGNAL_APPS_ITEM_DISABLE, "");

	return true;
}
