/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "main.h"

#define PROFILE_IMAGE_HEIGHT 150

typedef struct login_data {
	Evas_Object *nf;
	Evas_Object *account_entry;
	Evas_Object *password_entry;
	Evas_Object *login_button;

	char *account_text;
	char *password_text;
} login_data_s;

/*
 * @brief Function to create single line editfield layout
 * @param[in] parent The Evas object which layout object is created on
 * @param[out] Evas_Object The layout object which is created
 */
static Evas_Object *
create_singleline_editfield_layout(Evas_Object *parent)
{
	Evas_Object *editfield, *entry;

	/* Layout */
	editfield = elm_layout_add(parent);
	elm_layout_theme_set(editfield, "layout", "editfield", "singleline");

	/* Entry */
	entry = elm_entry_add(editfield);
	elm_entry_single_line_set(entry, EINA_TRUE);
	elm_entry_scrollable_set(entry, EINA_TRUE);
	elm_object_part_content_set(editfield, "elm.swallow.content", entry);

	return editfield;
}

/*
 * @brief Function will be operated when scroller is deleted
 * @param[in] data The data to be passed to the callback function
 * @param[in] e The Evas handle to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
static void
login_scroller_del_cb(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	login_data_s *ld = data;
	ld->nf = NULL;
	ld->account_entry = NULL;
	ld->password_entry = NULL;
	ld->login_button = NULL;
	if (ld->account_text) free(ld->account_text);
	if (ld->password_text) free(ld->password_text);
	ld->account_text = NULL;
	ld->password_text = NULL;
}

/*
 * @brief Function will be operated when activate event is triggered on account entry
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
static void
login_account_entry_activated_cb(void *data, Evas_Object *obj, void *event_info)
{
	login_data_s *ld = data;

	elm_object_focus_set(ld->password_entry, EINA_TRUE);
}

/*
 * @brief Function to save account info
 * @param[in] ld The data structure to manage account info
 */
static void
login_save_account_info(login_data_s *ld)
{
	if (ld->account_text) free(ld->account_text);
	if (ld->password_text) free(ld->password_text);
	ld->account_text = NULL;
	ld->password_text = NULL;

	ld->account_text = elm_entry_markup_to_utf8(elm_entry_entry_get(ld->account_entry));
	ld->password_text = strdup(elm_entry_entry_get(ld->password_entry));
}

/*
 * @brief Function will be operated when button on popup is clicked
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
static void
login_popup_button_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *popup = data;
	evas_object_del(popup);
}

/*
 * @brief Function to create popup
 * @param[in] ld The data structure to manage account info
 * @param[out] Evas_Object The popup object which is created
 */
static Evas_Object *
create_login_popup(login_data_s *ld)
{
	Evas_Object *popup, *button;
	char *password_markup;

	Eina_Strbuf *buf = eina_strbuf_new();
	password_markup = elm_entry_utf8_to_markup(ld->password_text);

	popup = elm_popup_add(ld->nf);
	eina_strbuf_append_printf(buf, "Account: %s<br/>Password: %s",
			ld->account_text, password_markup);
	elm_object_text_set(popup, eina_strbuf_string_steal(buf));
	elm_popup_align_set(popup, ELM_NOTIFY_ALIGN_FILL, 1.0);
	eext_object_event_callback_add(popup, EEXT_CALLBACK_BACK, eext_popup_back_cb, NULL);
	elm_object_part_text_set(popup, "title,text", "Login");
	evas_object_size_hint_weight_set(popup, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	button = elm_button_add(popup);
	elm_object_style_set(button, "popup");
	elm_object_text_set(button, "OK");
	elm_object_part_content_set(popup, "button1", button);
	evas_object_smart_callback_add(button, "clicked", login_popup_button_clicked_cb, popup);

	eina_strbuf_free(buf);
	free(password_markup);

	return popup;
}

/*
 * @brief Function will be operated when activate event is triggered on password entry
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
static void
login_password_entry_activated_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *popup;
	login_data_s *ld = data;

	if (!elm_entry_is_empty(ld->account_entry) && !elm_entry_is_empty(ld->password_entry)) {
		login_save_account_info(ld);
		popup = create_login_popup(ld);
		evas_object_show(popup);
	}
}

/*
 * @brief Function to create image object
 * @param[in] parent The Evas object which image object is created on
 * @param[in] image_path The string of characters that path of image resource
 * @param[out] Evas_Object The image object which is created
 */
static Evas_Object*
create_image(Evas_Object *parent, const char *image_path)
{
	char path[100] = {0};
	char res_path[PATH_MAX] = {0, };
	app_get_resource(ICON_DIR, res_path, (int)PATH_MAX);
	snprintf(path, sizeof(path), "%s/%s", res_path, image_path);
	Evas_Object *ic = elm_image_add(parent);
	elm_image_file_set(ic, path, NULL);
	return ic;
}

/*
 * @brief Function to create input field layout
 * @param[in] parent The Evas object which layout object is created on
 * @param[in] ld The data structure to manage account info
 * @param[out] Evas_Object The layout object which is created
 */
static Evas_Object *
create_input_field(Evas_Object *parent, login_data_s *ld)
{
	Evas_Object *box, *editfield, *label;

	/* Box */
	box = elm_box_add(parent);

	/* Label */
	label = elm_label_add(box);
	evas_object_size_hint_align_set(label, 0.05, EVAS_HINT_FILL);
	evas_object_size_hint_weight_set(label, EVAS_HINT_EXPAND, 0.0);
	elm_object_text_set(label, "<b>Email Account</b>");
	elm_box_pack_end(box, label);
	evas_object_show(label);

	/* Account Editfield */
	editfield = create_singleline_editfield_layout(box);
	evas_object_size_hint_align_set(editfield, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_size_hint_weight_set(editfield, EVAS_HINT_EXPAND, 0.0);

	ld->account_entry = elm_object_part_content_get(editfield, "elm.swallow.content");
	elm_object_part_text_set(ld->account_entry, "elm.guide", "Enter your email account");
	elm_entry_input_panel_layout_set(ld->account_entry, ELM_INPUT_PANEL_LAYOUT_EMAIL);
	elm_entry_input_panel_return_key_type_set(ld->account_entry, ELM_INPUT_PANEL_RETURN_KEY_TYPE_NEXT);
	evas_object_smart_callback_add(ld->account_entry, "activated", login_account_entry_activated_cb, ld);

	elm_box_pack_end(box, editfield);
	evas_object_show(editfield);

	/* Label */
	label = elm_label_add(box);
	evas_object_size_hint_align_set(label, 0.05, EVAS_HINT_FILL);
	evas_object_size_hint_weight_set(label, EVAS_HINT_EXPAND, 0.0);
	elm_object_text_set(label, "<b>Password</b>");
	elm_box_pack_end(box, label);
	evas_object_show(label);

	/* Password Editfield */
	editfield = create_singleline_editfield_layout(box);
	evas_object_size_hint_align_set(editfield, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_size_hint_weight_set(editfield, EVAS_HINT_EXPAND, 0.0);

	ld->password_entry = elm_object_part_content_get(editfield, "elm.swallow.content");
	elm_entry_password_set(ld->password_entry, EINA_TRUE);
	elm_object_part_text_set(ld->password_entry, "elm.guide", "Enter your password");
	elm_entry_input_panel_layout_set(ld->password_entry, ELM_INPUT_PANEL_LAYOUT_PASSWORD);
	elm_entry_input_panel_return_key_type_set(ld->password_entry, ELM_INPUT_PANEL_RETURN_KEY_TYPE_DONE);
	evas_object_smart_callback_add(ld->password_entry, "activated", login_password_entry_activated_cb, ld);

	elm_box_pack_end(box, editfield);
	evas_object_show(editfield);

	return box;
}

/*
 * @brief Function will be called when "cancel" button is clicked
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
static void
login_cancel_button_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
	login_data_s *ld = data;

	elm_naviframe_item_pop(ld->nf);
}

/*
 * @brief Function will be called when "login" button is clicked
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
static void
login_login_button_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *popup;
	login_data_s *ld = data;

	login_save_account_info(ld);

	popup = create_login_popup(ld);
	evas_object_show(popup);
}

/*
 * @brief Function will be called when account entry's text is changed
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
static void
login_account_entry_changed_cb(void *data, Evas_Object *obj, void *event_info)
{
	login_data_s *ld = data;

	if (!elm_entry_is_empty(ld->account_entry) && !elm_entry_is_empty(ld->password_entry))
		elm_object_disabled_set(ld->login_button, EINA_FALSE);
	else
		elm_object_disabled_set(ld->login_button, EINA_TRUE);
}

/*
 * @brief Function will be called when password entry's text is changed
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
static void
login_password_entry_changed_cb(void *data, Evas_Object *obj, void *event_info)
{
	login_data_s *ld = data;

	if (!elm_entry_is_empty(ld->account_entry) && !elm_entry_is_empty(ld->password_entry))
		elm_object_disabled_set(ld->login_button, EINA_FALSE);
	else
		elm_object_disabled_set(ld->login_button, EINA_TRUE);
}

/*
 * @brief Function to create button layout
 * @param[in] parent The Evas object which layout object is created on
 * @param[in] ld The data structure to manage account info
 * @param[out] Evas_Object The layout object which is created
 */
static Evas_Object *
create_button_field(Evas_Object *parent, login_data_s *ld)
{
	Evas_Object *box, *button;

	/* Box */
	box = elm_box_add(parent);
	elm_box_horizontal_set(box, EINA_TRUE);
	elm_box_padding_set(box, ELM_SCALE_SIZE(10), 0);

	/* Cancel Button */
	button = elm_button_add(box);
	evas_object_size_hint_align_set(button, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_size_hint_weight_set(button, EVAS_HINT_EXPAND, 0.0);
	evas_object_smart_callback_add(button, "clicked", login_cancel_button_clicked_cb, ld);
	elm_object_text_set(button, "Cancel");
	elm_box_pack_end(box, button);
	evas_object_show(button);

	/* Login Button */
	ld->login_button = elm_button_add(box);
	evas_object_size_hint_align_set(ld->login_button, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_size_hint_weight_set(ld->login_button, EVAS_HINT_EXPAND, 0.0);
	evas_object_smart_callback_add(ld->login_button, "clicked", login_login_button_clicked_cb, ld);
	elm_object_text_set(ld->login_button, "Login");
	elm_object_disabled_set(ld->login_button, EINA_TRUE);
	elm_box_pack_end(box, ld->login_button);
	evas_object_show(ld->login_button);

	evas_object_smart_callback_add(ld->account_entry, "changed", login_account_entry_changed_cb, ld);
	evas_object_smart_callback_add(ld->password_entry, "changed", login_password_entry_changed_cb, ld);

	return box;
}

/*
 * @brief Function will be operated when "Login" item on main layout is clicked
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
void
login_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = (appdata_s *)data;
	char edj_path[PATH_MAX] = {0, };
	Evas_Object *scroller, *layout, *image, *box;
	static login_data_s ld = {0,};
	ld.nf = ad->nf;

	/* Scroller */
	scroller = elm_scroller_add(ld.nf);
	evas_object_size_hint_align_set(scroller, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_size_hint_weight_set(scroller, EVAS_HINT_EXPAND, 0.0);
	evas_object_event_callback_add(scroller, EVAS_CALLBACK_DEL, login_scroller_del_cb, &ld);

	/* Layout */
	layout = elm_layout_add(scroller);
	app_get_resource(APP_VIEW_EDJ, edj_path, (int)PATH_MAX);
	elm_layout_file_set(layout, edj_path, "login_layout");
	evas_object_size_hint_align_set(layout, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, 0.0);

	/* Image */
	image = create_image(layout, "iu.jpg");
	elm_object_part_content_set(layout, "elm.swallow.icon", image);

	/* Input field layout */
	box = create_input_field(layout, &ld);
	elm_object_part_content_set(layout, "elm.swallow.content", box);

	/* Button layout */
	box = create_button_field(layout, &ld);
	elm_object_part_content_set(layout, "elm.swallow.buttons", box);

	elm_object_content_set(scroller, layout);

	elm_naviframe_item_push(ld.nf, "Login", NULL, NULL, scroller, NULL);
}
