/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "main.h"

static Evas_Object *check[5];

/*
 * @brief Function will be operated when check's state is changed
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
static void
_check_changed_cb(void *data, Evas_Object *obj, void *event_info)
{
	Eina_Bool checked = elm_check_state_get(obj);
	int i = (int)data;

	if (!checked) {
		elm_check_state_set(check[i], EINA_TRUE);
		while (++i < 5) elm_check_state_set(check[i], EINA_FALSE);
	} else
		while (--i >= 0) elm_check_state_set(check[i], EINA_TRUE);
}

/*
 * @brief Function to create image object
 * @param[in] parent The Evas object which image object is created on
 * @param[in] image_path The string of characters that path of image resource
 * @param[in] w The integer value that width of image
 * @param[in] h The integer value that height of image
 * @param[out] Evas_Object The image object which is created
 */
static Evas_Object*
create_image(Evas_Object *parent, const char *image_path, int w, int h)
{
	Evas_Object *image;
	char path[100] = {0};
	char res_path[PATH_MAX] = {0, };

	app_get_resource(ICON_DIR, res_path, (int)PATH_MAX);
	snprintf(path, sizeof(path), "%s/%s", res_path, image_path);
	image = elm_image_add(parent);
	elm_image_file_set(image, path, NULL);
	evas_object_size_hint_min_set(image, w, h);
	evas_object_size_hint_align_set(image, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_size_hint_weight_set(image, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	return image;
}

/*
 * @brief Function to create description layout
 * @param[in] parent The Evas object which layout object is created on
 */
static void
des_layout(Evas_Object *parent)
{
	Evas_Object *box;
	Evas_Object *label;

	/* Box */
	box = elm_box_add(parent);
	elm_object_part_content_set(parent, "elm.swallow.des", box);

	/* Label */
	label = elm_label_add(box);
	evas_object_size_hint_align_set(label, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_size_hint_weight_set(label, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_label_line_wrap_set(label, ELM_WRAP_MIXED);
	elm_object_text_set(label, "EFL is a collection of libraries that are independent or may build on top of each-other to provide useful features that complement an OS's existing environment, rather than wrap and abstract it, trying to be their own environment and OS in its entirety.");
	elm_box_pack_end(box, label);
	evas_object_show(label);

	evas_object_show(box);
}

/*
 * @brief Function to create score layout
 * @param[in] parent The Evas object which layout object is created on
 */
static void
score_layout(Evas_Object *parent)
{
	Evas_Object *box;
	int i = 0;

	/* Box */
	box = elm_box_add(parent);
	elm_object_part_content_set(parent, "elm.swallow.score", box);
	elm_box_horizontal_set(box, EINA_TRUE);

	/* Check for score view */
	for (i = 0; i < 5; i++) {
		check[i] = elm_check_add(box);
		elm_object_style_set(check[i], "favorite");
		elm_check_state_set(check[i], EINA_TRUE);
		evas_object_show(check[i]);
		elm_box_pack_end(box, check[i]);
		evas_object_smart_callback_add(check[i], "changed", _check_changed_cb, (void *)i);
	}

	evas_object_show(box);
}

/*
 * @brief Function to create image layout in scroller
 * @param[in] parent The Evas object which layout object is created on
 */
static void
scroller_box_layout(Evas_Object *parent)
{
	Evas_Object *box;
	Evas_Object *image;

	/* Box */
	box = elm_box_add(parent);
	elm_box_horizontal_set(box, EINA_TRUE);

	/* Image */
	image = create_image(box, "clock1.png", 250, 300);
	elm_box_pack_end(box, image);
	evas_object_show(image);
	image = create_image(box, "clock2.png", 250, 300);
	elm_box_pack_end(box, image);
	evas_object_show(image);
	image = create_image(box, "clock3.png", 250, 300);
	elm_box_pack_end(box, image);
	evas_object_show(image);
	image = create_image(box, "clock4.png", 250, 300);
	elm_box_pack_end(box, image);
	evas_object_show(image);
	image = create_image(box, "clock5.png", 250, 300);
	elm_box_pack_end(box, image);
	evas_object_show(image);
	image = create_image(box, "clock6.png", 250, 300);
	elm_box_pack_end(box, image);
	evas_object_show(image);

	evas_object_show(box);
	elm_object_content_set(parent, box);
}

/*
 * @brief Function to create scroller layout
 * @param[in] parent The Evas object which layout object is created on
 */
static void
scroller_layout(Evas_Object *parent)
{
	Evas_Object *scroller;

	/* Scroller */
	scroller = elm_scroller_add(parent);
	evas_object_size_hint_align_set(scroller, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_size_hint_weight_set(scroller, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	/* Scroller Box Image layout */
	scroller_box_layout(scroller);

	elm_object_part_content_set(parent, "elm.swallow.scroll", scroller);
	evas_object_show(scroller);
}

/*
 * @brief Function will be operated when "Scroll in scroll" item on main layout is clicked
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
void
scroll_in_scroll_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = (appdata_s *)data;
	char edj_path[PATH_MAX] = {0, };
	Evas_Object *layout;
	Evas_Object *scroller;
	Evas_Object *nf = ad->nf;

	/* Scroller */
	scroller = elm_scroller_add(nf);
	elm_scroller_policy_set(scroller, ELM_SCROLLER_POLICY_OFF, ELM_SCROLLER_POLICY_AUTO);
	elm_naviframe_item_push(nf, "Scroll in Scroll", NULL, NULL, scroller, NULL);

	/* Layout */
	layout = elm_layout_add(scroller);
	app_get_resource(APP_VIEW_EDJ, edj_path, (int)PATH_MAX);
	elm_layout_file_set(layout, edj_path, "scroll_in_scroll_layout");
	evas_object_size_hint_align_set(layout, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_object_content_set(scroller, layout);

	/* Scroller layout */
	scroller_layout(layout);

	/* Score layout */
	score_layout(layout);

	/* Description layout */
	des_layout(layout);
}
