/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "main.h"

/* search layout app data */
typedef struct _search_data {
	Evas_Object *editfield;
	Evas_Object *genlist;
	Elm_Genlist_Item_Class* itc;
} search_data;

/* genlist item data */
typedef struct _text_item {
	char text[100];
} text_item;

/* text database,
 * if you want you can change this to another database such as SQLlite, Oracle and so on */
static const int text_database_size = 20;
static const char *text_database[] = {
	"orange", "banana",
	"tomato", "mango",
	"almond", "blueberry",
	"melon", "peach",
	"cherry", "sweety",
	"pear", "avocado",
	"grapes", "grapefruit",
	"plum", "apricot",
	"kiwi", "guava",
	"coconut", "papaya"
};

/*
 * @brief Function to update genlist when search text is changed
 * @param[in] search_text The text which is changed
 * @param[in] data The data to be passed to the callback function
 */
void
update_genlist(const char* search_text, void* data)
{
	int i;
	search_data *sd = data;
	elm_genlist_clear(sd->genlist);

	if (!strcmp(search_text, "")) {
		for (i = 0; i < text_database_size; i++) {
			text_item *text_data = calloc(sizeof(text_item), 1);
			sprintf(text_data->text, "%s", text_database[i]);
			elm_genlist_item_append(sd->genlist, sd->itc, text_data, NULL, ELM_GENLIST_ITEM_NONE, NULL, NULL);
		}
	} else {
		char *search_text_lower = strdup(search_text);
		for (int j = 0; j < strlen(search_text_lower); j++)
			search_text_lower[j] = (char)tolower(search_text_lower[j]);

		for (i = 0; i < text_database_size; i++) {
			char *database_text_lower = strdup(text_database[i]);
			for (int j = 0; j < strlen(database_text_lower); j++)
				database_text_lower[j] = (char)tolower(database_text_lower[j]);

			if (strstr(database_text_lower, search_text_lower) != NULL) {
				text_item *text_data = calloc(sizeof(text_item), 1);
				sprintf(text_data->text, "%s", text_database[i]);
				elm_genlist_item_append(sd->genlist, sd->itc, text_data, NULL, ELM_GENLIST_ITEM_NONE, NULL, NULL);
			}

			free(database_text_lower);
		}

		free(search_text_lower);
	}
}

/*
 * @brief Function to get string on genlist item's text part
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] part The name of text part
 * @param[out] char* A string with the characters to use as genlist item's text part
 */
static char*
gl_text_get_cb(void *data, Evas_Object *obj, const char *part)
{
	text_item *text_data = data;
	return strdup(text_data->text);
}

/*
 * @brief Function will be operated when genlist is deleted
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 */
static void
gl_del_cb(void *data, Evas_Object *obj)
{
	text_item *text_data = data;
	free(text_data);
}

/*
 * @brief Function will be operated when focus event is triggered on editfield
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
static void
editfield_focused_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *editfield = (Evas_Object *)data;
	elm_object_signal_emit(editfield, "elm,state,focused", "");

	if (!elm_entry_is_empty(obj))
		elm_object_signal_emit(editfield, "elm,action,show,button", "");
}

/*
 * @brief Function will be operated when unfocus event is triggered on editfield
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
static void
editfield_unfocused_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *editfield = (Evas_Object *)data;
	elm_object_signal_emit(editfield, "elm,state,unfocused", "");
	elm_object_signal_emit(editfield, "elm,action,hide,button", "");
}

/*
 * @brief Function will be operated when editfield's text is changed
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
static void
editfield_changed_cb(void *data, Evas_Object *obj, void *event_info)
{
	search_data *sd = data;
	Evas_Object *editfield = sd->editfield;
	const char *search_text = elm_entry_entry_get(obj);

	if (!elm_entry_is_empty(obj) && elm_object_focus_get(obj))
		elm_object_signal_emit(editfield, "elm,action,show,button", "");
	else
		elm_object_signal_emit(editfield, "elm,action,hide,button", "");

	update_genlist(search_text, sd);
}

/*
 * @brief Function will be operated when "clear" button on editfield is clicked
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
static void
editfield_clear_button_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *entry = (Evas_Object *)data;
	elm_entry_entry_set(entry, "");
}

/*
 * @brief Function to create search field layout
 * @param[in] parent The Evas object which is layout object is created on
 * @param[in] search_data The data structure to manage search data
 * @param[out] Evas_Object The layout object which is created
 */
Evas_Object*
create_search_field(Evas_Object* parent, void* search_data)
{
	Evas_Object *editfield, *entry, *button;

	/* Editfield layout */
	editfield = elm_layout_add(parent);
	elm_layout_theme_set(editfield, "layout", "editfield", "singleline");

	/* Entry */
	entry = elm_entry_add(editfield);
	elm_entry_single_line_set(entry, EINA_TRUE);
	elm_entry_scrollable_set(entry, EINA_TRUE);
	elm_object_part_text_set(entry, "elm.guide", "Search");
	evas_object_smart_callback_add(entry, "focused", editfield_focused_cb, editfield);
	evas_object_smart_callback_add(entry, "unfocused", editfield_unfocused_cb, editfield);
	evas_object_smart_callback_add(entry, "changed", editfield_changed_cb, search_data);
	evas_object_smart_callback_add(entry, "preedit,changed", editfield_changed_cb, search_data);
	elm_object_part_content_set(editfield, "elm.swallow.content", entry);

	/* Button */
	button = elm_button_add(editfield);
	elm_object_style_set(button, "editfield_clear");
	evas_object_smart_callback_add(button, "clicked", editfield_clear_button_clicked_cb, entry);
	elm_object_part_content_set(editfield, "elm.swallow.button", button);

	return editfield;
}

/*
 * @brief Function to create text list
 * @param[in] layout The Evas object which list object is created on
 * @param[out] Evas_Object The list object which is created
 */
Evas_Object*
create_text_list(Evas_Object* layout)
{
	/* add an genlist to the layout */
	Evas_Object* genlist;
	genlist = elm_genlist_add(layout);

	return genlist;
}

/*
 * @brief Function to create item class for genlist
 * @param[out] Elm_Genlist_Item_Class The item class object which is created
 */
Elm_Genlist_Item_Class*
create_item_class()
{
	/* set genlist item style and callbacks */
	Elm_Genlist_Item_Class* itc;
	itc = elm_genlist_item_class_new();
	itc->item_style = "default";
	itc->func.content_get = NULL;
	itc->func.text_get = gl_text_get_cb;
	itc->func.del = gl_del_cb;

	return itc;
}

/*
 * @brief Function will be operated when search layout is deleted
 * @param[in] data The data to be passed to the callback function
 * @param[in] e The Evas handle to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
static void
search_layout_del_cb(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	search_data *sd = data;
	sd->editfield = NULL;
	sd->genlist = NULL;
	sd->itc = NULL;
	free(sd);
}

/*
 * @brief Function will be operated when "Search" item on main layout is clicked
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
void
search_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = (appdata_s *)data;
	char edj_path[PATH_MAX] = {0, };
	Evas_Object *layout;
	Evas_Object *nf = ad->nf;
	int i;

	/* create an search app data */
	search_data *sd = malloc(sizeof(search_data));

	/* create an layout */
	layout = elm_layout_add(nf);
	app_get_resource(APP_VIEW_EDJ, edj_path, (int)PATH_MAX);
	elm_layout_file_set(layout, edj_path, "search_layout");
	evas_object_event_callback_add(layout, EVAS_CALLBACK_DEL, search_layout_del_cb, sd);

	/* create an entry */
	sd->editfield = create_search_field(layout, sd);
	elm_object_part_content_set(layout, "elm.swallow.searchbar", sd->editfield);

	/* create an genlist */
	sd->genlist = create_text_list(layout);
	elm_object_part_content_set(layout, "elm.swallow.list", sd->genlist);

	/* create an genlist item class */
	sd->itc = create_item_class();

	/* initialize list data */
	for (i = 0; i < text_database_size; i++) {
		/* declare genlist item and set text of the item, then append to the genlist */
		text_item *text_data = calloc(sizeof(text_item), 1);
		sprintf(text_data->text, "%s", text_database[i]);
		elm_genlist_item_append(sd->genlist, sd->itc, text_data, NULL, ELM_GENLIST_ITEM_NONE, NULL, NULL);
	}

	elm_naviframe_item_push(nf, "Search", NULL, NULL, layout, NULL);
}
