/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "main.h"

/*
 * @brief Function will be operated when focus event is triggered on searchfield
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
static void
searchfield_focused_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *editfield = (Evas_Object *)data;
	elm_object_signal_emit(editfield, "elm,state,focused", "");

	if (!elm_entry_is_empty(obj))
		elm_object_signal_emit(editfield, "elm,action,show,button", "");
}

/*
 * @brief Function will be operated when unfocus event is triggered on searchfield
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
static void
searchfield_unfocused_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *editfield = (Evas_Object *)data;
	elm_object_signal_emit(editfield, "elm,state,unfocused", "");
	elm_object_signal_emit(editfield, "elm,action,hide,button", "");
}

/*
 * @brief Function will be operated when searchfield's text is changed
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
static void
searchfield_changed_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *editfield = (Evas_Object *)data;

	if (!elm_entry_is_empty(obj) && elm_object_focus_get(obj))
		elm_object_signal_emit(editfield, "elm,action,show,button", "");
	else
		elm_object_signal_emit(editfield, "elm,action,hide,button", "");
}

/*
 * @brief Function will be operated when "clear" button on searchfield layout is clicked
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
static void
searchfield_clear_button_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *entry = (Evas_Object *)data;

	elm_entry_entry_set(entry, "");
}

/*
 * @brief Function to create searchfield layout
 * @param[in] parent The Evas object which layout object is created on
 * @param[out] Evas_Object The layout object which is created
 */
static Evas_Object *
create_searchfield_layout(Evas_Object *parent)
{
	Evas_Object *searchfield, *entry, *button;

	/* Searchfield layout */
	searchfield = elm_layout_add(parent);
	elm_layout_theme_set(searchfield, "layout", "searchfield", "singleline");
	evas_object_size_hint_align_set(searchfield, EVAS_HINT_FILL, 0.0);
	evas_object_size_hint_weight_set(searchfield, EVAS_HINT_EXPAND, 0.0);

	/* Entry */
	entry = elm_entry_add(searchfield);
	elm_entry_single_line_set(entry, EINA_TRUE);
	elm_entry_scrollable_set(entry, EINA_TRUE);
	elm_entry_input_panel_return_key_type_set(entry, ELM_INPUT_PANEL_RETURN_KEY_TYPE_SEARCH);
	elm_object_part_text_set(entry, "elm.guide", "Search Field");
	evas_object_smart_callback_add(entry, "focused", searchfield_focused_cb, searchfield);
	evas_object_smart_callback_add(entry, "unfocused", searchfield_unfocused_cb, searchfield);
	evas_object_smart_callback_add(entry, "changed", searchfield_changed_cb, searchfield);
	evas_object_smart_callback_add(entry, "preedit,changed", searchfield_changed_cb, searchfield);
	elm_object_part_content_set(searchfield, "elm.swallow.content", entry);

	/* Button */
	button = elm_button_add(searchfield);
	elm_object_style_set(button, "editfield_clear");
	evas_object_smart_callback_add(button, "clicked", searchfield_clear_button_clicked_cb, entry);
	elm_object_part_content_set(searchfield, "elm.swallow.button", button);

	return searchfield;
}

/*
 * @brief Function to create nocontent layout
 * @param[in] parent The Evas object which layout object is created on
 * @param[in] text The text which is shown on layout
 * @param[out] Evas_Object The layout object which is created
 */
static Evas_Object*
create_nocontent_layout(Evas_Object *parent, const char *text)
{
	Evas_Object *layout, *scroller;

	/* Scroller */
	scroller = elm_scroller_add(parent);
	elm_scroller_bounce_set(scroller, EINA_FALSE, EINA_TRUE);
	elm_scroller_policy_set(scroller, ELM_SCROLLER_POLICY_OFF, ELM_SCROLLER_POLICY_AUTO);

	/* No Content Layout */
	layout = elm_layout_add(scroller);
	elm_layout_theme_set(layout, "layout", "nocontents", "default");
	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(layout, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_object_part_text_set(layout, "elm.text", "No Contents");

	elm_object_content_set(scroller, layout);

	return scroller;
}

/*
 * @brief Function to create searchbar view layout
 * @param[in] ad The data structure to manage gui object
 * @param[out] Evas_Object The layout object which is created
 */
static Evas_Object *
create_searchbar_view(appdata_s *ad)
{
	Evas_Object *main_scroller, *main_box, *searchfield, *layout;

	/* Scroller */
	main_scroller = elm_scroller_add(ad->nf);
	elm_scroller_bounce_set(main_scroller, EINA_FALSE, EINA_TRUE);
	evas_object_size_hint_weight_set(main_scroller, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(main_scroller, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_show(main_scroller);

	/* Box */
	main_box = elm_box_add(main_scroller);
	evas_object_size_hint_align_set(main_box, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_size_hint_weight_set(main_box, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_show(main_box);

	/* Searchfield layout */
	searchfield = create_searchfield_layout(main_box);
	elm_box_pack_end(main_box, searchfield);
	evas_object_show(searchfield);

	/* Nocontent layout */
	layout = create_nocontent_layout(main_box, "No Contents");
	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(layout, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_box_pack_end(main_box, layout);
	evas_object_show(layout);

	elm_object_content_set(main_scroller, main_box);

	return main_scroller;
}

/*
 * @brief Function will be operated when "Searchbar" item on main layout is clicked
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
void
searchbar_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = (appdata_s *)data;
	Evas_Object *searchbar_view;

	/* Search Bar view */
	searchbar_view = create_searchbar_view(ad);
	elm_naviframe_item_push(ad->nf, "Search Bar", NULL, NULL, searchbar_view, NULL);
}
