/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <app.h>
#include <efl_extension.h>
#include "$(appName).h"
#include <efl_extension.h>
#include "view/view_common.h"
#include "view_defines.h"

static void _close_popup_cb(void *data, Evas_Object *obj, void *event_info);

/**
 * @brief Function which creates fully qualified path to the provided resource file.
 * @param[in] edj_file_in The file name.
 * @param[out] edj_path_out The fully qualified path to the edj_file_in file.
 */
void view_common_get_app_resource(const char *edj_file_in, char *edj_path_out)
{
	char *res_path = app_get_resource_path();
	if (res_path) {
		snprintf(edj_path_out, PATH_MAX, "%s%s", res_path, edj_file_in);
		free(res_path);
	}
}

/**
 * @brief Creates a layout object based on provided EDJE script.
 * @param[in] parent The parent object for layout object.
 * @param[in] edj_file_name The relative path to the layout EDJE script.
 * @param[in] edj_group The name of the group to be loaded from the EDJE script.
 * @return The function returns layout object if it was created successfully,
 * otherwise 'NULL' is returned.
 */
Evas_Object *view_create_layout(Evas_Object *parent, const char *edj_file_name, const char *edj_group)
{
	char edj_path[PATH_MAX] = {0, };
	Evas_Object *layout = NULL;

	if (!parent || !edj_file_name || !edj_group) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong input arguments.");
		return NULL;
	}

	view_common_get_app_resource(edj_file_name, edj_path);

	layout = elm_layout_add(parent);
	if (!layout) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_layout_add() failed.");
		return NULL;
	}

	if (!elm_layout_file_set(layout, edj_path, edj_group)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_layout_file_set() failed.");
		evas_object_del(layout);
		return NULL;
	}

	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	return layout;
}

/**
 * @brief Creates a toast popup message.
 * @param parent The parent widget.
 * @param message The message to display.
 */
void view_common_create_popup(Evas_Object *parent, char *message)
{
	Evas_Object *popup = elm_popup_add(parent);
	if (!popup) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] popup == NULL", __FILE__, __LINE__);
		return;
	}

	elm_object_style_set(popup, "toast");
	elm_object_text_set(popup, message);
	elm_popup_timeout_set(popup, 3);
	evas_object_smart_callback_add(popup, "timeout", _close_popup_cb, NULL);
	evas_object_smart_callback_add(popup, "block,clicked", _close_popup_cb, NULL);
	evas_object_show(popup);
}

/**
 * @brief Callback invoked when a genlist's tree item is contracted.
 * @param data User data.
 * @param obj genlist object.
 * @param event_info The contracted item.
 */
void view_common_tree_item_contracted_cb(void *data, Evas_Object *obj, void *event_info)
{
	Elm_Object_Item *it_parent = (Elm_Object_Item*) event_info;
	Evas_Object *header = elm_object_item_part_content_get(it_parent, "elm.swallow.content");
	elm_genlist_item_subitems_clear(it_parent);
	elm_layout_signal_emit(header, SIGNAL_ARROW_STATE_CHANGE, "");
}

/**
 * @brief Internal callback function invoked when the popup is going to be hidden.
 * @param data User data.
 * @param obj genlist object.
 * @param event_info The contracted item.
 */
static void _close_popup_cb(void *data, Evas_Object *obj, void *event_info)
{
	evas_object_del(obj);
}
