/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <app.h>
#include <efl_extension.h>
#include "$(appName).h"
#include "view/view_list.h"

static void _tree_item_contracted(void *data, Evas_Object *o, void *event_info);
static void _selection_status_changed_cb(void *data, Evas_Object *obj, void *event_info);

/**
 * @brief Creates a genlist object.
 * @param[in] parent The genlist's parent.
 * @return The created genlist or NULL on error.
 */
Evas_Object *view_list_create(Evas_Object *parent)
{
	Evas_Object *list = elm_genlist_add(parent);
	if (!list) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] list == NULL", __FILE__, __LINE__);
		return NULL;
	}

	evas_object_smart_callback_add(list, "selected", _selection_status_changed_cb, (void*)EINA_TRUE);
	elm_scroller_gravity_set(list, 0.0, 1.0);

	return list;
}

/**
 * @brief Adds callback functions to a genlist that are used when an item is extended or contracted.
 * @param[in] list The genlist.
 * @param[in] itc_parent The parent item class (the expanding or contracting item).
 * @param[in] itc_child The sub object of the parent object.
 * @param[in] expand_callback Callback invoked when an item was expanded.
 * @param[in] contract_callback Callback invoked when an item was contracted.
 * @note If the contract callback is not provided a default one will be used. If a callback is provided it has to have the following code:
 * @code Elm_Object_Item *it_parent = (Elm_Object_Item*) event_info;
	elm_genlist_item_subitems_clear(it_parent);
 */
void view_list_add_tree_callbacks(Evas_Object *list, Elm_Gen_Item_Class *itc_parent, Elm_Gen_Item_Class *itc_child, Evas_Smart_Cb expand_callback, Evas_Smart_Cb contract_callback)
{
	if (!expand_callback) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] expand_callback == NULL", __FILE__, __LINE__);
		return;
	}

	evas_object_smart_callback_add(list, "expanded", expand_callback, itc_child);

	if (contract_callback)
		evas_object_smart_callback_add(list, "contracted", contract_callback, NULL);
	else
		evas_object_smart_callback_add(list, "contracted", _tree_item_contracted, NULL);
}

/**
 * @brief Creates a new item class.
 * @param[in] content_get_cb Callback invoked when a new item is going to be realized by the genlist.
 * @param[in] content_del_cb Callback invoked when a item is going to be deleted. If NULL is used a default callback will be invoked.
 * @return The newly created class or NULL on error.
 */
Elm_Genlist_Item_Class *view_list_add_itc(Elm_Gen_Item_Content_Get_Cb content_get_cb, Elm_Gen_Item_Del_Cb content_del_cb)
{
	Elm_Genlist_Item_Class *itc = elm_genlist_item_class_new();
	if (!itc) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] (*itc == NULL", __FILE__, __LINE__);
		return false;
	}
	itc->item_style = "full";
	itc->func.text_get = NULL;
	itc->func.content_get = content_get_cb;
	itc->func.state_get = NULL;
	itc->func.del = content_del_cb;

	return itc;
}

/**
 * @brief Creates a new item and adds it to the genlist.
 * @param[in] list The genlist.
 * @param[in] itc The item class to use.
 * @param[in] item_type Item type (NONE, TREE, GROUP).
 * @param[in] parent_item The parent item.
 * @param[in] data The data that will be passed to the content_get of the item class.
 * @param[in] add_first True - The item is going to be prepended. False - The item is going to be appended.
 * @param[in] mode The selection mode used by the newly created item.
 * @param[in] func Callback function invoked when the item is selected.
 * @param[in] func_data Data passed to the func callback function.
 * @return The newly created list item.
 */
Elm_Object_Item *view_list_add_item(Evas_Object *list, Elm_Genlist_Item_Class *itc,
		Elm_Genlist_Item_Type item_type, Elm_Object_Item *parent_item,
		void *data, bool add_first,
		Elm_Object_Select_Mode mode,
		Evas_Smart_Cb func, const void *func_data)
{
	Elm_Object_Item *item = NULL;

	if (add_first)
		item = elm_genlist_item_prepend(list, itc, data, parent_item, item_type, func, func_data);
	else
		item = elm_genlist_item_append(list, itc, data, parent_item, item_type, func, func_data);

	if (!item) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] item == NULL", __FILE__, __LINE__);
		return NULL;
	}

	elm_genlist_item_select_mode_set(item, mode);

	return item;
}

/**
 * @brief Updates the realized items.
 * @param[in] list The list to update.
 */
void view_list_update(Evas_Object *list)
{
	if (!list) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] list == NULL", __FILE__, __LINE__);
		return;
	}

	elm_genlist_realized_items_update(list);
}

/**
 * @brief Updates a selected item of the genlist.
 * @param[in] item The item to update.
 */
void view_list_item_update(Elm_Object_Item *item)
{
	if (!item) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] item == NULL", __FILE__, __LINE__);
		return;
	}

	elm_genlist_item_update(item);
}

/**
 * @brief Deletes an item from the genlist.
 * @param[in] list The genlist.
 * @param[in] item The item to remove.
 */
void view_list_del_item(Evas_Object *list, Elm_Object_Item *item)
{
	if (!list) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] list == NULL", __FILE__, __LINE__);
		return;
	}

	if (!item) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] item == NULL", __FILE__, __LINE__);
		return;
	}

	elm_object_item_del(item);
}

/**
 * @brief Default callback function invoked when a genlist's item is going to be deleted.
 * @param[in] data The data added to the item.
 * @param[in] obj The genlist object.
 */
void view_list_del_item_cb(void *data, Evas_Object *obj)
{
	if (data)
		free(data);
}

/**
 * @brief Internal default callback function invoked when a TREE item was contracted.
 * @param[in] data User data.
 * @param[in] obj The genlist.
 * @param[in] event_info The contracted item.
 */
static void _tree_item_contracted(void *data, Evas_Object *obj, void *event_info)
{
	Elm_Object_Item *it_parent = (Elm_Object_Item*) event_info;
	elm_genlist_item_subitems_clear(it_parent);
}

/**
 * @brief Internal callback function invoked when a TREE item is selected. It is used to change the 'expanded' state of the item.
 * @param[in] data User data.
 * @param[in] obj The genlist.
 * @param[in] event_info The selected item.
 */
static void _selection_status_changed_cb(void *data, Evas_Object *obj, void *event_info)
{
	Elm_Object_Item *item = (Elm_Object_Item *)event_info;

	if (elm_genlist_item_type_get(item) == ELM_GENLIST_ITEM_TREE) {
		elm_genlist_item_selected_set(item, EINA_FALSE);
		elm_genlist_item_expanded_set(item, !elm_genlist_item_expanded_get(item));
	}
}
