/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <app.h>
#include <efl_extension.h>
#include "$(appName).h"
#include "view/view_source.h"
#include "view/view_common.h"
#include "view/view_sink.h"
#include "view/view_list.h"
#include "view_defines.h"
#include "data.h"

#define ENTRY_FORMAT "DEFAULT='font=BreezeSans:style=Light color=#000000 font_size=26 wrap=mixed align=center'"
#define FORMAT(type) "%s: "type"<br><type>Type: %s</type>"

static struct view_info {
	Evas_Object *layout;
	Elm_Entry_Filter_Accept_Set accept_set;
	Evas_Object *list;
	Elm_Genlist_Item_Class *itc_item;
	Elm_Genlist_Item_Class *itc_header_data;
	Elm_Genlist_Item_Class *itc_header;
	Elm_Object_Item *header;
	Elm_Object_Item *header_data;
	bool is_byte;
	bool include_header;
} s_info = {
	.layout = NULL,
	.is_byte = true,
	.include_header = false,
	.list = NULL,
	.itc_item = NULL,
	.itc_header_data = NULL,
	.itc_header = NULL,
	.header = NULL,
	.header_data = NULL,
	.accept_set = {
		.accepted = "0123456789",
		.rejected = NULL,
	}
};

static bool _create_entry(char *part_name);
static void _add_button_clicked_cb(void *data, Evas_Object *obj, const char *emission, const char *source);
static void _send_button_clicked_cb(void *data, Evas_Object *obj, const char *emission, const char *source);
static void _data_type_selected_cb(void *data, Evas_Object *obj, const char *emission, const char *source);
static void _header_include_clicked_cb(void *data, Evas_Object *obj, const char *emission, const char *source);
static void _number_only(void);
static Evas_Object *_item_content_get_cb(void *data, Evas_Object *obj, const char *part);
static Evas_Object *_header_content_get_cb(void *data, Evas_Object *obj, const char *part);
static Evas_Object *_header_data_content_get_cb(void *data, Evas_Object *obj, const char *part);
static bool _create_list(void);
static void _tree_item_expanded(void *data, Evas_Object *obj, void *event_info);
static void _set_header_state(bool header_state);
static void _entry_text_changed_cb(void *data, Evas_Object *obj, void *event_info);
static void _list_item_selected_cb(void *data, Evas_Object *obj, void *event_info);

/**
 * @brief Function which creates the source view.
 * @return The function returns 'EINA_TRUE' if the main layout was created successfully,
 * otherwise 'EINA_FALSE' is returned.
 */
bool view_source_create(Evas_Object *parent)
{
	s_info.layout = view_create_layout(parent, EDJ_SOURCE_FILE_NAME, GROUP_SOURCE);
	if (!s_info.layout) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] s_info.layout == NULL", __FILE__, __LINE__);
		return false;
	}

	if (!_create_entry(PART_SOURCE_KEY)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] _create_entry() failed. Part: %s", __FILE__, __LINE__, PART_SOURCE_KEY);
		return false;
	}

	if (!_create_entry(PART_SOURCE_VALUE)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] _create_entry() failed. Part: %s", __FILE__, __LINE__, PART_SOURCE_VALUE);
		return false;
	}

	if (!_create_list()) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] _create_list() == NULL", __FILE__, __LINE__);
		return false;
	}

	elm_layout_signal_callback_add(s_info.layout, SIGNAL_SOURCE_ADD, "", _add_button_clicked_cb, NULL);
	elm_layout_signal_callback_add(s_info.layout, SIGNAL_SOURCE_SEND, "", _send_button_clicked_cb, NULL);

	elm_layout_signal_callback_add(s_info.layout, SIGNAL_SOURCE_BYTE_SELECTED, "", _data_type_selected_cb, (void *)true);
	elm_layout_signal_callback_add(s_info.layout, SIGNAL_SOURCE_STRING_SELECTED, "", _data_type_selected_cb, (void *)false);

	elm_layout_signal_callback_add(s_info.layout, SIGNAL_SOURCE_INCLUDE_CHECKED, "", _header_include_clicked_cb, (void*)true);
	elm_layout_signal_callback_add(s_info.layout, SIGNAL_SOURCE_INCLUDE_UNCHECKED, "", _header_include_clicked_cb, (void*)false);

	evas_object_size_hint_weight_set(s_info.layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	_number_only();

	return true;
}

/**
 * @brief Function called when the source view is going to be destroyed.
 */
void view_source_finalize(void)
{
	elm_genlist_item_class_free(s_info.itc_item);
}

/**
 * @brief Returns a pointer to the source view's layout.
 * @return The layout object.
 */
Evas_Object *view_source_get(void)
{
	return s_info.layout;
}

/**
 * @brief Internal functions creates a genlist and class items used by it.
 * @return true on success or false on fail.
 */
static bool _create_list(void)
{
	s_info.list = view_list_create(s_info.layout);
	if (!s_info.list) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] s_info.list == NULL", __FILE__, __LINE__);
		return false;
	}

	s_info.itc_item = view_list_add_itc(_item_content_get_cb, NULL);
	if (!s_info.itc_item) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] s_info.itc == NULL", __FILE__, __LINE__);
		return false;
	}

	s_info.itc_header_data = view_list_add_itc(_header_data_content_get_cb, NULL);
	if (!s_info.itc_header_data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] s_info.itc_sub == NULL", __FILE__, __LINE__);
		return false;
	}

	s_info.itc_header = view_list_add_itc(_header_content_get_cb, NULL);
	if (!s_info.itc_header) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] s_info.itc_tree == NULL", __FILE__, __LINE__);
		return false;
	}

	view_list_add_tree_callbacks(s_info.list, s_info.itc_header, s_info.itc_header_data, _tree_item_expanded, view_common_tree_item_contracted_cb);

	if (!elm_layout_content_set(s_info.layout, PART_LIST, s_info.list)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to set entry as a swallow %s", __FILE__, __LINE__, PART_LIST);
		return false;
	}

	return true;
}

/**
 * @brief Internal function that creates an entry widget and ads it to the layout.
 * @param[in] part_name The part where the entry will be added.
 * @return true on success or false on fail.
 */
static bool _create_entry(char *part_name)
{
	Evas_Object *entry = elm_entry_add(s_info.layout);
	if (!entry) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] entry == NULL", __FILE__, __LINE__);
		return false;
	}

	elm_entry_text_style_user_push(entry, ENTRY_FORMAT);
	elm_entry_single_line_set(entry, EINA_TRUE);
	evas_object_size_hint_weight_set(entry, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	if (!elm_layout_content_set(s_info.layout, part_name, entry)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to set entry as a swallow %s", __FILE__, __LINE__, part_name);
		return false;
	}

	evas_object_smart_callback_add(entry, "changed", _entry_text_changed_cb, NULL);

	return true;
}

/**
 * @brief Internal function that filters the keyboard input based on the type of the data value used.
 */
static void _number_only(void)
{
	Evas_Object *entry = elm_layout_content_get(s_info.layout, PART_SOURCE_VALUE);
	Elm_Input_Panel_Layout panel_layout = ELM_INPUT_PANEL_LAYOUT_NORMAL;

	elm_entry_markup_filter_remove(entry, elm_entry_filter_accept_set, &s_info.accept_set);

	if (s_info.is_byte) {
		s_info.accept_set.accepted = "0123456789";
		panel_layout =  ELM_INPUT_PANEL_LAYOUT_NUMBER;
	} else {
		s_info.accept_set.accepted = NULL;
		panel_layout =  ELM_INPUT_PANEL_LAYOUT_NORMAL;
	}

	elm_entry_input_panel_layout_set(entry, panel_layout);
	elm_entry_markup_filter_append(entry, elm_entry_filter_accept_set, &s_info.accept_set);
}

/**
 * @brief Internal callback function invoked when the add button is clicked.
 * @param[in] data The user data.
 * @param[in] obj The layout object.
 * @param[in] emission The emitted signal.
 * @param[in] source The signal's source.
 */
static void _add_button_clicked_cb(void *data, Evas_Object *obj, const char *emission, const char *source)
{
	const char *key = elm_object_text_get(elm_layout_content_get(s_info.layout, PART_SOURCE_KEY));
	const char *val = elm_object_text_get(elm_layout_content_get(s_info.layout, PART_SOURCE_VALUE));
	Evas_Object *entry_1 = elm_layout_content_get(s_info.layout, PART_SOURCE_KEY);
	Evas_Object *entry_2 = elm_layout_content_get(s_info.layout, PART_SOURCE_VALUE);
	bool ret = false;
	bool exists = data_key_exists(-1, key);


	if (elm_entry_is_empty(entry_1) || elm_entry_is_empty(entry_2))
		return;

	if (s_info.is_byte)
		data_add_byte(key, atoi(val));
	else
		data_add_string(key, val);

	if (!exists)
		view_list_add_item(s_info.list, s_info.itc_item, ELM_GENLIST_ITEM_NONE, NULL, (void*)strdup(key), false, ELM_OBJECT_SELECT_MODE_DEFAULT, _list_item_selected_cb, NULL);
	else
		view_list_update(s_info.list);

	if (s_info.include_header) {
		if (data_add_header())
			view_list_item_update(s_info.header);

		if (elm_genlist_item_expanded_get(s_info.header)) {
			elm_genlist_item_expanded_set(s_info.header, EINA_FALSE);
			elm_genlist_item_expanded_set(s_info.header, EINA_TRUE);
		}
	}

	elm_layout_signal_emit(s_info.layout, SIGNAL_SEND_ON, "");
	dlog_print(DLOG_INFO, LOG_TAG, "[%s:%d] %s [%s=%s] %d", __FILE__, __LINE__, emission, key, val, ret);
}

/**
 * @brief Internal callback function invoked when the send button is clicked. It is used to send data stored in the bundle object using the message system.
 * @param[in] data The user data.
 * @param[in] obj The layout object.
 * @param[in] emission The emitted signal.
 * @param[in] source The signal's source.
 */
static void _send_button_clicked_cb(void *data, Evas_Object *obj, const char *emission, const char *source)
{
	dlog_print(DLOG_INFO, LOG_TAG, "[%s:%d] %s", __FILE__, __LINE__, emission);

	if (data_get_count(-1) == 0 || (data_get_count(-1) == 2 && data_key_exists(-1, BUNDLE_HEADER_KEY)))
		return;

	if (data_send_message())
		view_common_create_popup(s_info.layout, "Message sent.");
	else
		view_common_create_popup(s_info.layout, "Failed to send the message.");

	_set_header_state(false);
	data_clear_bundle();
	elm_genlist_clear(s_info.list);
	elm_layout_signal_emit(s_info.layout, SIGNAL_HEADER_OFF, "");
	elm_layout_signal_emit(s_info.layout, SIGNAL_SEND_OFF, "");
}

/**
 * @brief Internal callback function invoked when the data type radio changes it's state.
 * @param[in] data The user data.
 * @param[in] obj The layout object.
 * @param[in] emission The emitted signal.
 * @param[in] source The signal's source.
 */
static void _data_type_selected_cb(void *data, Evas_Object *obj, const char *emission, const char *source)
{
	Evas_Object *entry = elm_layout_content_get(s_info.layout, PART_SOURCE_VALUE);
	s_info.is_byte = (bool)data;
	_number_only();
	elm_object_text_set(entry, "");

	dlog_print(DLOG_INFO, LOG_TAG, "[%s:%d] %s %d", __FILE__, __LINE__, emission, s_info.is_byte);
}

/**
 * @brief Internal function that adds or remove a header item from the list.
 * @param[in] header_state true - add header, false - remove header.
 */
static void _set_header_state(bool header_state)
{
	s_info.include_header = header_state;

	if (s_info.include_header) {
		if (data_add_header())
			s_info.header = view_list_add_item(s_info.list, s_info.itc_header, ELM_GENLIST_ITEM_TREE, NULL, (void*)strdup(BUNDLE_HEADER_KEY), true, ELM_OBJECT_SELECT_MODE_DEFAULT, NULL, NULL);
	} else {
		data_delete_key(-1, BUNDLE_HEADER_KEY);
		data_delete_key(-1, BUNDLE_HEADER_DATA_KEY);

		view_list_del_item(s_info.list, s_info.header);
	}
}

/**
 * @brief Internal callback function invoked when the add header checkbox changes its state.
 * @param[in] data The user data.
 * @param[in] obj The layout object.
 * @param[in] emission The emitted signal.
 * @param[in] source The signal's source.
 */
static void _header_include_clicked_cb(void *data, Evas_Object *obj, const char *emission, const char *source)
{
	_set_header_state((bool)data);
	dlog_print(DLOG_INFO, LOG_TAG, "[%s:%d] %s %d", __FILE__, __LINE__, emission, s_info.include_header);
}

/**
 * @brief Internal callback function invoked when a data item is realized.
 * @param[in] data User data.
 * @param[in] obj genlist object.
 * @param[in] part The part where the data item's layout will be embedded.
 * @return The created layout.
 */
static Evas_Object *_item_content_get_cb(void *data, Evas_Object *obj, const char *part)
{
	char *key = (char*)data;
	char buf[NAME_MAX] = {0,};

	Evas_Object *item_layout = view_create_layout(obj, EDJ_SOURCE_FILE_NAME, GROUP_SOURCE_ITEM);
	if (!item_layout) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] item_layout == NULL", __FILE__, __LINE__);
		return NULL;
	}

	if (data_is_byte(-1, key)) {
		dlog_print(DLOG_INFO, LOG_TAG, "[%s:%d] INT: [%s = %d](%s)", __FILE__, __LINE__, key, data_get_byte(-1, key), data_get_type_text(-1, key));
		snprintf(buf, NAME_MAX, FORMAT("%d"), key, data_get_byte(-1, key), data_get_type_text(-1, key));
	} else {
		dlog_print(DLOG_INFO, LOG_TAG, "[%s:%d] STR: [%s = %s](%s)", __FILE__, __LINE__, key, data_get_string(-1, key), data_get_type_text(-1, key));
		snprintf(buf, NAME_MAX, FORMAT("%s"), key, data_get_string(-1, key), data_get_type_text(-1, key));
	}

	elm_layout_text_set(item_layout, PART_ITEM, buf);

	return item_layout;
}

/**
 * @brief Internal callback function invoked when a header item is realized.
 * @param[in] data User data.
 * @param[in] obj genlist object.
 * @param[in] part The part where the item's layout will be embedded.
 * @return The created layout.
 */
static Evas_Object *_header_content_get_cb(void *data, Evas_Object *obj, const char *part)
{
	char buf[NAME_MAX] = {0,};
	char *text = (char*)data;

	Evas_Object *item_layout = view_create_layout(obj, EDJ_SOURCE_FILE_NAME, GROUP_SOURCE_HEADER);
	if (!item_layout) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] item_layout == NULL", __FILE__, __LINE__);
		return NULL;
	}

	dlog_print(DLOG_INFO, LOG_TAG, "Header count: %d", data_get_byte(-1, text));
	snprintf(buf, NAME_MAX, "Header count: %d", data_get_byte(-1, text));

	elm_layout_text_set(item_layout, PART_ITEM, buf);
	if (elm_genlist_item_expanded_get(s_info.header)) {
		elm_layout_signal_emit(item_layout, SIGNAL_ARROW_STATE_CHANGE, "");
	}

	return item_layout;
}

/**
 * @brief Internal callback function invoked when a header data item is realized.
 * @param[in] data User data.
 * @param[in] obj genlist object.
 * @param[in] part The part where the item's layout will be embedded.
 * @return The created layout.
 */
static Evas_Object *_header_data_content_get_cb(void *data, Evas_Object *obj, const char *part)
{
	char *text = (char*)data;
	char **tokens = NULL;
	char buf[NAME_MAX] = {0,};

	Evas_Object *item_layout = view_create_layout(obj, EDJ_SOURCE_FILE_NAME, GROUP_SOURCE_HEADER_DATA);
	if (!item_layout) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] item_layout == NULL", __FILE__, __LINE__);
		return NULL;
	}

	tokens = eina_str_split(text, "\2", 2);
	snprintf(buf, NAME_MAX, "<type>%s(%s)</type>", tokens[0], tokens[1]);
	elm_layout_text_set(item_layout, PART_ITEM, buf);

	return item_layout;
}

/**
 * @brief Internal callback function invoked when a TREE item is expanded.
 * @param[in] data User data.
 * @param[in] obj genlist object.
 * @param[in] event_info The expanded item.
 */
static void _tree_item_expanded(void *data, Evas_Object *obj, void *event_info)
{
	Elm_Object_Item *it_parent = (Elm_Object_Item*) event_info;
	Elm_Genlist_Item_Class *itc = data;
	Evas_Object *header = elm_object_item_part_content_get(it_parent, "elm.swallow.content");
	int i;
	int len;
	const char **values  = data_get_string_array(-1, BUNDLE_HEADER_DATA_KEY, &len);

	elm_layout_signal_emit(header, SIGNAL_ARROW_STATE_CHANGE, "");

	for (i = 0; i < len; i++)
		view_list_add_item(obj, itc, ELM_GENLIST_ITEM_NONE, it_parent, (void*)strdup(values[i]), false, ELM_OBJECT_SELECT_MODE_NONE, NULL, NULL);
}

/**
 * @brief Internal callback function invoked when the text in the key or value entries changes. It is used to check if the entries are empty. If they are, the add button is deactivated.
 * @param data User data.
 * @param obj The entry object.
 * @param event_info The event information.
 */
static void _entry_text_changed_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *entry_1 = elm_layout_content_get(s_info.layout, PART_SOURCE_KEY);
	Evas_Object *entry_2 = elm_layout_content_get(s_info.layout, PART_SOURCE_VALUE);
	static bool can_add = false;
	bool current = false;

	current = elm_entry_is_empty(entry_1) || elm_entry_is_empty(entry_2);

	if (current != can_add) {
		can_add = current;

		if (current)
			elm_layout_signal_emit(s_info.layout, SIGNAL_ADD_OFF, "");
		else
			elm_layout_signal_emit(s_info.layout, SIGNAL_ADD_ON, "");
	}
}

/**
 * @brief Callback function ivoked when a list item is selected. This function is used to update the texts in the key and value entries.
 * @param data User data.
 * @param obj The genlist
 * @param event_info The selected item.
 */
static void _list_item_selected_cb(void *data, Evas_Object *obj, void *event_info)
{
	Elm_Object_Item *item = (Elm_Object_Item *)event_info;
	char *key = (char *)elm_object_item_data_get(item);
	char buf[NAME_MAX] = {0,};
	Evas_Object *entry = elm_layout_content_get(s_info.layout, PART_SOURCE_KEY);

	elm_object_text_set(entry, key);
	entry = elm_layout_content_get(s_info.layout, PART_SOURCE_VALUE);
	s_info.is_byte = data_is_byte(-1, key);

	if (s_info.is_byte) {
		snprintf(buf, NAME_MAX, "%d", data_get_byte(-1, key));
		elm_object_text_set(entry, buf);
		elm_layout_signal_emit(s_info.layout, SIGNAL_SET_BYTE, "");
	} else {
		elm_object_text_set(entry, data_get_string(-1, key));
		elm_layout_signal_emit(s_info.layout, SIGNAL_SET_STRING, "");
	}

	_number_only();
}
