/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <Elementary.h>
#include <dlog.h>
#include <geofence_manager.h>
#include <context_trigger.h>

#include "$(appName).h"
#include "data.h"

#define INVALID_RULE_ID -1
rule_data_s rule_info[RULE_LAST] = {
	{INVALID_RULE_ID,
	 false,
	 "Low Battery",
	 "This rule will help you charge battery by notifying when the battery is running out of without charger.",
	 "Battery level is low. Please charge your battery.",
	 CONTEXT_TRIGGER_ERROR_NONE},
	{INVALID_RULE_ID,
	 false,
	 "Important Call",
	 "This rule will notify you when you are making/taking a call to/from someone important.",
	 "This person has been ranked in the Top 10 frequently contacted with you.",
	 CONTEXT_TRIGGER_ERROR_NONE},
	{INVALID_RULE_ID,
	 false,
	 "The 5th-day-no-driving System",
	 "In the morning, this rule will notify you when your car is restricted by the 5th-day-no-driving system.",
	 "Today is no driving day. You should use public transportation today.",
	 CONTEXT_TRIGGER_ERROR_NONE},
	{INVALID_RULE_ID,
	 false,
	 "Home Wi-Fi",
	 "This rule will help you not forget to enable Wi-Fi when you arrive home.",
	 "You arrived home. Do not forget to turn on your Wi-Fi.",
	 CONTEXT_TRIGGER_ERROR_NONE}
};

static int add_battery_rule(void);
static int add_call_rule(void);
static int add_driving_rule(void);
static int add_home_rule(void);
static bool place_cb(int place_id, const char *place_name, int place_index, int place_cnt, void *user_data);

/**
 * @brief Initialization function for data module.
 */
void data_initialize(void)
{
	/*
	 * If you need to initialize application data,
	 * please use this function.
	 */
}

/**
 * @brief Finalization function for data module.
 */
void data_finalize(void)
{
	/*
	 * If you need to finalize application data,
	 * please use this function.
	 */
}

/**
 * @brief Creates and register contextual rules.
 */
void add_rules(void)
{
	/* Add rules */
	rule_info[RULE_BATTERY].id = add_battery_rule();
	rule_info[RULE_CALL].id = add_call_rule();
	rule_info[RULE_DRIVING].id = add_driving_rule();
	rule_info[RULE_HOME].id = add_home_rule();
}

/**
 * @brief Unregisters contextual rules.
 * @details All the registered rules are disabled and removed.
 *          Enabled rules can be removed after being disabled.
 */
void remove_rules(void)
{
	/* Get rules */
	int enabled_rule_cnt = 0;
	int disabled_rule_cnt = 0;
	int *enabled_rule_ids = NULL;
	int *disabled_rule_ids = NULL;
	int error = context_trigger_get_own_rule_ids(&enabled_rule_ids, &enabled_rule_cnt, &disabled_rule_ids, &disabled_rule_cnt);
	if (error != CONTEXT_TRIGGER_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get rule ids");
		return;
	}

	/* Disable and remove enabled rules */
	int i = 0;
	for (i = 0; i < enabled_rule_cnt; i++) {
		context_trigger_disable_rule(enabled_rule_ids[i]);
		context_trigger_remove_rule(enabled_rule_ids[i]);
	}

	/* Remove disabled rules */
	for (i = 0; i < disabled_rule_cnt; i++)
		context_trigger_remove_rule(disabled_rule_ids[i]);

	if (enabled_rule_ids) {
		free(enabled_rule_ids);
		enabled_rule_ids = NULL;
	}

	if (disabled_rule_ids) {
		free(disabled_rule_ids);
		disabled_rule_ids = NULL;
	}
}

/**
 * @brief Activates the rule designated by rule id.
 */
int enable_rule(int rule_id)
{
	int error = context_trigger_enable_rule(rule_id);
	if (error != CONTEXT_TRIGGER_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to enable rule%d: %d", rule_id, error);

	return error;
}

/**
 * @brief Deactivates the rule designated by rule id.
 */
int disable_rule(int rule_id)
{
	int error = context_trigger_disable_rule(rule_id);
	if (error != CONTEXT_TRIGGER_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to disable rule%d: %d", rule_id, error);

	return error;
}

/**
 * @brief Creates and registers 'Low Battery' rule.
 * @details This rule notifies the user when the battery is running out of without charger.
 */
static int add_battery_rule(void)
{
	int rule_id = 0;
	context_trigger_rule_h rule = NULL;
	context_trigger_rule_entry_h battery_e = NULL;

	bool battery_e_supported;
	context_trigger_rule_event_is_supported(CONTEXT_TRIGGER_EVENT_BATTERY, &battery_e_supported);
	if (!battery_e_supported) {
		rule_info[RULE_BATTERY].result = CONTEXT_TRIGGER_ERROR_NOT_SUPPORTED;
		return rule_id;
	}

	context_trigger_rule_create(CONTEXT_TRIGGER_LOGICAL_CONJUNCTION, &rule);
	context_trigger_rule_set_description(rule, rule_info[RULE_BATTERY].description);

	context_trigger_rule_event_create(CONTEXT_TRIGGER_EVENT_BATTERY, CONTEXT_TRIGGER_LOGICAL_CONJUNCTION, &battery_e);
	context_trigger_rule_entry_add_key(battery_e, CONTEXT_TRIGGER_LOGICAL_DISJUNCTION, CONTEXT_TRIGGER_LEVEL);
	context_trigger_rule_entry_add_comparison_string(battery_e, CONTEXT_TRIGGER_LEVEL, CONTEXT_TRIGGER_EQUAL_TO, CONTEXT_TRIGGER_EMPTY);
	context_trigger_rule_entry_add_comparison_string(battery_e, CONTEXT_TRIGGER_LEVEL, CONTEXT_TRIGGER_EQUAL_TO, CONTEXT_TRIGGER_CRITICAL);
	context_trigger_rule_entry_add_comparison_string(battery_e, CONTEXT_TRIGGER_LEVEL, CONTEXT_TRIGGER_EQUAL_TO, CONTEXT_TRIGGER_LOW);
	context_trigger_rule_entry_add_key(battery_e, CONTEXT_TRIGGER_LOGICAL_CONJUNCTION, CONTEXT_TRIGGER_IS_CHARGING);
	context_trigger_rule_entry_add_comparison_int(battery_e, CONTEXT_TRIGGER_IS_CHARGING, CONTEXT_TRIGGER_EQUAL_TO, CONTEXT_TRIGGER_FALSE);
	context_trigger_rule_add_entry(rule, battery_e);

	context_trigger_rule_set_action_notification(rule, rule_info[RULE_BATTERY].name, rule_info[RULE_BATTERY].msg, NULL, NULL);

	rule_info[RULE_BATTERY].result = context_trigger_add_rule(rule, &rule_id);
	if (rule_info[RULE_BATTERY].result != CONTEXT_TRIGGER_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to add '%s' rule: %d", rule_info[RULE_BATTERY].name, rule_info[RULE_BATTERY].result);

	context_trigger_rule_entry_destroy(battery_e);
	context_trigger_rule_destroy(rule);

	return rule_id;
}

/**
 * @brief Creates and registers 'Important Call' rule.
 * @details This rule notifies when the user is making a call to or taking a call from someone who contacts frequently.
 */
static int add_call_rule(void)
{
	int rule_id = 0;
	context_trigger_rule_h rule = NULL;
	context_trigger_rule_entry_h call_e = NULL;
	context_trigger_rule_entry_h contact_c = NULL;

	bool call_e_supported;
	context_trigger_rule_event_is_supported(CONTEXT_TRIGGER_EVENT_CALL, &call_e_supported);

	bool contact_c_supported;
	context_trigger_rule_condition_is_supported(CONTEXT_TRIGGER_CONDITION_COMMUNICATION_FREQUENCY, &contact_c_supported);

	if (!call_e_supported || !contact_c_supported) {
		rule_info[RULE_CALL].result = CONTEXT_TRIGGER_ERROR_NOT_SUPPORTED;
		return rule_id;
	}

	context_trigger_rule_create(CONTEXT_TRIGGER_LOGICAL_CONJUNCTION, &rule);
	context_trigger_rule_set_description(rule, rule_info[RULE_CALL].description);

	context_trigger_rule_event_create(CONTEXT_TRIGGER_EVENT_CALL, CONTEXT_TRIGGER_LOGICAL_CONJUNCTION, &call_e);
	context_trigger_rule_entry_add_key(call_e, CONTEXT_TRIGGER_LOGICAL_CONJUNCTION, CONTEXT_TRIGGER_STATE);
	context_trigger_rule_entry_add_comparison_string(call_e, CONTEXT_TRIGGER_STATE, CONTEXT_TRIGGER_EQUAL_TO, CONTEXT_TRIGGER_CONNECTED);
	context_trigger_rule_add_entry(rule, call_e);

	context_trigger_rule_condition_create(CONTEXT_TRIGGER_CONDITION_COMMUNICATION_FREQUENCY, CONTEXT_TRIGGER_LOGICAL_CONJUNCTION, &contact_c);
	context_trigger_rule_entry_add_option(contact_c, CONTEXT_TRIGGER_ADDRESS, CONTEXT_TRIGGER_ADDRESS);
	context_trigger_rule_entry_add_key(contact_c, CONTEXT_TRIGGER_LOGICAL_CONJUNCTION, CONTEXT_TRIGGER_RANK);
	context_trigger_rule_entry_add_comparison_int(contact_c, CONTEXT_TRIGGER_RANK, CONTEXT_TRIGGER_LESS_THAN_OR_EQUAL_TO, 10);
	context_trigger_rule_add_entry(rule, contact_c);

	context_trigger_rule_set_action_notification(rule, rule_info[RULE_CALL].name, rule_info[RULE_CALL].msg, NULL, NULL);

	rule_info[RULE_CALL].result = context_trigger_add_rule(rule, &rule_id);
	if (rule_info[RULE_CALL].result != CONTEXT_TRIGGER_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to add '%s' rule: %d", rule_info[RULE_CALL].name, rule_info[RULE_CALL].result);

	context_trigger_rule_entry_destroy(call_e);
	context_trigger_rule_entry_destroy(contact_c);
	context_trigger_rule_destroy(rule);

	return rule_id;
}

/**
 * @brief Creates and registers 'The 5th-day-no-driving-rule System' rule.
 * @details In the morning, this rule notifies the user,
 *          when his/her car is restricted by the 5th-day-no-driving-system.
 */
static int add_driving_rule(void)
{
	int rule_id = 0;
	context_trigger_rule_h rule = NULL;
	context_trigger_rule_entry_h time_e = NULL;
	context_trigger_rule_entry_h time_c = NULL;

	bool time_e_supported;
	context_trigger_rule_event_is_supported(CONTEXT_TRIGGER_EVENT_TIME, &time_e_supported);

	bool time_c_supported;
	context_trigger_rule_condition_is_supported(CONTEXT_TRIGGER_CONDITION_TIME, &time_c_supported);

	if (!time_e_supported || !time_c_supported) {
		rule_info[RULE_DRIVING].result = CONTEXT_TRIGGER_ERROR_NOT_SUPPORTED;
		return rule_id;
	}

	context_trigger_rule_create(CONTEXT_TRIGGER_LOGICAL_CONJUNCTION, &rule);
	context_trigger_rule_set_description(rule, rule_info[RULE_DRIVING].description);

	context_trigger_rule_event_create(CONTEXT_TRIGGER_EVENT_TIME, CONTEXT_TRIGGER_LOGICAL_CONJUNCTION, &time_e);
	context_trigger_rule_entry_add_key(time_e, CONTEXT_TRIGGER_LOGICAL_CONJUNCTION, CONTEXT_TRIGGER_TIME_OF_DAY);
	context_trigger_rule_entry_add_comparison_int(time_e, CONTEXT_TRIGGER_TIME_OF_DAY, CONTEXT_TRIGGER_EQUAL_TO, 7 * 60);
	context_trigger_rule_entry_add_key(time_e, CONTEXT_TRIGGER_LOGICAL_CONJUNCTION, CONTEXT_TRIGGER_DAY_OF_WEEK);
	context_trigger_rule_entry_add_comparison_string(time_e, CONTEXT_TRIGGER_DAY_OF_WEEK, CONTEXT_TRIGGER_EQUAL_TO, CONTEXT_TRIGGER_WEEKDAY);
	context_trigger_rule_add_entry(rule, time_e);

	context_trigger_rule_condition_create(CONTEXT_TRIGGER_CONDITION_TIME, CONTEXT_TRIGGER_LOGICAL_CONJUNCTION, &time_c);
	context_trigger_rule_entry_add_key(time_c, CONTEXT_TRIGGER_LOGICAL_DISJUNCTION, CONTEXT_TRIGGER_DAY_OF_MONTH);
	context_trigger_rule_entry_add_comparison_int(time_c, CONTEXT_TRIGGER_DAY_OF_MONTH, CONTEXT_TRIGGER_EQUAL_TO, 1);
	context_trigger_rule_entry_add_comparison_int(time_c, CONTEXT_TRIGGER_DAY_OF_MONTH, CONTEXT_TRIGGER_EQUAL_TO, 6);
	context_trigger_rule_entry_add_comparison_int(time_c, CONTEXT_TRIGGER_DAY_OF_MONTH, CONTEXT_TRIGGER_EQUAL_TO, 11);
	context_trigger_rule_entry_add_comparison_int(time_c, CONTEXT_TRIGGER_DAY_OF_MONTH, CONTEXT_TRIGGER_EQUAL_TO, 16);
	context_trigger_rule_entry_add_comparison_int(time_c, CONTEXT_TRIGGER_DAY_OF_MONTH, CONTEXT_TRIGGER_EQUAL_TO, 21);
	context_trigger_rule_entry_add_comparison_int(time_c, CONTEXT_TRIGGER_DAY_OF_MONTH, CONTEXT_TRIGGER_EQUAL_TO, 26);
	context_trigger_rule_add_entry(rule, time_c);

	context_trigger_rule_set_action_notification(rule, rule_info[RULE_DRIVING].name, rule_info[RULE_DRIVING].msg, NULL, NULL);

	rule_info[RULE_DRIVING].result = context_trigger_add_rule(rule, &rule_id);
	if (rule_info[RULE_DRIVING].result != CONTEXT_TRIGGER_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to add '%s' rule: %d", rule_info[RULE_DRIVING].name, rule_info[RULE_DRIVING].result);

	context_trigger_rule_entry_destroy(time_e);
	context_trigger_rule_entry_destroy(time_c);
	context_trigger_rule_destroy(rule);

	return rule_id;
}

/**
 * @brief Creates and registers 'Home Wi-Fi' rule.
 * @details This rule notifies the user if Wi-Fi is disabled when s/he arrives home.
 */
static int add_home_rule(void)
{
	int rule_id = 0;
	int error;
	context_trigger_rule_h rule = NULL;
	context_trigger_rule_entry_h place_e = NULL;
	context_trigger_rule_entry_h wifi_c = NULL;

	bool place_e_supported;
	context_trigger_rule_event_is_supported(CONTEXT_TRIGGER_EVENT_PLACE, &place_e_supported);

	bool wifi_c_supported;
	context_trigger_rule_condition_is_supported(CONTEXT_TRIGGER_CONDITION_WIFI, &wifi_c_supported);

	if (!place_e_supported || !wifi_c_supported) {
		rule_info[RULE_HOME].result = CONTEXT_TRIGGER_ERROR_NOT_SUPPORTED;
		return rule_id;
	}

	bool supported;
	context_trigger_rule_event_is_supported(CONTEXT_TRIGGER_EVENT_PLACE, &supported);
	if (!supported) {
		rule_info[RULE_HOME].result = CONTEXT_TRIGGER_ERROR_NOT_SUPPORTED;
		return INVALID_RULE_ID;
	}

	/* Get place id of 'Home' */
	int home_id = 0;
	geofence_manager_h geo_mgr;
	error = geofence_manager_create(&geo_mgr);
	if (error != GEOFENCE_MANAGER_ERROR_NONE) {
		rule_info[RULE_HOME].result = error;
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to create geofence manager");
		return INVALID_RULE_ID;
	}
	error = geofence_manager_foreach_place_list(geo_mgr, place_cb, &home_id);
	if (error != GEOFENCE_MANAGER_ERROR_NONE) {
		rule_info[RULE_HOME].result = error;
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to retrieve place list");
		return INVALID_RULE_ID;
	}

	if (home_id == 0)
		dlog_print(DLOG_ERROR, LOG_TAG, "Place not found");

	geofence_manager_destroy(geo_mgr);

	context_trigger_rule_create(CONTEXT_TRIGGER_LOGICAL_CONJUNCTION, &rule);
	context_trigger_rule_set_description(rule, rule_info[RULE_HOME].description);

	context_trigger_rule_event_create(CONTEXT_TRIGGER_EVENT_PLACE, CONTEXT_TRIGGER_LOGICAL_CONJUNCTION, &place_e);
	context_trigger_rule_entry_add_option_int(place_e, CONTEXT_TRIGGER_PLACE_ID, home_id);
	context_trigger_rule_entry_add_key(place_e, CONTEXT_TRIGGER_LOGICAL_CONJUNCTION, CONTEXT_TRIGGER_EVENT);
	context_trigger_rule_entry_add_comparison_string(place_e, CONTEXT_TRIGGER_EVENT, CONTEXT_TRIGGER_EQUAL_TO, CONTEXT_TRIGGER_IN);
	context_trigger_rule_add_entry(rule, place_e);

	context_trigger_rule_condition_create(CONTEXT_TRIGGER_CONDITION_WIFI, CONTEXT_TRIGGER_LOGICAL_CONJUNCTION, &wifi_c);
	context_trigger_rule_entry_add_key(wifi_c, CONTEXT_TRIGGER_LOGICAL_CONJUNCTION, CONTEXT_TRIGGER_STATE);
	context_trigger_rule_entry_add_comparison_string(wifi_c, CONTEXT_TRIGGER_STATE, CONTEXT_TRIGGER_EQUAL_TO, CONTEXT_TRIGGER_DISABLED);
	context_trigger_rule_add_entry(rule, wifi_c);

	context_trigger_rule_set_action_notification(rule, rule_info[RULE_HOME].name, rule_info[RULE_HOME].msg, NULL, NULL);

	rule_info[RULE_HOME].result = context_trigger_add_rule(rule, &rule_id);
	if (rule_info[RULE_HOME].result != CONTEXT_TRIGGER_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to add '%s' rule: %d", rule_info[RULE_HOME].name, rule_info[RULE_HOME].result);

	context_trigger_rule_entry_destroy(place_e);
	context_trigger_rule_entry_destroy(wifi_c);
	context_trigger_rule_destroy(rule);

	return rule_id;
}

/**
 * @brief This function is called when the place list is requested.
 * @details Finds place id of 'Home' for 'Home Wi-Fi' rule.
 */
static bool place_cb(int place_id, const char *place_name, int place_index, int place_cnt, void *user_data)
{
	if (strcmp(place_name, "Home") == 0) {
		int *home_id = (int *)user_data;
		*home_id = place_id;

		return false;
	}
	return true;
}
