/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <dlog.h>
#include <app.h>
#include <efl_extension.h>
#include <Elementary.h>
#include <context_trigger.h>

#include "$(appName).h"
#include "view.h"
#include "data.h"

#define ERR_MSG_PERMISSION_DENIED "This rule cannot be added. Please check the privileges needed for the rule."
#define ERR_MSG_NOT_SUPPORTED "This rule is not supported on this device."
#define ERR_MSG_INVALID_RULE "This rule is invalid."
#define ERR_MSG_DEFAULT "The application failed to add this rule."

static struct view_info {
	Evas_Object *win;
	Evas_Object *conform;
	Evas_Object *nf;
	Evas_Object *genlist;
} s_info = {
	.win = NULL,
	.conform = NULL,
	.nf = NULL,
	.genlist = NULL,
};

static void _win_delete_request_cb(void *data, Evas_Object *obj, void *event_info);
static void _naviframe_back_cb(void *data, Evas_Object *obj, void *event_info);
static char *_gl_text_get_cb(void *data, Evas_Object *obj, const char *part);
static Evas_Object *_gl_content_get_cb(void *data, Evas_Object *obj, const char *part);
static void _gl_selected_cb(void *data, Evas_Object *obj, void *event_info);
static void _check_changed_cb(void *data, Evas_Object *obj, void *event_info);
static void _popup_btn_clicked_cb(void *data, Evas_Object *obj, void *event_info);

/**
 * @brief Creates essential objects: window, conformant, naviframe and genlist.
 */
Eina_Bool view_create(void *user_data)
{
	/* Create the window */
	s_info.win = view_create_win(PACKAGE);
	if (s_info.win == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create a window.");
		return EINA_FALSE;
	}

	/* Create the conformant */
	s_info.conform = view_create_conformant(s_info.win);
	if (s_info.conform == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create a conformant");
		return EINA_FALSE;
	}

	/* Create the naviframe */
	s_info.nf = view_create_naviframe(s_info.conform);
	if (s_info.nf == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create a naviframe");
		return EINA_FALSE;
	}

	/* Create the genlist */
	s_info.genlist = view_create_genlist(s_info.nf);
	if (s_info.genlist == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create a genlist");
		return EINA_FALSE;
	}

	elm_naviframe_item_push(s_info.nf, "Rules", NULL, NULL, s_info.genlist, NULL);

	/* Show the window after main view is set up */
	evas_object_show(s_info.win);
	return EINA_TRUE;
}

/**
 * @brief Creates a basic window named package.
 * @param[in] pkg_name Name of the window
 */
Evas_Object *view_create_win(const char *pkg_name)
{
	Evas_Object *win = NULL;

	/*
	 * Window
	 * Create and initialize elm_win.
	 * elm_win is mandatory to manipulate the window.
	 */
	win = elm_win_util_standard_add(pkg_name, pkg_name);
	elm_win_conformant_set(win, EINA_TRUE);
	elm_win_indicator_mode_set(win, ELM_WIN_INDICATOR_SHOW);
	elm_win_indicator_opacity_set(win, ELM_WIN_INDICATOR_OPAQUE);
	elm_win_autodel_set(win, EINA_TRUE);

	/* Rotation setting */
	if (elm_win_wm_rotation_supported_get(win)) {
		int rots[4] = { 0, 90, 180, 270 };
		elm_win_wm_rotation_available_rotations_set(win, (const int *)(&rots), 4);
	}

	evas_object_smart_callback_add(win, "delete,request", _win_delete_request_cb, NULL);

	return win;
}

/**
 * @brief Creates a conformant.
 * @param[in] win The object to which you want to set this conformant
 * Conformant is mandatory for base GUI to have proper size
 */
Evas_Object *view_create_conformant(Evas_Object *win)
{
	/*
	 * Conformant
	 * Create and initialize elm_conformant.
	 * elm_conformant is mandatory for base GUI to have proper size
	 * when indicator or virtual keypad is visible.
	 */
	Evas_Object *conform = NULL;

	if (win == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "window is NULL.");
		return NULL;
	}

	conform = elm_conformant_add(win);
	evas_object_size_hint_weight_set(conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(win, conform);

	evas_object_show(conform);

	return conform;
}

/**
 * @brief Creates a naviframe object.
 * @param[in] parent The object to which you want to set this naviframe
 */
Evas_Object *view_create_naviframe(Evas_Object *parent)
{
	Evas_Object *nf = elm_naviframe_add(parent);

	eext_object_event_callback_add(nf, EEXT_CALLBACK_BACK, _naviframe_back_cb, NULL);
	eext_object_event_callback_add(nf, EEXT_CALLBACK_MORE, eext_naviframe_more_cb, NULL);

	elm_object_content_set(parent, nf);
	evas_object_show(nf);

	return nf;
}

/**
 * @brief Creates a genlist object which shows contextual rules.
 * @param[in] parent The object to which you want to set this genlist
 */
Evas_Object *view_create_genlist(Evas_Object *parent)
{
	int index;
	Evas_Object *genlist;
	Elm_Genlist_Item_Class *itc;

	/* Create the item class */
	itc = elm_genlist_item_class_new();
	itc->item_style = "multiline";
	itc->func.text_get = _gl_text_get_cb;
	itc->func.content_get = _gl_content_get_cb;

	/* Create the genlist */
	genlist = elm_genlist_add(parent);
	elm_genlist_mode_set(genlist, ELM_LIST_COMPRESS);
	evas_object_size_hint_weight_set(genlist, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(genlist, EVAS_HINT_FILL, EVAS_HINT_FILL);

	for (index = RULE_FIRST; index < RULE_LAST; index++) {
		/* Append the rule items */
		elm_genlist_item_append(genlist, itc, (void *)&rule_info[index], NULL, ELM_GENLIST_ITEM_NONE, _gl_selected_cb, NULL);
	}

	elm_genlist_item_class_free(itc);
	evas_object_show(genlist);

	return genlist;
}

/**
 * @brief Destroys window and frees its resources.
 */
void view_destroy(void)
{
	if (s_info.win == NULL)
		return;

	evas_object_del(s_info.win);
}

/**
 * @brief Internal callback function invoked when the main window needs to be destroyed.
 * @param[in] data The user data passed to the evas_object_smart_callback_add() function
 * @param[in] obj The object invoking this callback function
 * @param[in] event_info The structure containing the information on this event
 */
static void _win_delete_request_cb(void *data, Evas_Object *obj, void *event_info)
{
	/*
	 * Write your code here for smart callback.
	 */
}

/**
 * @brief Internal callback function invoked when HW back button pressed.
 * @param[in] data The user data passed to the eext_object_event_callback_add() function
 * @param[in] obj The object invoking this callback function
 * @param[in] event_info The structure containing the information on this event
 */
static void _naviframe_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	elm_win_lower(s_info.win);
}

/**
 * @brief Internal callback function invoked when genlist item displayed.
 * This function is assigned to the func.text_get handler of genlist item's class structure.
 * The item's area consists if two labels:
 * - top label where the contextual rule's title is displayed,
 * - bottom label where the contextual rule's description is displayed.
 * @param[in] data The user data passed to the elm_genlist_item_append() function
 * @param[in] obj The object invoking this callback function
 * @param[in] part The name of the item's part being rendered
 */
static char *_gl_text_get_cb(void *data, Evas_Object *obj, const char *part)
{
	rule_data_s *info = (rule_data_s *) data;

	if (!strcmp(part, "elm.text"))
		return strdup(info->name);
	else if (!strcmp(part, "elm.text.multiline"))
		return strdup(info->description);

	return NULL;
}

/**
 * @brief Internal callback function invoked when genlist item displayed.
 * This function is assigned to the func.content_get handler of genlist item's class structure.
 * @param[in] data The user data passed to the elm_genlist_item_append() function
 * @param[in] obj The object invoking this callback function
 * @param[in] part The name of the item's part being rendered
 */
static Evas_Object *_gl_content_get_cb(void *data, Evas_Object *obj, const char *part)
{
	rule_data_s *info = (rule_data_s *) data;

	if (!strcmp(part, "elm.swallow.end")) {
		Evas_Object *check = elm_check_add(obj);

		evas_object_smart_callback_add(check, "changed", _check_changed_cb, info);
		evas_object_propagate_events_set(check, EINA_FALSE);

		evas_object_size_hint_weight_set(check, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
		evas_object_size_hint_align_set(check, EVAS_HINT_FILL, EVAS_HINT_FILL);
		elm_object_style_set(check, "on&off");
		elm_check_state_set(check, false);
		elm_check_state_pointer_set(check, &info->enabled);

		if (info->result != CONTEXT_TRIGGER_ERROR_NONE)
			elm_object_disabled_set(check, EINA_TRUE);

		return check;
	}
	return NULL;
}

/**
 * @brief Internal callback function invoked when the contextual rule genlist item is selected.
 * Depending on the current rule state, the rule can be activated or deactivated.
 * @param[in] data The user data passed to the elm_genlist_item_append() function
 * @param[in] obj The object invoking this callback function
 * @param[in] event_info The structure containing the information on this event
 */
static void _gl_selected_cb(void *data, Evas_Object *obj, void *event_info)
{
	Elm_Object_Item *it = (Elm_Object_Item *) event_info;
	elm_genlist_item_selected_set(it, false);

	rule_data_s *info = elm_object_item_data_get(it);

	if (info->result == CONTEXT_TRIGGER_ERROR_NONE) {
		int error = CONTEXT_TRIGGER_ERROR_NONE;
		Evas_Object *check = elm_object_item_part_content_get(it, "elm.swallow.end");

		if (elm_check_state_get(check)) {
			error = disable_rule(info->id);
			if (error == CONTEXT_TRIGGER_ERROR_NONE)
				elm_check_state_set(check, !info->enabled);
		} else {
			error = enable_rule(info->id);
			if (error == CONTEXT_TRIGGER_ERROR_NONE)
				elm_check_state_set(check, !info->enabled);
		}
	} else {
		view_create_error_popup(info);
	}
}

/**
 * @brief Internal callback function invoked when check button state is changed.
 * Depending on the current rule state, the rule can be activated or deactivated.
 * @param[in] data The user data passed to the evas_object_smart_callback_add() function
 * @param[in] obj The object invoking this callback function
 * @param[in] event_info The structure containing the information on this event
 */
static void _check_changed_cb(void *data, Evas_Object *obj, void *event_info)
{
	rule_data_s *info = (rule_data_s *) data;
	int error = CONTEXT_TRIGGER_ERROR_NONE;

	if (elm_check_state_get(obj)) {
		error = enable_rule(info->id);
		if (error != CONTEXT_TRIGGER_ERROR_NONE)
			elm_check_state_set(obj, !info->enabled);
	} else {
		error = disable_rule(info->id);
		if (error != CONTEXT_TRIGGER_ERROR_NONE)
			elm_check_state_set(obj, !info->enabled);
	}
}

/**
 * @brief Creates a popup which show error message.
 * @param[in] data The rule information data
 */
void view_create_error_popup(void *data)
{
	rule_data_s *info = (rule_data_s *) data;

	/* Check error code */
	char *err_msg = NULL;

	switch (info->result) {
	case CONTEXT_TRIGGER_ERROR_PERMISSION_DENIED:
		err_msg = ERR_MSG_PERMISSION_DENIED;
		break;
	case CONTEXT_TRIGGER_ERROR_NOT_SUPPORTED:
		err_msg = ERR_MSG_NOT_SUPPORTED;
		break;
	case CONTEXT_TRIGGER_ERROR_INVALID_RULE:
		err_msg = ERR_MSG_INVALID_RULE;
		break;
	default:
		err_msg = ERR_MSG_DEFAULT;
		break;
	}

	/* Create the popup */
	Evas_Object *popup = elm_popup_add(s_info.win);
	elm_popup_align_set(popup, ELM_NOTIFY_ALIGN_FILL, 1.0);
	eext_object_event_callback_add(popup, EEXT_CALLBACK_BACK, eext_popup_back_cb, NULL);
	elm_object_part_text_set(popup, "title,text", info->name);

	elm_object_text_set(popup, err_msg);

	Evas_Object *btn = elm_button_add(popup);
	elm_object_style_set(btn, "popup");
	elm_object_text_set(btn, "OK");
	elm_object_part_content_set(popup, "button1", btn);
	evas_object_smart_callback_add(btn, "clicked", _popup_btn_clicked_cb, popup);
	evas_object_show(popup);
}

/**
 * @brief Internal callback function invoked when the button in the popup is clicked.
 * @param[in] data The user data passed to the evas_object_smart_callback_add() function
 * @param[in] obj The object invoking this callback function
 * @param[in] event_info The structure containing the information on this event
 */
static void _popup_btn_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *popup = data;
	evas_object_del(popup);
}
