/*
 * Samsung API
 * Copyright (c) 2009-2015 Samsung Electronics Co., Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */



#include <tizen.h>
#include <time.h>

#include "log.h"
#include "main.h"
#include "eom_sample.h"
#include "view.h"

static void win_delete_request_cb(void *data , Evas_Object *obj , void *event_info)
{
	appdata_s *ad = NULL;

	ret_if(!data);

	ad = data;

	if (ad->vd) free(ad->vd);

	ui_app_exit();
}

static void win_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = NULL;

	ret_if(!data);

	_D("hide EOM");

	ad = data;

	/* Let window go to hide state. */
	elm_win_lower(ad->win);
}

static Evas_Object *_create_win(appdata_s *ad)
{
	Evas_Object *win = NULL;

	retv_if(!ad, NULL);

	win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	retv_if(!win, NULL);

	elm_win_conformant_set(win, EINA_TRUE);
	elm_win_autodel_set(win, EINA_TRUE);

	if (elm_win_wm_rotation_supported_get(win)) {
		int rots[4] = { 0, 90, 180, 270 };
		elm_win_wm_rotation_available_rotations_set(win, (const int *)(&rots), 4);
	}

	evas_object_smart_callback_add(win, "delete,request", win_delete_request_cb, ad);
	eext_object_event_callback_add(win, EEXT_CALLBACK_BACK, win_back_cb, ad);

	return win;
}

static void _destroy_window(Evas_Object *win)
{
	ret_if(!win);

	evas_object_del(win);
}

static Evas_Object *_create_conform(appdata_s *ad)
{
	Evas_Object *conform = NULL;

	retv_if(!ad, NULL);

	conform = elm_conformant_add(ad->win);
	retv_if(!conform, NULL);

	elm_win_indicator_mode_set(ad->win, ELM_WIN_INDICATOR_SHOW);
	elm_win_indicator_opacity_set(ad->win, ELM_WIN_INDICATOR_OPAQUE);
	evas_object_size_hint_weight_set(conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(ad->win, conform);
	evas_object_show(conform);

	return conform;
}

static void _destroy_conform(Evas_Object *conform)
{
	ret_if(!conform);

	evas_object_del(conform);
}

static void _set_indicator_bg(appdata_s *ad)
{
	Evas_Object *bg = NULL;

	ret_if(!ad);

	bg = elm_bg_add(ad->conform);
	ret_if(!bg);

	elm_object_style_set(bg, "indicator/headerbg");
	elm_object_part_content_set(ad->conform, "elm.swallow.indicator_bg", bg);
	evas_object_show(bg);
}

static Evas_Object *_create_navi(appdata_s *ad)
{
	Evas_Object *nf = NULL;

	retv_if(!ad, NULL);

	nf = elm_naviframe_add(ad->conform);
	retv_if(!nf, NULL);

	elm_object_content_set(ad->conform, nf);
	evas_object_show(nf);

	return nf;
}

static void _destroy_navi(Evas_Object *nf)
{
	ret_if(!nf);

	evas_object_del(nf);
}

static Evas_Object *_create_layout(appdata_s *ad)
{
	Evas_Object *layout = NULL;

	retv_if(!ad, NULL);

	layout = elm_layout_add(ad->nf);
	retv_if(!layout, NULL);

	elm_layout_file_set(layout, LAYOUTEDJ, "main");
	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	eext_object_event_callback_add(layout, EEXT_CALLBACK_BACK, win_back_cb, ad);

	return layout;
}

static void _destroy_layout(Evas_Object *layout)
{
	ret_if(!layout);

	evas_object_del(layout);
}

static int _sample_get_output_id (const char *output_name)
{
	eom_output_id *output_ids = NULL;
	eom_output_id output_id = 0;
	eom_output_type_e output_type = EOM_OUTPUT_TYPE_UNKNOWN;
	int id_cnt = 0;
	int i;

	/* get output_ids */
	output_ids = eom_get_eom_output_ids(&id_cnt);
	if (id_cnt == 0)
	{
		_D("no external outuputs supported\n");
		return 0;
	}

	/* find output ids interested */
	for (i = 0; i < id_cnt; i++)
	{
		eom_get_output_type(output_ids[i], &output_type);
		if (!strncmp(output_name, "HDMI", 4))
		{
			if (output_type == EOM_OUTPUT_TYPE_HDMIA || output_type == EOM_OUTPUT_TYPE_HDMIB)
			{
				output_id = output_ids[i];
				break;
			}
		}
	}

	if (output_ids)
		free (output_ids);

	return output_id;
}

static void eom_setup(appdata_s *ad)
{
	/* Get output ID */
	if (eom_init())
		_E("eom_init fail");
	else
		ad->output_id = _sample_get_output_id("HDMI");

	/* not support eom if output_id is 0 */
	if (ad->output_id == 0)
	{
		ad->support_external_display = EINA_FALSE;
		ad->connected = EINA_FALSE;
	}
	else
	{
		/* connnection check */
		eom_output_mode_e output_mode = EOM_OUTPUT_MODE_NONE;
		eom_get_output_mode(ad->output_id, &output_mode);
		if (output_mode != EOM_OUTPUT_MODE_NONE)
			ad->connected = EINA_TRUE;
		else
			ad->connected = EINA_FALSE;
		ad->support_external_display = EINA_TRUE;
		ad->mode = EOM_OUTPUT_MODE_MIRROR;
	}

	if (ad->support_external_display == EINA_TRUE)
	{
		/* attribute set for presentation mode */
		if (eom_set_output_attribute(ad->output_id, EOM_OUTPUT_ATTRIBUTE_NORMAL))
		{
			_D("eom_set_output_attribute EOM_OUTPUT_ATTRIBUTE_NORMAL fail");
			ad->attribute_set = EINA_FALSE;
		}
		else
		{
			_D("eom_set_output_attribute EOM_OUTPUT_ATTRIBUTE_NORMAL success");
			ad->attribute_set = EINA_TRUE;
		}

		/* set eom callback */
		eom_set_unset_cb(ad, EINA_TRUE);
	}
}

static void _create_eom_sample(appdata_s *ad)
{
	Elm_Object_Item *nf_it = NULL;
	Evas_Object *box1 = NULL;
	Evas_Object *box2 = NULL;

	ret_if(!ad);

	_D("Create EOM sample app");
	eom_setup(ad);

	/* Window */
	ad->win = _create_win(ad);
	ret_if(!ad->win);

	/* Conformant */
	ad->conform = _create_conform(ad);
	goto_if(!ad->conform, ERROR);

	/* Indicator BG */
	_set_indicator_bg(ad);

	/* Naviframe */
	ad->nf = _create_navi(ad);
	goto_if(!ad->nf, ERROR);

	/* Layout */
	ad->layout = _create_layout(ad);
	goto_if(!ad->layout, ERROR);

	/* two part in layout */
	box1 = view_create_eom_display(ad);
	goto_if(!box1, ERROR);
	elm_object_part_content_set(ad->layout, "main_display", box1);
	evas_object_data_set(ad->layout, "b_1", box1);

	box2 = view_create_eom_button(ad);
	goto_if(!box2, ERROR);
	elm_object_part_content_set(ad->layout, "button", box2);
	evas_object_data_set(ad->layout, "b_2", box2);

	evas_object_show(ad->layout);

	/* insert layout to naviframe */
	nf_it = elm_naviframe_item_push(ad->nf, "External Output Manager", NULL, NULL, ad->layout, NULL);
	goto_if(!nf_it, ERROR);

	/* Show window after base gui is set up */
	evas_object_show(ad->win);

	return;

ERROR:
	if (box2) {
		box2 = elm_object_part_content_unset(ad->layout, "button");
		evas_object_del(box2);
	}
	if (box1) {
		box1 = elm_object_part_content_unset(ad->layout, "main_display");
		evas_object_del(box1);
	}
	if (ad->layout) _destroy_layout(ad->layout);
	if (ad->nf) _destroy_navi(ad->nf);
	if (ad->conform) _destroy_conform(ad->conform);
	if (ad->win) _destroy_window(ad->win);
	if (ad->support_external_display)
	{
		eom_set_unset_cb(ad, EINA_FALSE);
		eom_deinit();
	}

	return;
}

static void _destroy_btn(Evas_Object *btn)
{
	ret_if(!btn);

	evas_object_del(btn);
}

static void _destroy_eom_sample(appdata_s *ad)
{
	Evas_Object *box1 = NULL;
	Evas_Object *box2 = NULL;

	ret_if(!ad);

	_D("Destroy stopwatch");

	if (ad->vd) {
		if (ad->vd->text1) evas_object_del(ad->vd->text1);
		if (ad->vd->text2) evas_object_del(ad->vd->text2);
		if (ad->vd->text3) evas_object_del(ad->vd->text3);
		if (ad->vd->bg) evas_object_del(ad->vd->bg);
		if (ad->vd->box_external) evas_object_del(ad->vd->box_external);

		free(ad->vd);
	}

	if (ad->btn_presentation) _destroy_btn(ad->btn_presentation);
	if (ad->btn_mirror) _destroy_btn(ad->btn_mirror);

	if (ad->nf) {
		ad->layout = elm_naviframe_item_pop(ad->nf);
		if (ad->layout) {
			box2 = evas_object_data_del(ad->layout, "b_2");
			if (box2) {
				box2 = elm_object_part_content_unset(ad->layout, "button");
				evas_object_del(box2);
			}

			box1 = evas_object_data_del(ad->layout, "b_1");
			if (box1) {
				box1 = elm_object_part_content_unset(ad->layout, "main_display");
				evas_object_del(box1);
			}

			if (ad->layout) _destroy_layout(ad->layout);
		}
	}

	if (ad->nf) _destroy_navi(ad->nf);
	if (ad->conform) _destroy_conform(ad->conform);
	if (ad->win) _destroy_window(ad->win);
	if (ad->support_external_display)
	{
		if (ad->external_win)
			_destroy_window(ad->external_win);
		eom_set_unset_cb(ad, EINA_FALSE);
		eom_deinit();
	}
}

static bool app_create(void *data)
{
	/* Hook to take necessary actions before main event loop starts
	   Initialize UI resources and application's data
	   If this function returns true, the main loop of application starts
	   If this function returns false, the application is terminated */
	appdata_s *ad = data;

	_create_eom_sample(ad);

	return true;
}

static void app_control(app_control_h app_control, void *data)
{
	/* Handle the launch request. */
}

static void app_pause(void *data)
{
	/* Take necessary actions when application becomes invisible. */
}

static void app_resume(void *data)
{
	/* Take necessary actions when application becomes visible. */
}

static void app_terminate(void *data)
{
	/* Release all resources. */
	appdata_s *ad = NULL;

	ret_if(!data);

	ad = data;

	_destroy_eom_sample(ad);
}

static void ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LANGUAGE_CHANGED*/
	char *locale = NULL;

	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);
	ret_if(!locale);

	elm_language_set(locale);
	free(locale);

	return;
}

static void ui_app_orient_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_DEVICE_ORIENTATION_CHANGED*/
	return;
}

static void ui_app_region_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_REGION_FORMAT_CHANGED*/
}

static void ui_app_low_battery(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_BATTERY*/
}

static void ui_app_low_memory(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_MEMORY*/
}

int main(int argc, char *argv[])
{
	appdata_s ad = {0,};
	int ret = 0;

	ui_app_lifecycle_callback_s event_callback = {0,};
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_BATTERY], APP_EVENT_LOW_BATTERY, ui_app_low_battery, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_MEMORY], APP_EVENT_LOW_MEMORY, ui_app_low_memory, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_DEVICE_ORIENTATION_CHANGED], APP_EVENT_DEVICE_ORIENTATION_CHANGED, ui_app_orient_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_REGION_FORMAT_CHANGED], APP_EVENT_REGION_FORMAT_CHANGED, ui_app_region_changed, &ad);
	ui_app_remove_event_handler(handlers[APP_EVENT_LOW_MEMORY]);

	ret = ui_app_main(argc, argv, &event_callback, &ad);
	if (ret != APP_ERROR_NONE) {
		_E("ui_app_main() is failed. err = %d", ret);
	}

	return ret;
}
/* End of file */
