/*
 * Samsung API
 * Copyright (c) 2009-2015 Samsung Electronics Co., Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "log.h"
#include "view.h"
#include "eom_sample.h"

static Evas_Object *_create_box(Evas_Object *parent)
{
	Evas_Object *box = NULL;

	retv_if(!parent, NULL);

	box = elm_box_add(parent);
	retv_if(!box, NULL);

	evas_object_size_hint_weight_set(box, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(box, EVAS_HINT_FILL, EVAS_HINT_FILL);

	return box;
}

static void _destroy_box(Evas_Object *box)
{
	ret_if(!box);

	evas_object_del(box);
}

static Evas_Object *_create_grid(Evas_Object *parent)
{
	Evas_Object *grid = NULL;

	retv_if(!parent, NULL);

	grid = elm_grid_add(parent);
	retv_if(!grid, NULL);

	evas_object_size_hint_weight_set(grid, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(grid, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_show(grid);

	return grid;
}

static void _destroy_grid(Evas_Object *grid)
{
	ret_if(!grid);

	evas_object_del(grid);
}

static Evas_Object *_create_btn(Evas_Object *parent, char *text)
{
	Evas_Object *btn = NULL;

	retv_if(!parent, NULL);

	btn = elm_button_add(parent);
	retv_if(!btn, NULL);

	elm_object_style_set(btn, "circle");
	elm_object_text_set(btn, text);
	evas_object_show(btn);

	return btn;
}

static void _destroy_btn(Evas_Object *btn)
{
	ret_if(!btn);

	evas_object_del(btn);
}

static Evas_Object *_create_bg(Evas_Object *parent, int type)
{
	Evas_Object *bg = NULL;

	retv_if(!parent, NULL);

	bg = elm_bg_add(parent);
	retv_if(!bg, NULL);

	evas_object_size_hint_weight_set(bg, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(bg, EVAS_HINT_FILL, EVAS_HINT_FILL);

	if (type == 1) {
		elm_bg_file_set(bg, BG_DIR, NULL);
		evas_object_size_hint_weight_set(bg, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
		evas_object_size_hint_align_set(bg, EVAS_HINT_FILL, EVAS_HINT_FILL);
		elm_win_resize_object_add(parent, bg);
	} else {
		elm_bg_color_set(bg, 0, 40, 40);
	}
	evas_object_show(bg);

	return bg;
}

static void _destroy_bg(Evas_Object *bg)
{
	ret_if(!bg);

	evas_object_del(bg);
}

extern Evas_Object *view_create_eom_display(appdata_s *ad)
{
	Evas_Object *box = NULL;
	Evas_Object *grid = NULL;
	Evas_Object *bg = NULL;
	viewdata_s *vd = NULL;

	retv_if(!ad, NULL);

	box = _create_box(ad->layout);
	retv_if(!box, NULL);
	grid = _create_grid(box);
	goto_if(!grid, ERROR);
	bg = _create_bg(grid, 1);
	goto_if(!bg, ERROR);
	elm_grid_pack(grid, bg, 0, 0, 100, 100);

	/* memory allocate */
	vd = calloc(1, sizeof(viewdata_s));
	goto_if(!vd, ERROR);
	vd->bg = bg;
	ad->vd = vd;

	/* set label */
	vd->text1 = elm_label_add(grid);
	goto_if(!vd->text1, ERROR);

	if (ad->support_external_display)
	{
		elm_object_text_set(vd->text1, "<font_size=50><color=#ffffff>Supporting EOM.</color></font_size>");
		evas_object_show(vd->text1);
		elm_grid_pack(grid, vd->text1, 2, 10, 100, 100);

		vd->text2 = elm_label_add(grid);
		goto_if(!vd->text2, ERROR);
		if (ad->connected)
		{
			elm_object_text_set(vd->text2, "<font_size=40><color=#ffffff>HDMI Connected.</color></font_size>");
			evas_object_show(vd->text2);
			elm_grid_pack(grid, vd->text2, 2, 30, 100, 100);
		}
		else
		{
			elm_object_text_set(vd->text2, "<font_size=40><color=#ffffff>Connect HDMI cable.</color></font_size>");
			evas_object_show(vd->text2);
			elm_grid_pack(grid, vd->text2, 2, 30, 100, 100);
		}

		vd->text3 = elm_label_add(grid);
		goto_if(!vd->text3, ERROR);
		if (ad->mode == EOM_OUTPUT_MODE_MIRROR)
		{
			elm_object_text_set(vd->text3, "<font_size=40><color=#ffffff>Mode : mirror</color></font_size>");
			evas_object_show(vd->text3);
			elm_grid_pack(grid, vd->text3, 2, 45, 100, 100);
		}
		else
		{
			elm_object_text_set(vd->text3, "<font_size=40><color=#ffffff>Mode : presentation</color></font_size>");
			evas_object_show(vd->text3);
			elm_grid_pack(grid, vd->text3, 2, 45, 100, 100);
		}
	}
	else
	{
		elm_object_text_set(vd->text1, "<font_size=50><color=#ffffff>Not supporting EOM.</color></font_size>");
		evas_object_show(vd->text1);
		elm_grid_pack(grid, vd->text1, 2, 10, 100, 100);
	}

	elm_box_pack_end(box, grid);

	return box;

ERROR:
	if (vd) free(vd);
	if (bg) _destroy_bg(bg);
	if (grid) _destroy_grid(grid);
	if (box) _destroy_box(box);

	return NULL;

}

extern Evas_Object *view_create_eom_button(appdata_s *ad)
{
	Evas_Object *box = NULL;
	Evas_Object *grid = NULL;

	retv_if(!ad, NULL);

	box = _create_box(ad->layout);
	retv_if(!box, NULL);
	grid = _create_grid(box);
	goto_if(!grid, ERROR);

	/* create button */
	ad->btn_mirror = _create_btn(grid, "MIRROR");
	goto_if(!ad->btn_mirror, ERROR);
	elm_grid_pack(grid, ad->btn_mirror, 0, 20, 50, 100);
	evas_object_smart_callback_add(ad->btn_mirror, "clicked", eom_button_clicked, ad);

	ad->btn_presentation = _create_btn(grid, "PRESENTATION");
	goto_if(!ad->btn_presentation, ERROR);
	elm_grid_pack(grid, ad->btn_presentation, 50, 20, 50, 100);
	evas_object_smart_callback_add(ad->btn_presentation, "clicked", eom_button_clicked, ad);

	elm_box_pack_end(box, grid);

	return box;

ERROR:
	if (ad->btn_mirror) _destroy_btn(ad->btn_mirror);
	if (grid) _destroy_grid(grid);
	if (box) _destroy_box(box);

	return NULL;
}

/* End of file */
