/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <app_manager.h>
#include "$(appName).h"
#include "data.h"

#define SYSTEM_EVENT_COUNT_MAX 25
#define CUSTOM_EVENT_COUNT_MAX 25

/*
 * The statements below are to flatten the system events definition inconsistency
 * between 2.4 and 3.0 platform binary version.
 */
#if !defined(SYSTEM_EVENT_INCOMMING_MSG) && defined(SYSTEM_EVENT_INCOMING_MSG)
#define SYSTEM_EVENT_INCOMMING_MSG SYSTEM_EVENT_INCOMING_MSG
#endif

static struct data_info {
	int custom_ev_count;
	custom_ev_info_s custom_ev[CUSTOM_EVENT_COUNT_MAX];
	system_ev_info_s system_ev[SYSTEM_EVENT_COUNT_MAX];
} s_info = {
	.custom_ev_count = 0,
	.custom_ev = {{0,},},
	.system_ev = {
		{ .type = ET_BATTERY_CHARGER_STATUS, .name = SYSTEM_EVENT_BATTERY_CHARGER_STATUS, .desc = "Battery charger status", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_BATTERY_LEVEL_STATUS, .name = SYSTEM_EVENT_BATTERY_LEVEL_STATUS, .desc = "Battery level status", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_USB_STATUS, .name = SYSTEM_EVENT_USB_STATUS, .desc = "USB status", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_EARJACK_STATUS, .name = SYSTEM_EVENT_EARJACK_STATUS, .desc = "Earjack status", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_DISPLAY_STATE, .name = SYSTEM_EVENT_DISPLAY_STATE, .desc = "Display state", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_BOOT_COMPLETED, .name = SYSTEM_EVENT_BOOT_COMPLETED, .desc = "Boot completed", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_SYSTEM_SHUTDOWN, .name = SYSTEM_EVENT_SYSTEM_SHUTDOWN, .desc = "System shutdown", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_LOW_MEMORY, .name = SYSTEM_EVENT_LOW_MEMORY, .desc = "Low memory", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_WIFI_STATE, .name = SYSTEM_EVENT_WIFI_STATE, .desc = "Wi-Fi state", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_BT_STATE, .name = SYSTEM_EVENT_BT_STATE, .desc = "Bluetooth state", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_LOCATION_ENABLE_STATE, .name = SYSTEM_EVENT_LOCATION_ENABLE_STATE, .desc = "Location state", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_GPS_ENABLE_STATE, .name = SYSTEM_EVENT_GPS_ENABLE_STATE, .desc = "GPS state", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_NPS_ENABLE_STATE, .name = SYSTEM_EVENT_NPS_ENABLE_STATE, .desc = "NPS state", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_INCOMMING_MSG, .name = SYSTEM_EVENT_INCOMMING_MSG, .desc = "Incoming message", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_TIME_CHANGED, .name = SYSTEM_EVENT_TIME_CHANGED, .desc = "Time changed", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_TIME_ZONE, .name = SYSTEM_EVENT_TIME_ZONE, .desc = "Time zone", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_HOUR_FORMAT, .name = SYSTEM_EVENT_HOUR_FORMAT, .desc = "Hour format", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_LANGUAGE_SET, .name = SYSTEM_EVENT_LANGUAGE_SET, .desc = "Language set", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_REGION_FORMAT, .name = SYSTEM_EVENT_REGION_FORMAT, .desc = "Region format", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_SILENT_MODE, .name = SYSTEM_EVENT_SILENT_MODE, .desc = "Silent mode", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_VIBRATION_STATE, .name = SYSTEM_EVENT_VIBRATION_STATE, .desc = "Vibration state", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_SCREEN_AUTOROTATE_STATE, .name = SYSTEM_EVENT_SCREEN_AUTOROTATE_STATE, .desc = "Screen auto-rotate state", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_MOBILE_DATA_STATE, .name = SYSTEM_EVENT_MOBILE_DATA_STATE, .desc = "Mobile data state", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_DATA_ROAMING_STATE, .name = SYSTEM_EVENT_DATA_ROAMING_STATE, .desc = "Data roaming state", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },
		{ .type = ET_FONT_SET, .name = SYSTEM_EVENT_FONT_SET, .desc = "Font set", .status_1 = NULL, .status_2 = NULL, .status_3 = NULL, },},
};

static void _release_system_events(void);
static void _release_custom_events(void);
static void _format_custom_event_name(const char *custom_name, char **ev_name);

/**
 * @brief Finalization function for data module.
 */
void data_finalize(void)
{
	_release_system_events();
	_release_custom_events();
}

/**
 * @brief Function assigns a function to the system event handler.
 * @param[in] ev_info The system information structure.
 * @param[in] callback The callback function to assign.
 * @param[in] user_data The user data passed to the assigned callback function.
 * @return The function returns 'true' is the callback function was successfully assigned,
 * otherwise 'false' is returned.
 */
bool data_add_system_event_handler(system_ev_info_s *ev_info, event_cb callback, void *user_data)
{
	if (!ev_info || !callback) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong argument provided.");
		return false;
	}

	int ret = event_add_event_handler(ev_info->name, callback, user_data, &ev_info->event_h);
	if (ret != EVENT_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function event_add_event_handler() failed with error = %d.", ret);
		return false;
	}

	return true;
}

/**
 * @brief Function gets the number of registered and handled system events.
 */
int data_get_system_events_count(void)
{
	return SYSTEM_EVENT_COUNT_MAX;
}

/**
 * @brief Function gets the system event's information structure based on its index.
 * @param[in] index The index of the system information structure to be returned.
 * @param[out] ev_info The requested system event's information structure.
 * @return The function returns 'true' is the system event's information structure
 * was successfully acquired, otherwise 'false' is returned.
 */
bool data_get_system_event_info(int index, system_ev_info_s **ev_info)
{
	*ev_info = NULL;

	if (index < 0 || index >= SYSTEM_EVENT_COUNT_MAX) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong argument provided.");
		return false;
	}

	*ev_info = &s_info.system_ev[index];

	return (*ev_info != NULL);
}

/**
 * @brief Function creates the custom event's information structure.
 * @param[in] event_name The name of the event to be created.
 * @param[out] ev_info The requested custom event's information structure.
 * @return The function returns 'true' is the system event's information structure
 * was successfully created, otherwise 'false' is returned.
 */
bool data_create_custom_event_info(const char *event_name, custom_ev_info_s **ev_info)
{
	*ev_info = NULL;

	if (!event_name) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong argument provided.");
		return false;
	}

	if (s_info.custom_ev_count == CUSTOM_EVENT_COUNT_MAX) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Maximum number of custom events is already registered.");
		return false;
	}

	s_info.custom_ev_count++;

	*ev_info = &s_info.custom_ev[s_info.custom_ev_count - 1];

	memset(*ev_info, 0, sizeof(custom_ev_info_s));

	_format_custom_event_name(event_name, &(*ev_info)->name);
	(*ev_info)->desc = strndup(event_name, strlen(event_name));

	return true;
}

/**
 * @brief Function assigns a function to the custom event handler.
 * @param[in] ev_info The custom information structure.
 * @param[in] callback The callback function to assign.
 * @param[in] user_data The user data passed to the assigned callback function.
 * @return The function returns 'true' is the callback function was successfully assigned,
 * otherwise 'false' is returned.
 */
bool data_add_custom_event_handler(custom_ev_info_s *ev_info, event_cb callback, void *user_data)
{
	if (!ev_info || !callback) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong argument provided.");
		return false;
	}

	int ret = event_add_event_handler(ev_info->name, callback, user_data, &ev_info->event_h);
	if (ret != EVENT_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function event_add_event_handler() failed with error = %d.", ret);
		return false;
	}

	return true;
}

/**
 * @brief Function publishes the custom event with given name.
 * @param[in] event_name The name of the event to be published.
 * @return The function returns 'true' is the custom event was successfully published,
 * otherwise 'false' is returned.
 */
bool data_publish_event(const char *event_name)
{
	static unsigned int custom_ev_counter = 0;
	bundle *bundle_ev;
	char status[256] = {0,};
	int ret;

	if (!event_name) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong argument provided.");
		return false;
	}

	bundle_ev = bundle_create();
	if (!bundle_ev) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function bundle_create() failed.");
		return false;
	}

	custom_ev_counter++;
	snprintf(status, sizeof(status), "status value %u", custom_ev_counter);

	ret = bundle_add_str(bundle_ev, CUSTOM_EVENT_KEY_STATUS, status);
	if (ret != EVENT_ERROR_NONE) {
		bundle_free(bundle_ev);
		dlog_print(DLOG_ERROR, LOG_TAG, "Function bundle_add_str() failed with error %d.", ret);
		return false;
	}

	ret = event_publish_app_event(event_name, bundle_ev);
	bundle_free(bundle_ev);

	if (ret != EVENT_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function event_publish_app_event() failed with error %d.", ret);
		return false;
	}

	return true;
}

/**
 * @brief Function checks if a custom event with given name is already registered.
 * @param[in] event_name The name of the event to be checked.
 * @param[out] exists The custom event's existence status.
 * @return The function returns 'true' if the verification passed successfully,
 * otherwise 'false' is returned.
 */
bool data_check_event_exists(const char *event_name, bool *exists)
{
	int i;
	char *formatted_name = NULL;

	*exists = false;

	if (!event_name) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong argument provided.");
		return false;
	}

	_format_custom_event_name(event_name, &formatted_name);

	for (i = 0; i < s_info.custom_ev_count; i++) {
		if (strncmp(formatted_name, s_info.custom_ev[i].name, strlen(formatted_name)) == 0 &&
			strlen(formatted_name) == strlen(s_info.custom_ev[i].name)) {
			*exists = true;
			free(formatted_name);
			return true;
		}
	}

	free(formatted_name);

	return true;
}

/**
 * @brief Internal function which gets the string value from bundle object for given key.
 * @param[in] bundle_obj The bundle object containing the string to be acquired.
 * @param[in] key The key name associated with the string value to be acquired.
 * @param[out] str The acquired string value for given key name.
 * @return The function returns 'true' if the string value is successfully acquired,
 * otherwise 'false' is returned.
 */
bool data_get_bundle_str(bundle *bundle_obj, const char *key, char **str)
{
	int ret;

	*str = NULL;

	ret = bundle_get_str(bundle_obj, key, str);
	if (ret != BUNDLE_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function bundle_get_str() failed with error = %d.", ret);
		return false;
	}

	return true;
}

/**
 * @brief Internal function which removes the given event's handler.
 * @param[in] event_handler The event's handler to be removed.
 * @return The function returns 'true' if the event's handler is successfully removed,
 * otherwise 'false' is returned.
 */
static bool _remove_event_handler(event_handler_h event_handler)
{
	int ret = event_remove_event_handler(event_handler);
	if (ret != EVENT_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function event_remove_event_handler() failed with error = %d.", ret);
		return false;
	}

	return true;
}

/**
 * @brief Internal function which frees all the memory allocated for system event's information structure.
 * @param[in] ev_info Pointer to the system event's information structure to be freed.
 * @return The function returns 'true' if the memory allocated for system event's information structure
 * is successfully freed, otherwise 'false' is returned.
 */
static void _free_system_event_info(system_ev_info_s *ev_info)
{
	if (!ev_info) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong argument provided.");
		return;
	}

	free(ev_info->status_1);
	free(ev_info->status_2);
	free(ev_info->status_3);
}

/**
 * @brief Internal function which frees all the memory allocated for custom event's information structure.
 * @param[in] ev_info Pointer to the custom event's information structure to be freed.
 * @return The function returns 'true' if the memory allocated for custom event's information structure
 * is successfully freed, otherwise 'false' is returned.
 */
static void _free_custom_event_info(custom_ev_info_s *ev_info)
{
	if (!ev_info) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong argument provided.");
		return;
	}

	free(ev_info->name);
	free(ev_info->desc);
	free(ev_info->status_1);
	free(ev_info->status_2);
	free(ev_info->status_3);
}

/**
 * @brief Internal function which frees the memory allocated for all the system event's
 * information structures and removes attached callback handlers.
 */
static void _release_system_events(void)
{
	int i;
	for (i = 0; i < SYSTEM_EVENT_COUNT_MAX; i++) {
		_remove_event_handler(s_info.system_ev[i].event_h);
		_free_system_event_info(&s_info.system_ev[i]);
	}
}

/**
 * @brief Internal function which frees the memory allocated for all the custom event's
 * information structures and removes attached callback handlers.
 */
static void _release_custom_events(void)
{
	int i;
	for (i = 0; i < s_info.custom_ev_count; i++) {
		_remove_event_handler(s_info.custom_ev[i].event_h);
		_free_custom_event_info(&s_info.custom_ev[i]);
	}

	s_info.custom_ev_count = 0;
}

/**
 * @brief Internal function which creates a fully qualified name for custom event,
 * based on given name.
 * The custom event's name must match the following format:
 * event.package-name.custom-name.
 * Example: event.org.example.sample_event.custom_event_name, where:
 * - package-name: org.example.sample_event
 * - custom-name: sample_event
 * @param[in] custom_name The event's custom name assigned by the user.
 * @param[out] ev_name The fully qualified custom event's name.
 */
static void _format_custom_event_name(const char *custom_name, char **ev_name)
{
	int i;
	int ev_name_len = strlen("event") + strlen(PACKAGE) + strlen(custom_name) + 3;
	*ev_name = (char *)calloc(ev_name_len, sizeof(char));
	snprintf(*ev_name, ev_name_len, "event.%s.%s", PACKAGE, custom_name);

	for (i = 0; i < ev_name_len; i++) {
		(*ev_name)[i] = tolower((*ev_name)[i]);

		if ((*ev_name)[i] == 32)
			(*ev_name)[i] = '_';
	}
}
