/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <Elementary.h>
#include <system_settings.h>
#include "$(appName).h"
#include "data.h"
#include "view.h"

static void _publish_event_cb(const char *event_name);
static bool _get_system_event_info_cb(int index, void **ev_info);
static bool _set_custom_event_info_cb(const char *event_name, void **ev_info);
static void _system_event_cb(const char *event_name, bundle *event_data, void *user_data);
static void _custom_event_cb(const char *event_name, bundle *event_data, void *user_data);
static void _add_system_event_handlers(void);
static void _assign_event_status(char **ev_status, const char *new_status);

/**
 * @brief Hook to take necessary actions before main event loop starts.
 * Initialize UI resources and application's data.
 * If this function returns true, the main loop of application starts.
 * If this function returns false, the application is terminated.
 */
static bool app_create(void *user_data)
{
	view_set_callbacks(_publish_event_cb, _get_system_event_info_cb, _set_custom_event_info_cb);

	if (!view_create(user_data))
		return false;

	_add_system_event_handlers();

	return true;
}

/**
 * @brief This callback function is called when another application
 * sends a launch request to the application.
 */
static void app_control(app_control_h app_control, void *user_data)
{
	/* Handle the launch request. */
}

/**
 * @brief This callback function is called each time
 * the application is completely obscured by another application
 * and becomes invisible to the user.
 */
static void app_pause(void *user_data)
{
	/* Take necessary actions when application becomes invisible. */
}

/**
 * @brief This callback function is called each time
 * the application becomes visible to the user.
 */
static void app_resume(void *user_data)
{
	/* Take necessary actions when application becomes visible. */
}

/**
 * @brief This callback function is called once after the main loop of the application exits.
 */
static void app_terminate(void *user_data)
{
	view_destroy();
	data_finalize();
}

/**
 * @brief This function will be called when the language is changed.
 */
static void ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/* APP_EVENT_LANGUAGE_CHANGED */
	char *locale = NULL;

	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);

	if (locale != NULL) {
		elm_language_set(locale);
		free(locale);
	}
	return;
}

/**
 * @brief Main function of the application.
 */
int main(int argc, char *argv[])
{
	int ret;

	ui_app_lifecycle_callback_s event_callback = {0, };
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	/*
	 * If you want to handle more events,
	 * please check the application life cycle guide.
	 */
	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, NULL);

	ret = ui_app_main(argc, argv, &event_callback, NULL);
	if (ret != APP_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "ui_app_main() failed. err = %d", ret);

	return ret;
}

/**
 * @brief Internal callback function invoked on custom event publish request.
 * This callback function is invoked when the user selects an event from
 * the list in the 'Custom events' tab.
 * @param[in] event_name The fully qualified name of the custom event to be published.
 */
static void _publish_event_cb(const char *event_name)
{
	dlog_print(DLOG_INFO, LOG_TAG, "Event publishing: '%s'.", event_name);

	data_publish_event(event_name);
}

/**
 * @brief Internal callback function responsible for supported system events acquisition
 * for display purpose.
 * It is invoked during the view creation.
 * @param[in] index The index of the system event to be acquired.
 * @param[out] ev_info The system event's information structure stored in data
 * module under the given index.
 * @return The function returns 'true' if the system event's information structure
 * was successfully acquired, otherwise 'false' is returned.
 */
static bool _get_system_event_info_cb(int index, void **ev_info)
{
	system_ev_info_s *ev_info_tmp = NULL;

	*ev_info = NULL;

	if (index >= data_get_system_events_count())
		return false;

	if (!data_get_system_event_info(index, &ev_info_tmp))
		return false;

	*ev_info = (system_ev_info_s *)ev_info_tmp;

	return true;
}

/**
 * @brief Internal callback function responsible for custom event registration.
 * This callback function is invoked when the user clicks the Plus
 * button in the UI.
 * @param[in] event_name The custom event's name to be registered.
 * @param[out] ev_info The custom event's information structure created
 * based on the event's name provided.
 * @return The function returns 'true' if the custom event was successfully registered,
 * otherwise 'false' is returned.
 */
static bool _set_custom_event_info_cb(const char *event_name, void **ev_info)
{
	custom_ev_info_s *ev = NULL;
	bool name_exists = false;

	if (!data_check_event_exists(event_name, &name_exists))
		return false;

	if (name_exists) {
		dlog_print(DLOG_WARN, LOG_TAG, "Custom event '%s' already registered.", event_name);
		return false;
	}

	if (!data_create_custom_event_info(event_name, &ev))
		return false;

	if (!data_add_custom_event_handler(ev, _custom_event_cb, (void *)(ev)))
		return false;

	*ev_info = (void *)ev;

	dlog_print(DLOG_INFO, LOG_TAG, "Custom event registered: '%s'.", ev->name);

	return true;
}

/**
 * @brief Internal callback function responsible for system event handling once it occurs.
 * This callback function is invoked when the system event occurs and it displays
 * basic information on the event.
 * @param[in] event_name The occurred system event's name.
 * @param[in] event_data The bundle object associated with the event.
 * @param[in] user_data The user data passed to the event's handler.
 * This is event's information carrier.
 */
static void _system_event_cb(const char *event_name, bundle *event_data, void *user_data)
{
	system_ev_info_s *ev_info = (system_ev_info_s *)user_data;
	char *status_1 = NULL;
	char *status_2 = NULL;
	char *status_3 = NULL;

	dlog_print(DLOG_INFO, LOG_TAG, "System event '%s' occurred.", event_name);

	switch (ev_info->type) {
	case ET_BATTERY_CHARGER_STATUS:
		data_get_bundle_str(event_data, EVENT_KEY_BATTERY_CHARGER_STATUS, &status_1);
		break;
	case ET_BATTERY_LEVEL_STATUS:
		data_get_bundle_str(event_data, EVENT_KEY_BATTERY_LEVEL_STATUS, &status_1);
		break;
	case ET_USB_STATUS:
		data_get_bundle_str(event_data, EVENT_KEY_USB_STATUS, &status_1);
		break;
	case ET_EARJACK_STATUS:
		data_get_bundle_str(event_data, EVENT_KEY_EARJACK_STATUS, &status_1);
		break;
	case ET_DISPLAY_STATE:
		data_get_bundle_str(event_data, EVENT_KEY_DISPLAY_STATE, &status_1);
		break;
	case ET_BOOT_COMPLETED:
		status_1 = "boot completed";
		break;
	case ET_SYSTEM_SHUTDOWN:
		status_1 = "system shutdown";
		break;
	case ET_LOW_MEMORY:
		data_get_bundle_str(event_data, EVENT_KEY_LOW_MEMORY, &status_1);
		break;
	case ET_WIFI_STATE:
		data_get_bundle_str(event_data, EVENT_KEY_WIFI_STATE, &status_1);
		break;
	case ET_BT_STATE:
		data_get_bundle_str(event_data, EVENT_KEY_BT_STATE, &status_1);
		data_get_bundle_str(event_data, EVENT_KEY_BT_LE_STATE, &status_2);
		data_get_bundle_str(event_data, EVENT_KEY_BT_TRANSFERING_STATE, &status_3);
		break;
	case ET_LOCATION_ENABLE_STATE:
		data_get_bundle_str(event_data, EVENT_KEY_LOCATION_ENABLE_STATE, &status_1);
		break;
	case ET_GPS_ENABLE_STATE:
		data_get_bundle_str(event_data, EVENT_KEY_GPS_ENABLE_STATE, &status_1);
		break;
	case ET_NPS_ENABLE_STATE:
		data_get_bundle_str(event_data, EVENT_KEY_NPS_ENABLE_STATE, &status_1);
		break;
	case ET_INCOMMING_MSG:
		data_get_bundle_str(event_data, EVENT_KEY_MSG_TYPE, &status_1);
		data_get_bundle_str(event_data, EVENT_KEY_MSG_ID, &status_2);
		break;
	case ET_TIME_CHANGED:
		status_1 = "time changed";
		break;
	case ET_TIME_ZONE:
		data_get_bundle_str(event_data, EVENT_KEY_TIME_ZONE, &status_1);
		break;
	case ET_HOUR_FORMAT:
		data_get_bundle_str(event_data, EVENT_KEY_HOUR_FORMAT, &status_1);
		break;
	case ET_LANGUAGE_SET:
		data_get_bundle_str(event_data, EVENT_KEY_LANGUAGE_SET, &status_1);
		break;
	case ET_REGION_FORMAT:
		data_get_bundle_str(event_data, EVENT_KEY_REGION_FORMAT, &status_1);
		break;
	case ET_SILENT_MODE:
		data_get_bundle_str(event_data, EVENT_KEY_SILENT_MODE, &status_1);
		break;
	case ET_VIBRATION_STATE:
		data_get_bundle_str(event_data, EVENT_KEY_VIBRATION_STATE, &status_1);
		break;
	case ET_SCREEN_AUTOROTATE_STATE:
		data_get_bundle_str(event_data, EVENT_KEY_SCREEN_AUTOROTATE_STATE, &status_1);
		break;
	case ET_MOBILE_DATA_STATE:
		data_get_bundle_str(event_data, EVENT_KEY_MOBILE_DATA_STATE, &status_1);
		break;
	case ET_DATA_ROAMING_STATE:
		data_get_bundle_str(event_data, EVENT_KEY_DATA_ROAMING_STATE, &status_1);
		break;
	case ET_FONT_SET:
		data_get_bundle_str(event_data, EVENT_KEY_FONT_SET, &status_1);
		break;
	default:
		return;
	}

	_assign_event_status(&ev_info->status_1, status_1);
	_assign_event_status(&ev_info->status_2, status_2);
	_assign_event_status(&ev_info->status_3, status_3);

	view_update_system_events();
}

/**
 * @brief Internal callback function responsible for custom event handling once it occurs.
 * This callback function is invoked when the custom event occurs and it displays
 * basic information on the event.
 * @param[in] event_name The occurred system event's name.
 * @param[in] event_data The bundle object associated with the event.
 * This is event's information carrier.
 * @param[in] user_data The user data passed to the event's handler.
 */
static void _custom_event_cb(const char *event_name, bundle *event_data, void *user_data)
{
	custom_ev_info_s *ev_info = (custom_ev_info_s *)user_data;
	char *status = NULL;

	if (!data_get_bundle_str(event_data, CUSTOM_EVENT_KEY_STATUS, &status))
		return;

	_assign_event_status(&ev_info->status_1, status);

	//TODO: add those funcs
	view_display_custom_event(ev_info->name, ev_info->status_1);
	view_update_custom_events();
}

/**
 * @brief Internal function responsible for callback functions attachment
 * to the system event handlers.
 */
static void _add_system_event_handlers(void)
{
	int i;
	system_ev_info_s *ev_info = NULL;

	for (i = 0; i < data_get_system_events_count(); i++)
		if (data_get_system_event_info(i, &ev_info))
			data_add_system_event_handler(ev_info, _system_event_cb, (void *)ev_info);
}

/**
 * @brief Internal helper function responsible for events status strings duplication.
 * @param[out] ev_status The duplicated 'new_status' string string is assigned here.
 * If this argument points to other valid string, then it is freed before assignment.
 * @param[in] new_status The event's status string to be duplicated.
 */
static void _assign_event_status(char **ev_status, const char *new_status)
{
	if (!new_status)
		return;

	if (*ev_status)
		free(*ev_status);

	*ev_status = strdup(new_status);

	return;
}
