/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <Eina.h>
#include <Elementary.h>
#include <efl_extension.h>
#include "$(appName).h"
#include "data.h"
#include "view.h"
#include "view_defines.h"

#define ITEM_DATA "item_data"
#define ITEM_TO_DEL "item_to_del"
#define DONE_BT_CODE 36

typedef enum { TOP_ICON_STATE_ROUND = 0, TOP_ICON_STATE_RECT } icon_state_t;

static struct view_info {
	Evas_Object *win;
	Evas_Object *conform;
	Evas_Object *scroller;
	Evas_Object *main_layout;
	Evas_Object *sys_ev_layout;
	Evas_Object *cstm_ev_layout;
	Evas_Object *sys_ev_list;
	Evas_Object *cstm_ev_list;
	Evas_Object *cstm_ev_name_entry;
	Evas_Object *cstm_ev_popup;
	Evas_Object *cstm_ev_published_event_popup;
	Elm_Genlist_Item_Class *sys_ev_itc;
	Elm_Genlist_Item_Class *cstm_ev_itc;
	icon_state_t sys_icon_state;
	icon_state_t cstm_icon_state;
	Eina_List *sys_ev_item_list;
	int max_visible_cstm_ev_items;
	event_do_publish_cb do_publish_func;
	event_get_system_info_cb get_system_info_func;
	event_set_custom_info_cb set_custom_info_func;
} s_info = {
	.win = NULL,
	.conform = NULL,
	.scroller = NULL,
	.main_layout = NULL,
	.sys_ev_layout = NULL,
	.cstm_ev_layout = NULL,
	.sys_ev_list = NULL,
	.cstm_ev_list = NULL,
	.cstm_ev_name_entry = NULL,
	.cstm_ev_popup = NULL,
	.cstm_ev_published_event_popup = NULL,
	.sys_ev_itc = NULL,
	.cstm_ev_itc = NULL,
	.sys_icon_state = TOP_ICON_STATE_ROUND,
	.cstm_icon_state = TOP_ICON_STATE_ROUND,
	.sys_ev_item_list = NULL,
	.max_visible_cstm_ev_items = -1,
	.do_publish_func = NULL,
	.get_system_info_func = NULL,
	.set_custom_info_func = NULL
};

typedef struct _sys_ev_item_data {
	int id;
	Elm_Object_Item *parent;
	Eina_Bool favorite;
	system_ev_info_s *system_ev_info;
} sys_ev_item_data;

typedef struct _cstm_ev_item_data {
	Elm_Object_Item *parent;
	custom_ev_info_s *custom_ev_info;
} cstm_ev_item_data;

static void _delete_win_request_cb(void *data, Evas_Object *obj, void *event_info);
static void _layout_back_cb(void *data, Evas_Object *obj, void *event_info);
static void _input_panel_state_changed_cb(void *data, Ecore_IMF_Context *ctx, int value);
static void _input_panel_key_down_cb(void *data, Evas *e, Evas_Object *obj, void *event_info);
static void _toolbar_tab_selected_cb(void *data, Evas_Object *obj, const char *emission, const char *source);
static void _top_icon_clicked_cb(void *data, Evas_Object *obj, const char *emission, const char *source);
static void _added_new_item_cb(void *data, Evas_Object *obj, const char *emission, const char *source);
static void _popup_btn_clicked(void *data, Evas_Object *obj, const char *emission, const char *source);
static void _published_event_popup_btn_clicked_cb(void *data, Evas_Object *obj, const char *emission, const char *source);
static void _signal_from_item_cb(void *data, Evas_Object *obj, const char *emission, const char *source);
static void _scroll_cb(void *data, Evas_Object *obj, void *event_info);
static void _edge_top_cb(void *data, Evas_Object *obj, void *event_info);
static void _edge_bottom_cb(void *data, Evas_Object *obj, void *event_info);
static Evas_Object *_get_sys_ev_list_content_cb(void *data, Evas_Object *obj, const char *part);
static Evas_Object *_get_cstm_ev_list_content_cb(void *data, Evas_Object *obj, const char *part);
static void _cstm_ev_entry_focused_cb(void *data, Evas_Object *obj, void *event_info);
static void _cstm_ev_entry_unfocused_cb(void *data, Evas_Object *obj, void *event_info);
static void _cstm_ev_list_height_get_cb(void *data, Evas *e, Evas_Object *obj, void *event_info);
static void _delete_cstm_event_cb(void *data, Evas_Object *obj, const char *emission, const char *source);
static void _del_sys_ev_list_item_cb(void *data, Evas_Object *obj);
static void _del_cstm_ev_list_item_cb(void *data, Evas_Object *obj);
static void _event_occured_cb(void *data, Evas_Object *obj, const char *emission, const char *source);
static int _compare_items_cb(const void *data1, const void *data2);
static void _get_app_resource(const char *edj_file_in, char *edj_path_out);
static bool _is_same_string(const char *s1, const char *s2);
static Eina_Bool _create_main_layout(void);
static Eina_Bool _create_sys_ev_view(void);
static Eina_Bool _create_cstm_ev_view(void);
static Eina_Bool _create_cstm_ev_popup(void);
static Eina_Bool _create_cstm_ev_published_event_popup(void);
static void _update_sys_ev_items(void);
static void _update_cstm_ev_items(void);
static Eina_Bool _is_last_item_visible(Evas_Object *genlist);
static void _add_remove_item_in_favorite_list(void *data, const char *action);
static void _cstm_ev_append_item(Evas_Object *entry);

/**
 * @brief Creates essential objects: window, conformant and view layouts.
 * @param[in] user_data The user data.
 * @return This function returns EINA_TRUE if the view was created successfully,
 * otherwise EINA_FALSE is returned.
 */
Eina_Bool view_create(void *user_data)
{
	/* Create the window */
	s_info.win = view_create_win(PACKAGE);
	if (s_info.win == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create a window.");
		return EINA_FALSE;
	}

	/* Create the conformant */
	s_info.conform = view_create_conformant(s_info.win);
	if (s_info.conform == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create a conformant");
		return EINA_FALSE;
	}

	s_info.scroller = view_create_scroller(s_info.conform, "default");
	if (!s_info.scroller)
		return EINA_FALSE;

	if (!_create_main_layout())
		return EINA_FALSE;

	if (!_create_sys_ev_view())
		return EINA_FALSE;

	if (!_create_cstm_ev_view())
		return EINA_FALSE;

	if (!_create_cstm_ev_popup())
		return EINA_FALSE;

	if (!_create_cstm_ev_published_event_popup())
		return EINA_FALSE;

	/* Show the window after main view is set up */
	evas_object_show(s_info.win);

	return EINA_TRUE;
}

/**
 * @brief Creates a basic window named package.
 * @param[in] pkg_name Name of the window
 */
Evas_Object *view_create_win(const char *pkg_name)
{
	Evas_Object *win = NULL;

	/*
	 * Window
	 * Create and initialize elm_win.
	 * elm_win is mandatory to manipulate the window.
	 */
	win = elm_win_util_standard_add(pkg_name, pkg_name);
	elm_win_conformant_set(win, EINA_TRUE);
	elm_win_indicator_mode_set(win, ELM_WIN_INDICATOR_SHOW);
	elm_win_indicator_opacity_set(win, ELM_WIN_INDICATOR_OPAQUE);
	elm_win_autodel_set(win, EINA_TRUE);

	evas_object_smart_callback_add(win, "delete,request", _delete_win_request_cb, NULL);

	return win;
}

/**
 * @brief Creates a conformant object for parent object.
 * @param[in] parent The parent object for conformant object.
 * @return The function returns conformant object if it was created successfully,
 * otherwise NULL is returned.
 */
Evas_Object *view_create_conformant(Evas_Object *parent)
{
	Evas_Object *conform = NULL;

	if (!parent) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong input arguments.");
		return NULL;
	}

	conform = elm_conformant_add(parent);
	if (!conform) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_conformant_add() failed.");
		return NULL;
	}

	elm_object_signal_emit(conform, "elm,state,indicator,overlap", "elm");

	evas_object_size_hint_weight_set(conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(parent, conform);
	evas_object_show(conform);

	return conform;
}

/**
 * @brief Creates a layout object and sets it to the parent object based on provided EDJE script.
 * @param[in] parent The parent object for layout object.
 * @param[in] edj_file_name The relative path to the layout EDJE script.
 * @param[in] edj_group The name of the group to be loaded from the EDJE script.
 * @param[in] target_part_name The name of the EDJE part where the layout object has to be set.
 * @return The function returns layout object set to the parent object if it was created successfully,
 * otherwise NULL is returned.
 */
Evas_Object *view_create_set_layout(Evas_Object *parent, const char *edj_file_name, const char *edj_group, const char *target_part_name)
{
	Evas_Object *layout = view_create_layout(parent, edj_file_name, edj_group);
	if (!layout)
		return NULL;

	if (!target_part_name)
		elm_object_content_set(parent, layout);
	else
		elm_object_part_content_set(parent, target_part_name, layout);

	return layout;
}

/**
 * @brief Creates a layout object for parent object based on provided EDJE script.
 * @param[in] parent The parent object for layout object.
 * @param[in] edj_file_name The relative path to the layout EDJE script.
 * @param[in] edj_group The name of the group to be loaded from the EDJE script.
 * @return The function returns layout object if it was created successfully,
 * otherwise NULL is returned.
 */
Evas_Object *view_create_layout(Evas_Object *parent, const char *edj_file_name, const char *edj_group)
{
	char edj_path[PATH_MAX] = {0, };
	Evas_Object *layout = NULL;

	if (!parent || !edj_file_name || !edj_group) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong input arguments.");
		return NULL;
	}

	_get_app_resource(edj_file_name, edj_path);

	layout = elm_layout_add(parent);
	if (!layout) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_layout_add() failed.");
		return NULL;
	}

	if (!elm_layout_file_set(layout, edj_path, edj_group)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_layout_file_set() failed.");
		evas_object_del(layout);
		return NULL;
	}

	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	return layout;
}

/**
 * @brief Creates a genlist object and sets it to the specified part of the parent object.
 * @param[in] parent The parent object for genlist object.
 * @param[in] target_part_name The name of the EDJE part where the genlist object has to be set.
 * @return The function returns genlist object if it was created successfully,
 * otherwise NULL is returned.
 */
Evas_Object *view_create_genlist(Evas_Object *parent, const char *target_part_name)
{
	Evas_Object *list = NULL;

	if (!parent || !target_part_name) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong input arguments.");
		return NULL;
	}

	list = elm_genlist_add(parent);
	if (!list) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_genlist_add() failed.");
		return NULL;
	}

	evas_object_size_hint_weight_set(list, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_object_part_content_set(parent, target_part_name, list);

	evas_object_show(list);

	return list;
}

/**
 * @brief Creates entry object for text input.
 * @param[in] parent The parent object for the entry.
 * @param[in] target_part_name The name of the part where the entry is to be embedded.
 * @return Function returns an entry object if successfully created,
 * otherwise NULL is returned.
 */
Evas_Object *view_create_entry(Evas_Object *parent, const char *target_part_name)
{
	Evas_Object *entry = NULL;

	if (!parent || !target_part_name) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong input arguments.");
		return NULL;
	}

	entry = elm_entry_add(parent);
	if (!entry) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_entry_add() failed.");
		return NULL;
	}

	elm_entry_single_line_set(entry, EINA_TRUE);
	elm_entry_line_wrap_set(entry, ELM_WRAP_NONE);
	elm_entry_scrollable_set(entry, EINA_TRUE);
	elm_object_part_content_set(parent, target_part_name, entry);

	return entry;
}

/**
 * @brief Creates a scroller object and sets it to the parent object.
 * @param[in] parent The parent object for scroller object.
 * @param[in] target_part_name The name of the EDJE part where the scroller object has to be set.
 * @return The function returns a scroller object if it was created successfully,
 * otherwise NULL is returned.
 */
Evas_Object *view_create_scroller(Evas_Object *parent, const char *target_part_name)
{
	Evas_Object *scroller = NULL;

	if (!parent || !target_part_name) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong input arguments.");
		return NULL;
	}

	scroller = elm_scroller_add(parent);
	if (!scroller) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_scroller_add() failed.");
		return NULL;
	}

	evas_object_size_hint_weight_set(scroller, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_scroller_bounce_set(scroller, EINA_FALSE, EINA_FALSE);
	elm_scroller_policy_set(scroller, ELM_SCROLLER_POLICY_OFF, ELM_SCROLLER_POLICY_OFF);
	elm_object_part_content_set(parent, target_part_name, scroller);
	evas_object_show(scroller);

	return scroller;
}

/**
 * @brief Creates a popup window with the layout set according to provided EDJE script.
 * @param[in] parent The parent object for popup window object.
 * @param[in] edj_file_name The relative path to the layout EDJE script.
 * @param[in] edj_group The name of the group to be loaded from the EDJE script.
 */
Evas_Object *view_create_popup(Evas_Object *parent, const char *edj_file_name, const char *edj_group)
{
	Evas_Object *popup = NULL;

	if (!parent || !edj_file_name || !edj_group) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong input arguments.");
		return NULL;
	}

	popup = elm_popup_add(parent);
	if (!popup) {
		dlog_print(DLOG_ERROR, LOG_TAG, "elm_popup_add() failed.");
		return NULL;
	}

	evas_object_size_hint_weight_set(popup, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	view_create_set_layout(popup, edj_file_name, edj_group, "default");

	return popup;
}

/**
 * @brief Function responsible for custom event display once it occurs.
 * This function is called from callback function attached to the custom event
 * once it occurs after being published. In order to notify the user about the
 * custom event occurrence, the toast message is displayed.
 * @param[in] event_name The name of the custom event which occurred.
 * @param[in] event_status The text status of the custom event which occurred.
 */
void view_display_custom_event(const char *event_name, const char *event_status)
{
	Evas_Object *cstm_ev_published_event_popup = elm_object_part_content_get(s_info.cstm_ev_published_event_popup, "default");
	char popup_message[512] = {0,};
	snprintf(popup_message, sizeof(popup_message), "Custom event occurred:<br><b>%s</b><br>with status:<br><b>%s</b>", event_name, event_status);
	elm_object_part_text_set(cstm_ev_published_event_popup, PART_CSTM_EV_PUBLISHED_POPUP_MESSAGE, popup_message);
	evas_object_show(s_info.cstm_ev_published_event_popup);
}

/**
 * @brief Destroys window and frees its resources.
 */
void view_destroy(void)
{
	if (s_info.win == NULL)
		return;

	evas_object_del(s_info.win);
}

/**
 * @brief Creates an item class for genlist object.
 * @param[in] on_content_get_cb The callback function's handler invoked when the genlist's item has to be redrawn.
 * @param[in] on_del_cb The callback function's handler invoked when the genlist's item has to be deleted.
 * @return The function returns item class if it was created successfully,
 * otherwise 'NULL' is returned.
 */
Elm_Genlist_Item_Class *view_create_genlist_item_class(Elm_Genlist_Item_Content_Get_Cb on_content_get_cb, Elm_Genlist_Item_Del_Cb on_del_cb)
{
	Elm_Genlist_Item_Class *itc = elm_genlist_item_class_new();

	if (!itc) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_genlist_item_class_new() failed.");
		return NULL;
	}

	itc->item_style = "full";
	itc->func.content_get = on_content_get_cb;
	itc->func.del = on_del_cb;

	return itc;
}

/**
 * @brief Adds callback functions invoked on user's actions performed via the UI.
 * @param[in] do_publish_func The callback function invoked on event publishing.
 * The event is published when the user selects an item from the genlist.
 * @param[in] get_system_info_func The callback function invoked on system events creation.
 * @param[in] set_custom_info_func The callback function invoked on custom event creation.
 * The custom event is created when the user clicks the 'Register event' button.
 */
void view_set_callbacks(event_do_publish_cb do_publish_func, event_get_system_info_cb get_system_info_func, event_set_custom_info_cb set_custom_info_func)
{
	s_info.do_publish_func = do_publish_func;
	s_info.get_system_info_func = get_system_info_func;
	s_info.set_custom_info_func = set_custom_info_func;
}

/**
 * @brief Function responsible for system events list update/refresh.
 */
void view_update_system_events(void)
{
	_update_sys_ev_items();
}

/**
 * @brief Function responsible for system events list update/refresh.
 */
void view_update_custom_events(void)
{
	_update_cstm_ev_items();
}

/**
 * @brief Internal callback function invoked when the main window needs to be destroyed.
 * @param[in] data The user data passed to the evas_object_smart_callback_add() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] event_info The structure containing the information on this event.
 */
static void _delete_win_request_cb(void *data, Evas_Object *obj, void *event_info)
{
	ui_app_exit();
}

/**
 * @brief Internal callback function invoked on HW Back button press.
 * @param[in] data The user data passed to the eext_object_event_callback_add() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] event_info The structure containing the information on this event.
 */
static void _layout_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	elm_win_lower(s_info.win);
}

/**
 * @brief Internal callback function to be invoked when the software input panel
 * state is changed. This function is responsible for the focus removal from the
 * custom entry name input widget when the input panel hides. Once the focus is removed,
 * the guide text is displayed in the event's name field.
 * @param[in] data Custom data passed to the callback attachment function.
 * @param[in] ctx The input method context.
 * @param[in] value The changed state value.
 */
static void _input_panel_state_changed_cb(void *data, Ecore_IMF_Context *ctx, int value)
{
	if (value == ECORE_IMF_INPUT_PANEL_STATE_HIDE)
		elm_object_focus_set(s_info.cstm_ev_name_entry, EINA_FALSE);
}

/**
 * @brief Internal callback function to be invoked when any key from the software keyboard is pressed.
 * Internally, this function hooks the "Done" button in order to close the input panel.
 * @param[in] data Custom data passed to the callback attachment function.
 * @param[in] e The evas surface of the object.
 * @param[in] obj Object to which the handler is attached.
 * @param[in] event_info The information on the event occurred.
 */
static void _input_panel_key_down_cb(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	Evas_Event_Key_Down *ev = (Evas_Event_Key_Down *)event_info;

	if (ev->keycode == DONE_BT_CODE)
		_cstm_ev_append_item(obj);
}

/**
 * @brief Internal callback function invoked on toolbar item click.
 * This callback function responds to the signal emitted from the EDJE script
 * and is responsible for layouts switching when toolbar's tab is changed.
 * @param[in] data The user data passed to the callback attachment function.
 * @param[in] obj The layout object emitting the signal.
 * @param[in] emission The name of the signal emitted.
 * @param[in] source The name of the part emitting the signal.
 */
static void _toolbar_tab_selected_cb(void *data, Evas_Object *obj, const char *emission, const char *source)
{
	Evas_Object *current_page = elm_object_part_content_unset(s_info.main_layout, PART_MAIN_CONTENT);
	if (current_page)
		evas_object_hide(current_page);

	if (_is_same_string(source, PART_MAIN_TOOLBAR_TAB_SYS_EV))
		elm_object_part_content_set(s_info.main_layout, PART_MAIN_CONTENT, s_info.sys_ev_layout);
	else if (_is_same_string(source, PART_MAIN_TOOLBAR_TAB_CSTM_EV))
		elm_object_part_content_set(s_info.main_layout, PART_MAIN_CONTENT, s_info.cstm_ev_layout);
}

/**
 * @brief Internal callback function invoked when the top icon was clicked.
 * @param[in] data The user data passed to the callback attachment function.
 * @param[in] obj The layout object emitting the signal.
 * @param[in] emission The name of the signal emitted.
 * @param[in] source The name of the part emitting the signal.
 */
static void _top_icon_clicked_cb(void *data, Evas_Object *obj, const char *emission, const char *source)
{
	Elm_Object_Item *first;

	if (obj == s_info.sys_ev_layout) {
		first = elm_genlist_first_item_get(s_info.sys_ev_list);
		s_info.sys_icon_state = TOP_ICON_STATE_ROUND;
	} else if (obj == s_info.cstm_ev_layout) {
		first = elm_genlist_first_item_get(s_info.cstm_ev_list);
		s_info.cstm_icon_state = TOP_ICON_STATE_ROUND;
	} else {
		return;
	}

	edje_object_signal_emit(elm_layout_edje_get(obj), SIGNAL_TOP_ICON_NORMAL, "");
	elm_genlist_item_bring_in(first, ELM_GENLIST_ITEM_SCROLLTO_TOP);
}

/**
 * @brief Internal callback function invoked when custom event add button was clicked.
 * @param[in] data The user data passed to the callback attachment function.
 * @param[in] obj The layout object emitting the signal.
 * @param[in] emission The name of the signal emitted.
 * @param[in] source The name of the part emitting the signal.
 */
static void _added_new_item_cb(void *data, Evas_Object *obj, const char *emission, const char *source)
{
	_cstm_ev_append_item(s_info.cstm_ev_name_entry);
}

/**
 * @brief Internal callback function invoked when any of the popup buttons is clicked.
 * For each button, a specific preassigned action is performed.
 * @param[in] data The user data passed to the callback attachment function.
 * @param[in] obj The layout object emitting the signal.
 * @param[in] emission The name of the signal emitted.
 * @param[in] source The name of the part emitting the signal.
 */
static void _popup_btn_clicked(void *data, Evas_Object *obj, const char *emission, const char *source)
{
	if (_is_same_string(emission, SIGNAL_POPUP_YES)) {
		elm_object_item_del((Elm_Object_Item *)evas_object_data_get(s_info.cstm_ev_popup, ITEM_TO_DEL));

		if (elm_genlist_items_count(s_info.cstm_ev_list) <= s_info.max_visible_cstm_ev_items) {
			s_info.cstm_icon_state = TOP_ICON_STATE_ROUND;
			edje_object_signal_emit(elm_layout_edje_get(s_info.cstm_ev_layout), SIGNAL_TOP_ICON_HIDDEN, "");
		}

		_update_cstm_ev_items();
	}

	evas_object_hide(s_info.cstm_ev_popup);
}

/**
 * @brief Internal callback function invoked when published event popup button was clicked.
 * @param[in] data The user data passed to the callback attachment function.
 * @param[in] obj The layout object emitting the signal.
 * @param[in] emission The name of the signal emitted.
 * @param[in] source The name of the part emitting the signal.
 */
static void _published_event_popup_btn_clicked_cb(void *data, Evas_Object *obj, const char *emission, const char *source)
{
	evas_object_hide(s_info.cstm_ev_published_event_popup);
}

/**
 * @brief Internal callback function invoked when system events' list item sends a signal.
 * @param[in] data The user data passed to the callback attachment function.
 * @param[in] obj The layout object emitting the signal.
 * @param[in] emission The name of the signal emitted.
 * @param[in] source The name of the part emitting the signal.
 */
static void _signal_from_item_cb(void *data, Evas_Object *obj, const char *emission, const char *source)
{
	if (_is_same_string(emission, SIGNAL_ITEM_FAVORITE_ADDED))
		_add_remove_item_in_favorite_list(evas_object_data_get(obj, ITEM_DATA), "add");
	else if (_is_same_string(emission, SIGNAL_ITEM_FAVORITE_REMOVED))
		_add_remove_item_in_favorite_list(evas_object_data_get(obj, ITEM_DATA), "remove");

	/* Update items background. */
	_update_sys_ev_items();
}

/**
 * @brief Internal callback function to be invoked when the genlist object
 * is being scrolled.
 * @param[in] data Custom data passed to the callback attachment function.
 * @param[in] obj Object to which the handler is attached.
 * @param[in] event_info The information on the event occurred.
 */
static void _scroll_cb(void *data, Evas_Object *obj, void *event_info)
{
	if (obj == s_info.sys_ev_list) {
		if (!_is_last_item_visible(s_info.sys_ev_list) && s_info.sys_icon_state == TOP_ICON_STATE_RECT) {
			s_info.sys_icon_state = TOP_ICON_STATE_ROUND;
			edje_object_signal_emit(elm_layout_edje_get(s_info.sys_ev_layout), SIGNAL_TOP_ICON_NORMAL, "");
		}
	} else if (obj == s_info.cstm_ev_list) {
		if (!_is_last_item_visible(s_info.cstm_ev_list) && s_info.cstm_icon_state == TOP_ICON_STATE_RECT) {
			s_info.cstm_icon_state = TOP_ICON_STATE_ROUND;
			edje_object_signal_emit(elm_layout_edje_get(s_info.cstm_ev_layout), SIGNAL_TOP_ICON_NORMAL, "");
		}
	}
}

/**
 * @brief Internal callback function to be invoked when the genlist object
 * reaches the top edge.
 * @param[in] data Custom data passed to the callback attachment function.
 * @param[in] obj Object to which the handler is attached.
 * @param[in] event_info The information on the event occurred.
 */
static void _edge_top_cb(void *data, Evas_Object *obj, void *event_info)
{
	if (s_info.cstm_icon_state == TOP_ICON_STATE_RECT) {
		s_info.cstm_icon_state = TOP_ICON_STATE_ROUND;
		edje_object_signal_emit(elm_layout_edje_get(s_info.cstm_ev_layout), SIGNAL_TOP_ICON_NORMAL, "");
	}
}

/**
 * @brief Internal callback function to be invoked when the genlist object
 * reaches the bottom edge.
 * @param[in] data Custom data passed to the callback attachment function.
 * @param[in] obj Object to which the handler is attached.
 * @param[in] event_info The information on the event occurred.
 */
static void _edge_bottom_cb(void *data, Evas_Object *obj, void *event_info)
{
	if (obj == s_info.sys_ev_list) {
		s_info.sys_icon_state = TOP_ICON_STATE_RECT;
		edje_object_signal_emit(elm_layout_edje_get(s_info.sys_ev_layout), SIGNAL_TOP_ICON_ACTIVE, "");
	} else if (obj == s_info.cstm_ev_list) {
		s_info.cstm_icon_state = TOP_ICON_STATE_RECT;
		edje_object_signal_emit(elm_layout_edje_get(s_info.cstm_ev_layout), SIGNAL_TOP_ICON_ACTIVE, "");
	}
}

/**
 * @brief Internal callback function invoked on system events list's item display.
 * This function is assigned to the func.content_get handler of genlist item's class structure.
 * @param[in] data The user data passed to the item add function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] part The name of the item's part being rendered.
 * @return The layout to be displayed is returned, if created successfully,
 * otherwise NULL is returned.
 */
static Evas_Object *_get_sys_ev_list_content_cb(void *data, Evas_Object *obj, const char *part)
{
	Evas_Object *layout = NULL;
	sys_ev_item_data *it = (sys_ev_item_data *)data;

	if (_is_same_string(part, "elm.swallow.content") && data) {
		layout = view_create_layout(obj, EDJ_SYS_EV, GRP_SYS_EV_ITEM);
		if (!layout)
			return NULL;

		evas_object_data_set(layout, ITEM_DATA, it);

		if (it->favorite == EINA_TRUE)
			edje_object_signal_emit(elm_layout_edje_get(layout), SIGNAL_ITEM_FAVORITE_ADDED, "");

		if (elm_genlist_item_index_get(it->parent) % 2 == 0)
			edje_object_signal_emit(elm_layout_edje_get(layout), SIGNAL_ITEM_BG_DARK, "");

		elm_layout_signal_callback_add(layout, SIGNAL_ITEM, PART_SYS_EV_ITEM_FAV_ICON, _signal_from_item_cb, NULL);
		elm_object_part_text_set(layout, PART_EV_ITEM_TITLE, it->system_ev_info->desc);

		if (it->system_ev_info->status_1)
			elm_object_part_text_set(layout, PART_EV_ITEM_STATUS, it->system_ev_info->status_1);
		if (it->system_ev_info->status_2)
			elm_object_part_text_set(layout, PART_EV_ITEM_STATUS, it->system_ev_info->status_2);
		if (it->system_ev_info->status_3)
			elm_object_part_text_set(layout, PART_EV_ITEM_STATUS, it->system_ev_info->status_3);
		if (!it->system_ev_info->status_1 && !it->system_ev_info->status_2 && !it->system_ev_info->status_3)
			elm_object_part_text_set(layout, PART_EV_ITEM_STATUS, "---");

		evas_object_show(layout);
	}

	return layout;
}

/**
 * @brief Internal callback function invoked on custom events list's item display.
 * This function is assigned to the func.content_get handler of genlist item's class structure.
 * @param[in] data The user data passed to the item add function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] part The name of the item's part being rendered.
 * @return The layout to be displayed is returned, if created successfully,
 * otherwise NULL is returned.
 */
static Evas_Object *_get_cstm_ev_list_content_cb(void *data, Evas_Object *obj, const char *part)
{
	Evas_Object *layout = NULL;
	cstm_ev_item_data *it = (cstm_ev_item_data *)data;

	if (_is_same_string(part, "elm.swallow.content") && data) {
		layout = view_create_layout(obj, EDJ_CSTM_EV, GRP_CSTM_EV_ITEM);
		if (!layout)
			return NULL;

		evas_object_data_set(layout, ITEM_DATA, it);

		if (elm_genlist_item_index_get(it->parent) % 2 == 0)
			edje_object_signal_emit(elm_layout_edje_get(layout), SIGNAL_ITEM_BG_DARK, "");

		elm_object_part_text_set(layout, PART_EV_ITEM_TITLE, it->custom_ev_info->name);

		if (it->custom_ev_info->status_1)
			elm_object_part_text_set(layout, PART_EV_ITEM_STATUS, it->custom_ev_info->status_1);
		else
			elm_object_part_text_set(layout, PART_EV_ITEM_STATUS, "---");

		elm_layout_signal_callback_add(layout, SIGNAL_ITEM_BG_CLICKED, PART_EV_ITEM_BG, _event_occured_cb, layout);

		elm_layout_signal_callback_add(layout, SIGNAL_TRASH_BTN_CLICKED, PART_CSTM_EV_ITEM_TRASH_BTN, _delete_cstm_event_cb, it->parent);
		evas_object_show(layout);
	}

	return layout;
}

/**
 * @brief Internal callback function to be invoked when the event name entry gains focus.
 * This function hides the guide text displayed in an custom event's entry widget.
 * @param[in] data Custom data passed to the callback attachment function.
 * @param[in] obj Object to which the handler is attached.
 * @param[in] event_info The information on the event occurred.
 */
static void _cstm_ev_entry_focused_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *layout = (Evas_Object *)data;
	elm_layout_signal_emit(layout, SIGNAL_TEXT_ENTRY_HINT_HIDE, "");
}

/**
 * @brief Internal callback function to be invoked when the event name entry looses focus.
 * This function shows the guide text in an custom event's entry widget.
 * @param[in] data Custom data passed to the callback attachment function.
 * @param[in] obj Object to which the handler is attached.
 * @param[in] event_info The information on the event occurred.
 */
static void _cstm_ev_entry_unfocused_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *layout = (Evas_Object *)data;
	const char *input = elm_entry_entry_get(obj);

	if (!input || strlen(input) == 0)
		elm_layout_signal_emit(layout, SIGNAL_TEXT_ENTRY_HINT_SHOW, "");
}

/**
 * @brief Internal callback function to be invoked when the custom events list is resized.
 * This is used to compute the maximum number of items visible in a viewport.
 * @param[in] data Custom data passed to the callback attachment function.
 * @param[in] obj Object to which the handler is attached.
 * @param[in] event_info The information on the event occurred.
 */
static void _cstm_ev_list_height_get_cb(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	int h = -1;

	evas_object_geometry_get(obj, NULL, NULL, NULL, &h);

	s_info.max_visible_cstm_ev_items = h / ITEM_HEIGHT;

	evas_object_event_callback_del(s_info.cstm_ev_list, EVAS_CALLBACK_RESIZE, _cstm_ev_list_height_get_cb);
}

/**
 * @brief Internal callback function invoked when trashcan button was clicked.
 * It shows the popup where decision, to delete item or not, is made.
 * @param[in] data The user data passed to the callback attachment function.
 * @param[in] obj The layout object emitting the signal.
 * @param[in] emission The name of the signal emitted.
 * @param[in] source The name of the part emitting the signal.
 */
static void _delete_cstm_event_cb(void *data, Evas_Object *obj, const char *emission, const char *source)
{
	evas_object_data_set(s_info.cstm_ev_popup, ITEM_TO_DEL, data);
	evas_object_show(s_info.cstm_ev_popup);
}

/**
 * @brief Internal callback function invoked on system event list's item deletion.
 * This function is assigned to the func.del handler of genlist item's class structure.
 * The item data assigned to list's item is deleted here.
 * @param[in] data The user data passed to the item add function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] part The name of the item's part being rendered.
 */
static void _del_sys_ev_list_item_cb(void *data, Evas_Object *obj)
{
	if (data)
		free(data);
}

/**
 * @brief Internal callback function invoked on custom event list's item deletion.
 * This function is assigned to the func.del handler of genlist item's class structure.
 * The item data assigned to list's item is deleted here.
 * @param[in] data The user data passed to the item add function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] part The name of the item's part being rendered.
 */
static void _del_cstm_ev_list_item_cb(void *data, Evas_Object *obj)
{
	cstm_ev_item_data *it = (cstm_ev_item_data *)data;

	if (!data)
		return;

	if (it->custom_ev_info->name)
		free(it->custom_ev_info->name);

	free(data);
}

/**
 * @brief Internal callback function invoked when custom list's item was clicked.
 * It uses do_publish_func() callback function to publish an event.
 * @param[in] data The user data passed to the callback attachment function.
 * @param[in] obj The layout object emitting the signal.
 * @param[in] emission The name of the signal emitted.
 * @param[in] source The name of the part emitting the signal.
 */
static void _event_occured_cb(void *data, Evas_Object *obj, const char *emission, const char *source)
{
	Evas_Object *item_layout = data;
	cstm_ev_item_data *it = evas_object_data_get(item_layout, ITEM_DATA);
	custom_ev_info_s *ev_info = it->custom_ev_info;

	if (!ev_info || !ev_info->name) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_custom_event_item_select_cb() failed. Invalid input argument");
		return;
	}

	if (s_info.do_publish_func)
		s_info.do_publish_func(ev_info->name);

	/* After highlighting an item in edje file, the background should be set to the previous state. */
	if (elm_genlist_item_index_get(it->parent) % 2 == 0)
		edje_object_signal_emit(elm_layout_edje_get(item_layout), SIGNAL_ITEM_BG_DARK, "");
	else
		edje_object_signal_emit(elm_layout_edje_get(item_layout), SIGNAL_ITEM_BG_LIGHT, "");
}

/**
 * @brief Function used to compare two genlist items for sorted insert purpose.
 * It compares data1 and data2. If data1 is 'less' than data2, -1 must be returned,
 * if it is 'greater', 1 must be returned, and if they are equal, 0 must be returned.
 * The first condition passes if an item belongs to the favourite list.
 * The favourite items are stored alphabetically. Non-favourite items are stored by their index.
 * @param[in] data1 The sorted item.
 * @param[in] data2 The sorted item.
 * @return This function returns -1, 0, 1 depending on the comparison result.
 */
static int _compare_items_cb(const void *data1, const void *data2)
{
	sys_ev_item_data *it_data1 = (sys_ev_item_data *)data1;
	sys_ev_item_data *it_data2 = (sys_ev_item_data *)data2;

	if (it_data1->favorite < it_data2->favorite) {
		return 1;
	} else if (it_data1->favorite == it_data2->favorite) {
		if (it_data1->favorite == EINA_TRUE) {
			return strcmp(it_data1->system_ev_info->desc, it_data2->system_ev_info->desc);
		} else {
			if (it_data1->id > it_data2->id)
				return 1;
			else if (it_data1->id == it_data2->id)
				return 0;
			else
				return -1;
		}
	}

	return -1;
}

/**
 * @brief Internal function which creates fully qualified path to the provided resource file.
 * @param[in] edj_file_in The file name.
 * @param[out] edj_path_out The fully qualified path to the edj_file_in file.
 */
static void _get_app_resource(const char *edj_file_in, char *edj_path_out)
{
	char *res_path = app_get_resource_path();
	if (res_path) {
		snprintf(edj_path_out, PATH_MAX, "%s%s", res_path, edj_file_in);
		free(res_path);
	}
}

/**
 * @brief Internal function which checks whether two strings are the same.
 * @param[in] s1 The first string.
 * @param[in] s2 The second string.
 * @return This function returns 'true' is both given strings are the same,
 * otherwise 'false' is returned.
 */
static bool _is_same_string(const char *s1, const char *s2)
{
	if (!s1 || !s2)
		return false;

	return (strncmp(s1, s2, strlen(s1)) == 0 && strlen(s1) == strlen(s2));
}

/**
 * @brief Internal function which creates the main panel (Space for collage and photos browser).
 * @return The function returns EINA_TRUE if the main layout was created successfully,
 * otherwise EINA_FALSE is returned.
 */
static Eina_Bool _create_main_layout(void)
{
	s_info.main_layout = view_create_set_layout(s_info.scroller, EDJ_MAIN, GRP_MAIN, "default");
	if (!s_info.main_layout)
		return EINA_FALSE;

	eext_object_event_callback_add(s_info.main_layout, EEXT_CALLBACK_BACK, _layout_back_cb, NULL);
	elm_layout_signal_callback_add(s_info.main_layout, SIGNAL_TOOLBAR_TAB_SELECTED, PART_MAIN_TOOLBAR_TABS, _toolbar_tab_selected_cb, NULL);

	edje_object_signal_emit(elm_layout_edje_get(s_info.main_layout), SIGNAL_TOOLBAR_TAB_SYS_EV_SELECT, "");

	return EINA_TRUE;
}

/**
 * @brief Internal function which creates the layout for 'System Events' tab.
 * @return The function returns EINA_TRUE if the layout was created successfully,
 * otherwise EINA_FALSE is returned.
 */
static Eina_Bool _create_sys_ev_view(void)
{
	int i = 0;
	Eina_List *it = NULL;
	sys_ev_item_data *data = NULL;
	system_ev_info_s *ev_info = NULL;

	s_info.sys_ev_layout = view_create_layout(s_info.main_layout, EDJ_SYS_EV, GRP_SYS_EV);
	if (!s_info.sys_ev_layout)
		return EINA_FALSE;

	s_info.sys_ev_list = view_create_genlist(s_info.sys_ev_layout, PART_SYS_EV_CONTENT);
	if (!s_info.sys_ev_list)
		return EINA_FALSE;

	evas_object_smart_callback_add(s_info.sys_ev_list, "edge,bottom", _edge_bottom_cb, NULL);
	evas_object_smart_callback_add(s_info.sys_ev_list, "scroll", _scroll_cb, NULL);

	s_info.sys_ev_itc = view_create_genlist_item_class(_get_sys_ev_list_content_cb, _del_sys_ev_list_item_cb);

	if (!s_info.sys_ev_itc)
		return EINA_FALSE;

	elm_layout_signal_callback_add(s_info.sys_ev_layout, "mouse,clicked,1", PART_TOP_ICON_BG, _top_icon_clicked_cb, NULL);

	if (!s_info.get_system_info_func)
		return EINA_FALSE;

	while (s_info.get_system_info_func(i, (void*)&ev_info)) {
		sys_ev_item_data *it = (sys_ev_item_data *)calloc(1, sizeof(sys_ev_item_data));
		if (!it) {
			dlog_print(DLOG_ERROR, LOG_TAG, "Function calloc() failed. Memory allocation error.");
			continue;
		}

		it->favorite = EINA_FALSE;
		it->id = i;
		it->system_ev_info = ev_info;
		s_info.sys_ev_item_list = eina_list_sorted_insert(s_info.sys_ev_item_list, _compare_items_cb, it);
		i++;
	}

	EINA_LIST_FOREACH(s_info.sys_ev_item_list, it, data)
		if (data)
			data->parent = elm_genlist_item_append(s_info.sys_ev_list, s_info.sys_ev_itc, data, NULL, ELM_GENLIST_ITEM_NONE, NULL, NULL);

	return EINA_TRUE;
}

/**
 * @brief Internal function which creates the layout for 'Custom Events' tab.
 * @return The function returns 'EINA_TRUE' if the layout was created successfully,
 * otherwise 'EINA_FALSE' is returned.
 */
static Eina_Bool _create_cstm_ev_view(void)
{
	Ecore_IMF_Context *ctx;

	s_info.cstm_ev_layout = view_create_layout(s_info.main_layout, EDJ_CSTM_EV, GRP_CSTM_EV);
	if (!s_info.sys_ev_layout)
		return EINA_FALSE;

	edje_object_signal_emit(elm_layout_edje_get(s_info.cstm_ev_layout), SIGNAL_TOP_ICON_HIDDEN, "");
	elm_layout_signal_callback_add(s_info.cstm_ev_layout, "mouse,clicked,1", PART_TOP_ICON_BG, _top_icon_clicked_cb, NULL);
	elm_layout_signal_callback_add(s_info.cstm_ev_layout, SIGNAL_ADD_EVENT_BTN_CLICKED, PART_CSTM_EV_ADD_EVENT_BTN, _added_new_item_cb, NULL);

	s_info.cstm_ev_name_entry = view_create_entry(s_info.cstm_ev_layout, PART_CSTM_EV_TEXT_ENTRY);
	if (!s_info.cstm_ev_name_entry)
		return EINA_FALSE;

	elm_entry_text_style_user_push(s_info.cstm_ev_name_entry, STYLE_CSTM_EV_TEXT_STYLE);
	evas_object_event_callback_add(s_info.cstm_ev_name_entry, EVAS_CALLBACK_KEY_DOWN, _input_panel_key_down_cb, NULL);
	evas_object_smart_callback_add(s_info.cstm_ev_name_entry, "focused", _cstm_ev_entry_focused_cb, (void *)s_info.cstm_ev_layout);
	evas_object_smart_callback_add(s_info.cstm_ev_name_entry, "unfocused", _cstm_ev_entry_unfocused_cb, (void *)s_info.cstm_ev_layout);

	s_info.cstm_ev_list = view_create_genlist(s_info.cstm_ev_layout, PART_CSTM_EV_CONTENT);
	if (!s_info.cstm_ev_list)
		return EINA_FALSE;

	elm_scroller_movement_block_set(s_info.cstm_ev_list, ELM_SCROLLER_MOVEMENT_BLOCK_HORIZONTAL);
	evas_object_event_callback_add(s_info.cstm_ev_list, EVAS_CALLBACK_RESIZE, _cstm_ev_list_height_get_cb, NULL);
	evas_object_smart_callback_add(s_info.cstm_ev_list, "edge,bottom", _edge_bottom_cb, NULL);
	evas_object_smart_callback_add(s_info.cstm_ev_list, "scroll", _scroll_cb, NULL);
	evas_object_smart_callback_add(s_info.cstm_ev_list, "edge,top", _edge_top_cb, NULL);

	s_info.cstm_ev_itc = view_create_genlist_item_class(_get_cstm_ev_list_content_cb, _del_cstm_ev_list_item_cb);
	if (!s_info.cstm_ev_itc)
		return EINA_FALSE;

	ctx = (Ecore_IMF_Context *)elm_entry_imf_context_get(s_info.cstm_ev_name_entry);
	if (ctx)
		ecore_imf_context_input_panel_event_callback_add(ctx, ECORE_IMF_INPUT_PANEL_STATE_EVENT, _input_panel_state_changed_cb, NULL);
	else
		dlog_print(DLOG_WARN, LOG_TAG, "Function elm_entry_imf_context_get() failed.");

	return EINA_TRUE;
}

/**
 * @brief Internal function which creates a popup window for custom event deletion.
 * @return The function returns EINA_TRUE if the popup was created successfully,
 * otherwise EINA_FALSE is returned.
 */
static Eina_Bool _create_cstm_ev_popup(void)
{
	Evas_Object *popup_layout;
	s_info.cstm_ev_popup = view_create_popup(s_info.cstm_ev_layout, EDJ_CSTM_EV, GRP_CSTM_EV_POPUP);

	if (!s_info.cstm_ev_popup)
		return EINA_FALSE;

	popup_layout = elm_object_part_content_get(s_info.cstm_ev_popup, "default");
	elm_layout_signal_callback_add(popup_layout, SIGNAL_POPUP, PART_CSTM_EV_POPUP_BTN, _popup_btn_clicked, NULL);

	return EINA_TRUE;
}

/**
 * @brief Internal function which creates a popup window for custom event's status display.
 * @return The function returns EINA_TRUE if the popup was created successfully,
 * otherwise EINA_FALSE is returned.
 */
static Eina_Bool _create_cstm_ev_published_event_popup(void)
{
	Evas_Object *popup_layout;

	s_info.cstm_ev_published_event_popup = view_create_popup(s_info.cstm_ev_layout, EDJ_CSTM_EV, GRP_CSTM_EV_PUBLISHED_POPUP);
	if (!s_info.cstm_ev_published_event_popup)
		return EINA_FALSE;

	popup_layout = elm_object_part_content_get(s_info.cstm_ev_published_event_popup, "default");
	elm_layout_signal_callback_add(popup_layout, SIGNAL_PUBLISHED_POPUP_OK, PART_CSTM_EV_PUBLISHED_POPUP_BTN_OK, _published_event_popup_btn_clicked_cb, NULL);

	return EINA_TRUE;
}

/**
 * @brief Internal function used to update realized items of system events list.
 * It works on a list of realized items, changes its status and background.
 * It doesn't use elm_genlist_realized_items_update() function, as it is not suitable here.
 */
static void _update_sys_ev_items(void)
{
	Eina_List *realized_items = elm_genlist_realized_items_get(s_info.sys_ev_list);
	Eina_List *it = NULL;
	Elm_Object_Item *list_item = NULL;
	Evas_Object *it_layout;
	sys_ev_item_data *it_data;

	EINA_LIST_FOREACH(realized_items, it, list_item) {
		if (!list_item)
			continue;

		it_layout = elm_object_item_part_content_get(list_item, "elm.swallow.content");

		if (elm_genlist_item_index_get(list_item) % 2 == 0)
			edje_object_signal_emit(elm_layout_edje_get(it_layout), SIGNAL_ITEM_BG_DARK, "");
		else
			edje_object_signal_emit(elm_layout_edje_get(it_layout), SIGNAL_ITEM_BG_LIGHT, "");

		it_data = evas_object_data_get(it_layout, ITEM_DATA);
		if (it_data->system_ev_info->status_1)
			elm_object_part_text_set(it_layout, PART_EV_ITEM_STATUS, it_data->system_ev_info->status_1);
		if (it_data->system_ev_info->status_2)
			elm_object_part_text_set(it_layout, PART_EV_ITEM_STATUS, it_data->system_ev_info->status_2);
		if (it_data->system_ev_info->status_3)
			elm_object_part_text_set(it_layout, PART_EV_ITEM_STATUS, it_data->system_ev_info->status_3);
	}

	eina_list_free(realized_items);
}

/**
 * @brief Internal function used to update realized items of custom events list.
 * It works on a list of realized items, changes its status and background.
 * It doesn't use elm_genlist_realized_items_update() function, as it is not suitable here.
 */
static void _update_cstm_ev_items(void)
{
	Eina_List *realized_items = elm_genlist_realized_items_get(s_info.cstm_ev_list);
	Eina_List *it = NULL;
	Elm_Object_Item *list_item = NULL;
	Evas_Object *it_layout;
	cstm_ev_item_data *it_data;

	EINA_LIST_FOREACH(realized_items, it, list_item) {
		if (!list_item)
			continue;

		it_layout = elm_object_item_part_content_get(list_item, "elm.swallow.content");

		if (elm_genlist_item_index_get(list_item) % 2 == 0)
			edje_object_signal_emit(elm_layout_edje_get(it_layout), SIGNAL_ITEM_BG_DARK, "");
		else
			edje_object_signal_emit(elm_layout_edje_get(it_layout), SIGNAL_ITEM_BG_LIGHT, "");

		it_data = evas_object_data_get(it_layout, ITEM_DATA);
		if (it_data->custom_ev_info->status_1)
			elm_object_part_text_set(it_layout, PART_EV_ITEM_STATUS, it_data->custom_ev_info->status_1);
	}

	eina_list_free(realized_items);
}

/**
 * @brief Internal function that checks if the last item of genlist is visible (realized).
 * Function takes a list of realized items and its last item, then performs its comparison
 * with the last item of the genlist.
 * @param[in] genlist The genlist object from which last item is taken.
 * @return This function returns EINA_TRUE if the last item is visible,
 * otherwise EINA_FALSE is returned.
 */
static Eina_Bool _is_last_item_visible(Evas_Object *genlist)
{
	Eina_Bool ret;
	Elm_Object_Item *last_item;
	Eina_List *realized_items = elm_genlist_realized_items_get(genlist);

	if (!realized_items) {
		dlog_print(DLOG_ERROR, LOG_TAG, "elm_genlist_realized_items_get() failed.");
		return EINA_FALSE;
	}

	last_item = eina_list_last_data_get(realized_items);
	ret = (Eina_Bool)(last_item == elm_genlist_last_item_get(genlist));

	eina_list_free(realized_items);

	return ret;
}

/**
 * @brief Internal function that adds or removes system event item from favorite list.
 * @param[in] data Item data assigned to this function.
 * @param[in] action String that determines the action to perform: add or remove an item.
 * Set 'add' or 'remove' to add to remove an item respectively.
 */
static void _add_remove_item_in_favorite_list(void *data, const char *action)
{
	sys_ev_item_data *it_data = (sys_ev_item_data *)data;
	sys_ev_item_data *new_data;
	sys_ev_item_data *item_prev_data;
	Eina_List *item;

	/* Remove item from eina list. */
	s_info.sys_ev_item_list = eina_list_remove(s_info.sys_ev_item_list, it_data);

	/* Change the flag in item data based on the argument provided. */
	if (_is_same_string(action, "add"))
		it_data->favorite = EINA_TRUE;
	else if (_is_same_string(action, "remove"))
		it_data->favorite = EINA_FALSE;

	/* Copy old item data to the new one. */
	new_data = (sys_ev_item_data *)calloc(1, sizeof(sys_ev_item_data));
	if (!new_data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_add_remove_item_in_favorite_list() failed. Memory allocation error.");
		return;
	}

	memcpy(new_data, it_data, sizeof(sys_ev_item_data));
	new_data->parent = NULL;

	/* Delete old item (the old item data will be deleted after calling item deletion). */
	elm_object_item_del(it_data->parent);
	/* Insert new item to eina list, then map the position in genlist */
	s_info.sys_ev_item_list = eina_list_sorted_insert(s_info.sys_ev_item_list, _compare_items_cb, new_data);
	item = eina_list_search_sorted_list(s_info.sys_ev_item_list, _compare_items_cb, new_data);
	item =  eina_list_prev(item);

	if (item == NULL) {
		new_data->parent = elm_genlist_item_insert_before(s_info.sys_ev_list, s_info.sys_ev_itc, new_data, NULL,  elm_genlist_first_item_get(s_info.sys_ev_list), ELM_GENLIST_ITEM_NONE, NULL, NULL);
	} else {
		item_prev_data = eina_list_data_get(item);
		new_data->parent = elm_genlist_item_insert_after(s_info.sys_ev_list, s_info.sys_ev_itc, new_data, NULL, item_prev_data->parent, ELM_GENLIST_ITEM_NONE, NULL, NULL);
	}
}

/**
 * @brief Internal function that appends new item to custom events list.
 * @param[in] entry Entry object which holds custom event name.
 */
static void _cstm_ev_append_item(Evas_Object *entry)
{
	cstm_ev_item_data *it;
	custom_ev_info_s *ev_info = NULL;
	const char *text = elm_entry_entry_get(entry);

	if (!text || strlen(text) == 0) {
		dlog_print(DLOG_WARN, LOG_TAG, "Function _cstm_ev_append_item() failed. Event's name not assigned.");
		return;
	}

	if (!s_info.set_custom_info_func) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function _cstm_ev_append_item() failed. Not assigned set_custom_info_func callback.");
		return;
	}

	if (!s_info.set_custom_info_func(text, (void *)&ev_info))
		return;

	it = (cstm_ev_item_data *)calloc(1, sizeof(cstm_ev_item_data));
	if (!it) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_cstm_ev_append_item() failed. Memory allocation error.");
		return;
	}

	it->custom_ev_info = ev_info;
	it->parent = elm_genlist_item_append(s_info.cstm_ev_list, s_info.cstm_ev_itc, it, NULL, ELM_GENLIST_ITEM_NONE, NULL, NULL);

	elm_entry_input_panel_hide(entry);
	elm_object_focus_set(entry, EINA_FALSE);
	elm_entry_entry_set(entry, "");
	elm_layout_signal_emit(s_info.cstm_ev_layout, SIGNAL_TEXT_ENTRY_HINT_SHOW, "");

	if (elm_genlist_items_count(s_info.cstm_ev_list) > s_info.max_visible_cstm_ev_items &&
		s_info.cstm_icon_state != TOP_ICON_STATE_RECT)
		edje_object_signal_emit(elm_layout_edje_get(s_info.cstm_ev_layout), SIGNAL_TOP_ICON_ENTER, "");
}
