/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <linux/limits.h>
#include "$(appName).h"
#include "data.h"

#define BIND_FILE "bind.txt"
#define BIND_FILE_CONTENT "Hello world !"

static struct data_info {
	bool autosave_on;
} s_info = {
	.autosave_on = false
};

static const char *_get_path(const char *file_name);

/**
 * @brief Initialization function for data module.
 * Within this function, the binding file is created. If the file already exists
 * then nothing happens.
 */
void data_initialize(void)
{
	FILE *fp = NULL;
	const char *bind_file_path = NULL;

	bind_file_path = _get_path(BIND_FILE);

	/* Check if file does not exist. */
	if (access(bind_file_path, F_OK)) {
	    fp = fopen(bind_file_path, "w+");
	    if (!fp) {
		    dlog_print(DLOG_ERROR, LOG_TAG, "fopen() failed.");
		    return;
	    }

	    fwrite(BIND_FILE_CONTENT, sizeof(char), strlen(BIND_FILE_CONTENT) + 1, fp);
	    fclose(fp);
	}
}

/**
 * @brief Function sets the auto-save feature state.
 * @param[in] state The state of the auto-save feature to be set.
 */
void data_set_autosave_state(bool state)
{
	s_info.autosave_on = state;
}

/**
 * @brief Function returns the fully qualified path to the binding file created
 * during the data module initialization.
 * @return The fully qualified path to the binding file or NULL if the function fails.
 */
const char *data_get_bind_file_path(void)
{
	return _get_path(BIND_FILE);
}

/**
 * @brief Function returns the current state of the auto-save feature.
 * @return The state of the auto-save feature.
 */
bool data_get_autosave_state(void)
{
	return s_info.autosave_on;
}

/**
 * @brief Function returns the fully qualified path to the given file stored within
 * the application's directory.
 * @param[in] file_name The name of the file for which the fully qualified path is to be created.
 * @return The fully qualified path to the given file or NULL if the function fails.
 */
static const char *_get_path(const char *file_name)
{
	static char path_out[PATH_MAX] = {0,};
	char *res_path;

	res_path = app_get_data_path();
	if (res_path) {
		snprintf(path_out, PATH_MAX, "%s%s", res_path, file_name);
		free(res_path);
		return &path_out[0];
	}

	return NULL;
}
