/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <dlog.h>
#include <efl_extension.h>
#include "view.h"
#include "view_defines.h"
#include "$(appName).h"

#define WIN_WIDTH_HD 720
#define WIN_HEIGHT_HD 1280
#define INPUT_PANEL_LAYOUT_COUNT 4

static struct view_info {
	Evas_Object *win;
	Evas_Object *main_layout;
	Evas_Object *entry;
	int kbd_current_layout;
	autosave_toggle_cb autosave_func;
	get_bind_file_path_cb bind_file_func;
} s_info = {
	.win = NULL,
	.main_layout = NULL,
	.entry = NULL,
	.kbd_current_layout = 0,
	.autosave_func = NULL,
	.bind_file_func = NULL,
};

static Elm_Input_Panel_Layout available_layouts[] = { ELM_INPUT_PANEL_LAYOUT_NORMAL,
													  ELM_INPUT_PANEL_LAYOUT_NUMBERONLY,
													  ELM_INPUT_PANEL_LAYOUT_PHONENUMBER,
													  ELM_INPUT_PANEL_LAYOUT_MONTH };

static void _delete_win_request_cb(void *data, Evas_Object *obj, void *event_info);
static void _layout_back_cb(void *data, Evas_Object *obj, void *event_info);
static void _toggle_autosave_cb(void *data, Evas_Object *obj, void *event_info);
static void _bind_file_cb(void *data, Evas_Object *obj, void *event_info);
static void _keypad_layout_changed_cb(void *data, Evas_Object *obj, void *event_info);
static void _get_app_resource(const char *edj_file_in, char *edj_path_out);
static Eina_Bool _create_main_layout(void);
static Eina_Bool _create_entry_field(void);
static Eina_Bool _create_toolbar(void);

/**
 * @brief Creates essential objects: window, conformant and layout.
 */
Eina_Bool view_create(void)
{
	/* Create the window */
	s_info.win = view_create_win(PACKAGE);
	if (s_info.win == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create a window.");
		return EINA_FALSE;
	}

	if (!_create_main_layout())
		return EINA_FALSE;

	if (!_create_entry_field())
		return EINA_FALSE;

	if (!_create_toolbar())
		return EINA_FALSE;

	/* Show the window after main view is set up */
	evas_object_show(s_info.win);

	return EINA_TRUE;
}

/**
 * @brief Sets the view callback functions.
 * @param[in] autosave_func The callback function invoked when the user
 * toggles the state of the 'Autosave' checkbox.
 * @param[in] bind_file_func The callback function invoked when the user
 * clicks the 'Bind file' button.
 */
void view_set_callbacks(autosave_toggle_cb autosave_func, get_bind_file_path_cb bind_file_func)
{
	s_info.autosave_func = autosave_func;
	s_info.bind_file_func = bind_file_func;
}

/**
 * @brief Creates a basic window named package
 * @param[in] pkg_name Name of the window
 * @return The function returns window object if it was created successfully,
 * otherwise 'NULL' is returned.
 */
Evas_Object *view_create_win(const char *pkg_name)
{
	Evas_Object *win = NULL;

	/*
	 * Window
	 * Create and initialize elm_win.
	 * elm_win is mandatory to manipulate the window.
	 */
	win = elm_win_util_standard_add(pkg_name, pkg_name);
	elm_win_conformant_set(win, EINA_TRUE);
	elm_win_indicator_mode_set(win, ELM_WIN_INDICATOR_SHOW);
	elm_win_indicator_opacity_set(win, ELM_WIN_INDICATOR_OPAQUE);
	elm_win_autodel_set(win, EINA_TRUE);

	evas_object_smart_callback_add(win, "delete,request", _delete_win_request_cb, NULL);

	return win;
}

/**
 * @brief Creates a layout object for parent object based on provided EDJE script.
 * @param[in] parent The parent object for layout object.
 * @param[in] edj_file_name The relative path to the layout EDJE script.
 * @param[in] edj_group The name of the group to be loaded from the EDJE script.
 * @param[in] target_part_name The EDJE part's name where the layout is to be set.
 * @return The function returns layout object if it was created successfully,
 * otherwise 'NULL' is returned.
 */
Evas_Object *view_create_layout(Evas_Object *parent, const char *edj_file_name, const char *edj_group, const char *target_part_name)
{
	char edj_path[PATH_MAX] = {0, };
	Evas_Object *layout = NULL;

	if (!parent || !edj_file_name || !edj_group) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong input arguments.");
		return NULL;
	}

	_get_app_resource(edj_file_name, edj_path);

	layout = elm_layout_add(parent);
	if (!layout) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_layout_add() failed.");
		return NULL;
	}

	if (!elm_layout_file_set(layout, edj_path, edj_group)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_layout_file_set() failed.");
		evas_object_del(layout);
		return NULL;
	}

	if (target_part_name) {
		elm_object_part_content_set(parent, target_part_name, layout);
	} else {
		evas_object_resize(layout, WIN_WIDTH_HD, WIN_HEIGHT_HD);
		evas_object_show(layout);
	}

	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	return layout;
}

/**
 * @brief Internal function which creates the button.
 * @param[in] parent The parent object for button object.
 * @param[in] target_part_name The EDJE part's name where the button is to be set.
 * @param[in] on_click_cb The callback function to be attached to the 'clicked' event.
 * @param[in] data The user data to be passed to the on_click_cb callback function.
 * @return The function returns 'EINA_TRUE' if the button was created successfully,
 * otherwise 'EINA_FALSE' is returned.
 */
Evas_Object *view_create_button(Evas_Object *parent, const char *target_part_name, Evas_Smart_Cb on_click_cb, void *data)
{
	Evas_Object *button = elm_button_add(parent);
	if (!button) {
		dlog_print(DLOG_ERROR, LOG_TAG, "elm_button_add() failed.");
		return NULL;
	}

	evas_object_smart_callback_add(button, "clicked", on_click_cb, data);

	evas_object_size_hint_weight_set(button, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_object_part_content_set(parent, target_part_name, button);
	evas_object_show(button);

	return button;
}

/**
 * @brief Creates a checkbox object and sets it to the parent object.
 * @param[in] parent The parent object for checkbox object.
 * @param[in] target_part_name The name of the EDJE part where the checkbox object has to be set.
 * @param[in] on_change_cb The callback function to be attached to the 'changed' event.
 * @return The function returns checkbox object if it was created successfully,
 * otherwise 'NULL' is returned.
 */
Evas_Object *view_create_checkbox(Evas_Object *parent, const char *target_part_name, Evas_Smart_Cb on_change_cb)
{
	Evas_Object *checkbox = elm_check_add(parent);
	if (!checkbox) {
		dlog_print(DLOG_ERROR, LOG_TAG, "elm_check_add() failed.");
		return NULL;
	}

	evas_object_smart_callback_add(checkbox, "changed", on_change_cb, NULL);

	elm_check_state_set(checkbox, EINA_FALSE);
	evas_object_size_hint_weight_set(checkbox, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_object_part_content_set(parent, target_part_name, checkbox);

	evas_object_show(checkbox);

	return checkbox;
}

/**
 * @brief Destroys window and frees its resources.
 */
void view_destroy(void)
{
	if (s_info.win == NULL)
		return;

	evas_object_del(s_info.win);
}

/**
 * @brief Internal callback function invoked when the main window needs to be destroyed.
 * @param[in] data The user data passed to the evas_object_smart_callback_add() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] event_info The structure containing the information on this event.
 */
static void _delete_win_request_cb(void *data, Evas_Object *obj, void *event_info)
{
	ui_app_exit();
}

/**
 * @brief Internal callback function invoked on HW Back button press.
 * @param[in] data The user data passed to the eext_object_event_callback_add() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] event_info The structure containing the information on this event.
 */
static void _layout_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	/* Let window go to hide state. */
	elm_win_lower(s_info.win);
}

/**
 * @brief Internal callback function invoked when the state of the 'Autosave' check is changed.
 * @param[in] data The user data passed to the evas_object_smart_callback_add() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] event_info The structure containing the information on this event.
 */
static void _toggle_autosave_cb(void *data, Evas_Object *obj, void *event_info)
{
	bool autosave_state;

	if (!s_info.autosave_func) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_toggle_autosave_cb() failed. Callback function not attached.");
		return;
	}

	autosave_state = (bool)elm_check_state_get(obj);

	dlog_print(DLOG_INFO, LOG_TAG, "Autosave toggle: %d", (int)autosave_state);
	s_info.autosave_func(autosave_state);
	elm_entry_autosave_set(s_info.entry, autosave_state);
}

/**
 * @brief Internal callback function invoked when the 'Bind file' button is pressed.
 * @param[in] data The user data passed to the evas_object_smart_callback_add() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] event_info The structure containing the information on this event.
 */
static void _bind_file_cb(void *data, Evas_Object *obj, void *event_info)
{
	const char *path = NULL;
	bool autosave_state = false;

	if (!s_info.bind_file_func) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_bind_file_cb() failed. Callback function not attached.");
		return;
	}

	dlog_print(DLOG_INFO, LOG_TAG, "Binding file");
	s_info.bind_file_func(&path, &autosave_state);

	if (!path) {
		dlog_print(DLOG_ERROR, LOG_TAG, "bind callback function failed. Path is empty.");
		return;
	}

	if (!elm_entry_file_set(s_info.entry, path, ELM_TEXT_FORMAT_PLAIN_UTF8))
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to bind file %s to the entry widget.", path);

	elm_entry_autosave_set(s_info.entry, EINA_TRUE);
	elm_entry_autosave_set(s_info.entry, autosave_state);
}

/**
 * @brief Internal callback function invoked when the 'Next kbd' / 'Prev kbd' button is pressed.
 * Internally, the layout of the system keyboard is changed.
 * @param[in] data The user data passed to the evas_object_smart_callback_add() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] event_info The structure containing the information on this event.
 */
static void _keypad_layout_changed_cb(void *data, Evas_Object *obj, void *event_info)
{
	if (data == LAYOUT_PREV)
		--s_info.kbd_current_layout;
	else
		++s_info.kbd_current_layout;

	if (s_info.kbd_current_layout < 0)
		s_info.kbd_current_layout = INPUT_PANEL_LAYOUT_COUNT - 1;

	if (s_info.kbd_current_layout >= INPUT_PANEL_LAYOUT_COUNT)
		s_info.kbd_current_layout = 0;

	elm_entry_input_panel_hide(s_info.entry);
	elm_entry_input_panel_layout_set(s_info.entry, available_layouts[s_info.kbd_current_layout]);
	elm_entry_input_panel_show(s_info.entry);
	elm_object_focus_allow_set(s_info.entry, EINA_TRUE);
	elm_object_focus_set(s_info.entry, EINA_TRUE);
}

/**
 * @brief Internal function which creates fully qualified path to the provided resource file.
 * @param[in] edj_file_in The file name.
 * @param[out] edj_path_out The fully qualified path to the edj_file_in file.
 */
static void _get_app_resource(const char *edj_file_in, char *edj_path_out)
{
	char *res_path = app_get_resource_path();
	if (res_path) {
		snprintf(edj_path_out, PATH_MAX, "%s%s", res_path, edj_file_in);
		free(res_path);
	}
}

/**
 * @brief Internal function which creates the main layout.
 * @return The function returns 'EINA_TRUE' if the main layout was created successfully,
 * otherwise 'EINA_FALSE' is returned.
 */
static Eina_Bool _create_main_layout(void)
{
	s_info.main_layout = view_create_layout(s_info.win, EDJ_FILE, GRP_MAIN, NULL);
	if (!s_info.main_layout)
		return EINA_FALSE;

	eext_object_event_callback_add(s_info.main_layout, EEXT_CALLBACK_BACK, _layout_back_cb, NULL);

	return EINA_TRUE;
}

/**
 * @brief Internal function which creates the entry field.
 * @return The function returns 'EINA_TRUE' if the entry field was created successfully,
 * otherwise 'EINA_FALSE' is returned.
 */
static Eina_Bool _create_entry_field(void)
{
	s_info.entry = elm_entry_add(s_info.win);
	if (!s_info.entry) {
		dlog_print(DLOG_ERROR, LOG_TAG, "elm_entry_add() failed.");
		return EINA_FALSE;
	}

	evas_object_size_hint_weight_set(s_info.entry, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(s_info.entry, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_entry_scrollable_set(s_info.entry, EINA_TRUE);
	elm_object_part_content_set(s_info.main_layout, PART_PANEL_BOTTOM, s_info.entry);

	return EINA_TRUE;
}

/**
 * @brief Internal function which creates the upper toolbar.
 * @return The function returns 'EINA_TRUE' if the toolbar was created successfully,
 * otherwise 'EINA_FALSE' is returned.
 */
static Eina_Bool _create_toolbar(void)
{
	Evas_Object *button;
	Evas_Object *layout;
	Evas_Object *check;

	layout = view_create_layout(s_info.main_layout, EDJ_FILE, GRP_TOOLBAR, PART_PANEL_TOP);
	if (!layout)
		return EINA_FALSE;

	button = view_create_button(layout, PART_BUTTON_PREV, _keypad_layout_changed_cb, (void *)LAYOUT_PREV);
	if (!button)
		return EINA_FALSE;

	button = view_create_button(layout, PART_BUTTON_NEXT, _keypad_layout_changed_cb, (void *)LAYOUT_NEXT);
	if (!button)
		return EINA_FALSE;

	check = view_create_checkbox(layout, PART_AUTOSAVE, _toggle_autosave_cb);
	if (!check)
		return EINA_FALSE;

	button = view_create_button(layout, PART_BUTTON_BIND_FILE, _bind_file_cb, NULL);
	if (!button)
		return EINA_FALSE;

	evas_object_show(layout);

	return EINA_TRUE;
}
