/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <Elementary.h>
#include <system_settings.h>
#include <badge.h>
#include "$(appName).h"
#include "notification/notifications.h"
#include "notification/ongoing-notifications.h"
#include "notification/notification-defines.h"
#include "notification/notification-common.h"
#include "view.h"

static struct app_info {
	int service_request_count;
} s_info = {
	.service_request_count = 0,
};

static bool badge_update_cb(unsigned int *count, bool step);
static void notification_launch_cb(notification_h notification, const char *argument, const char *pkgname,
									const char *uri_text, const char *mime_text, const char *operation,
									const char **array_result, const char **array_key);
static bool _create_notification_data(notification_data **notify_data);
static void _free_strings_array(char **strings, int length);
static bool _report_app_control_error(int err_code, const char *err_message);

/**
 * @brief Hook to take necessary actions before main event loop starts.
 * Initialize UI resources and application's data.
 * If this function returns true, the main loop of application starts.
 * If this function returns false, the application is terminated.
 */
static bool app_create(void *user_data)
{
	int ret = badge_add(PACKAGE);
	if (ret != BADGE_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "badge_add() failed. Err = %d", ret);

	notification_set_callbacks(notification_launch_cb, badge_update_cb);
	ongoing_notification_set_callbacks(notification_launch_cb, badge_update_cb);

	return (bool)view_create(NULL);
}

/**
 * @brief This callback function is called when another application
 * sends a launch request to the application.
 */
static void app_control(app_control_h app_control, void *user_data)
{
	char *extra_data = NULL;
	char *uri = NULL;
	char *operation = NULL;
	char *mime = NULL;
	char **result_service_data = NULL;
	char **result_service_data_to = NULL;
	int length = 0;
	char extra_data_to[TEXT_SHORT_LEN] = { '\0'};
	char extra_data_text[TEXT_SHORT_LEN] = {'\0'};
	notification_data *notify_data = NULL;
	int ret;

	ret = app_control_get_extra_data(app_control, MESSAGE_POST, &extra_data);
	if (_report_app_control_error(ret, "app_control_get_extra_data() failed")) {
		if (!_create_notification_data(&notify_data)) {
			view_rise_window();
			return;
		}

		_report_app_control_error(app_control_get_uri(app_control, &uri), "app_control_get_uri() failed");
		_report_app_control_error(app_control_get_operation(app_control, &operation), "app_control_get_operation() failed");
		_report_app_control_error(app_control_get_mime(app_control, &mime), "app_control_get_mime() failed");

		ret = app_control_get_extra_data_array(app_control, SERVICE_DATA_TEXT, &result_service_data, &length);
		if (_report_app_control_error(ret, "app_control_get_extra_data_array() failed")) {
			snprintf(extra_data_text, TEXT_SHORT_LEN, "# 1%s (%s | %s)<br>", key_value, SERVICE_DATA_TEXT, result_service_data[0]);
			_free_strings_array(result_service_data, length);
		}

		ret = app_control_get_extra_data_array(app_control, SERVICE_DATA_TO, &result_service_data_to, &length);
		if (_report_app_control_error(ret, "app_control_get_extra_data_array() failed")) {
			snprintf(extra_data_to, TEXT_SHORT_LEN, "# 2%s (%s | %s)", key_value, SERVICE_DATA_TEXT, result_service_data_to[0]);
			_free_strings_array(result_service_data_to, length);
		}

		snprintf(notify_data->name, TEXT_SHORT_LEN, "%s", app_control_request);
		snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s %i<br>%s %s<br>%s %s<br>%s %s<br><br>%s<br># 0%s (%s | %s)<br>%s%s",
				req_id, ++s_info.service_request_count, operation_id, operation, uri_data_text, uri,
				mime_type, mime, extra_data, key_value, extra_data_tizen_org, extra_data, extra_data_text, extra_data_to);

		free(operation);
		free(uri);
		free(mime);
		free(extra_data);

		view_display_notification(notify_data);
		free(notify_data);
	}

	view_rise_window();
}

/**
 * @brief This callback function is called each time
 * the application is completely obscured by another application
 * and becomes invisible to the user.
 */
static void app_pause(void *user_data)
{
	/* Take necessary actions when application becomes invisible. */
}

/**
 * @brief This callback function is called each time
 * the application becomes visible to the user.
 */
static void app_resume(void *user_data)
{
	/* Take necessary actions when application becomes visible. */
}

/**
 * @brief This callback function is called once after the main loop of the application exits.
 */
static void app_terminate(void *user_data)
{
	view_destroy();
}

/**
 * @brief This function will be called when the language is changed.
 */
static void ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/* APP_EVENT_LANGUAGE_CHANGED */
	char *locale = NULL;

	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);

	if (locale != NULL) {
		elm_language_set(locale);
		free(locale);
	}

	return;
}

/**
 * @brief Main function of the application.
 */
int main(int argc, char *argv[])
{
	int ret;

	ui_app_lifecycle_callback_s event_callback = {0, };
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	/*
	 * If you want to handle more events,
	 * please check the application lifecycle guide.
	 */
	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, NULL);

	ret = ui_app_main(argc, argv, &event_callback, NULL);
	if (ret != APP_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "ui_app_main() failed. err = %d", ret);

	return ret;
}

/**
 * @brief Internal function which modifies the badge counter value.
 * @param[out] count The badge counter value.
 * @return This function returns 'true' if the badge counter was successfully updated,
 * otherwise 'false' is returned.
 */
static bool badge_update_cb(unsigned int *count, bool step)
{
	int ret = badge_get_count(PACKAGE, count);
	if (ret != BADGE_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "badge_get_count() failed. Err = %d", ret);
		return false;
	}

	if (step) {
		(*count)++;
	} else if (*count > 0) {
		(*count)--;
	} else {
		dlog_print(DLOG_ERROR, LOG_TAG, "Cannot decrease the badge counter.");
		return false;
	}

	ret = badge_set_count(PACKAGE, *count);
	if (ret != BADGE_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "badge_set_count() failed. Err = %d", ret);
		return false;
	}

	return true;
}

/**
 * @brief Internal function which sets the launch arguments for the application
 * which is to be launched after notification is selected from the Notification Tray.
 * @param[in] notification Notification handle.
 * @param[in] argument Extra data.
 * @param[in] pkgname Name of the package containing the application to be launched.
 * @param[in] uri_text Uri text.
 * @param[in] mime_text Mime information.
 * @param[in] operation Operation to be performed.
 * @param[in] array_result Array of extra data.
 * @param[in] array_key The array of keys for extra data array items identification.
 */
static void notification_launch_cb(notification_h notification, const char *argument, const char *pkgname,
									const char *uri_text, const char *mime_text, const char *operation,
									const char **array_result, const char **array_key)
{
	int i = 0;
	app_control_h service = NULL;
	int ret = app_control_create(&service);
	if (ret != APP_CONTROL_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_control_create() failed. Err = %d", ret);
		return;
	}

	if (uri_text)
		app_control_set_uri(service, uri_text);

	if (mime_text)
		app_control_set_mime(service, mime_text);

	if (pkgname)
		app_control_set_app_id(service, pkgname);

	if (argument)
		app_control_add_extra_data(service, MESSAGE_POST, argument);

	if (operation)
		app_control_set_operation(service, operation);

	if (array_result && array_key)
		for (i = 0; array_key[i] && array_result[i]; i++)
			app_control_add_extra_data_array(service, array_key[i], &array_result[i], 1);

	notification_set_launch_option(notification, NOTIFICATION_LAUNCH_OPTION_APP_CONTROL, service);
	app_control_destroy(service);
}

/**
 * @brief Internal function which allocates the memory for notification data structure.
 * @param[out] notify_data The newly allocated notification data structure.
 * @return This function returns 'true' if the notification data structure was
 * successfully allocated, otherwise 'false' is returned.
 */
static bool _create_notification_data(notification_data **notify_data)
{
	*notify_data = (notification_data *)malloc(sizeof(notification_data));
	if (!*notify_data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Memory allocation failed.");
		return false;
	}

	return true;
}

/**
 * @brief Internal function which frees all the strings stored in an array.
 * @param[in] strings The array of strings to be freed.
 * @param[in] length The length of the array.
 */
static void _free_strings_array(char **strings, int length)
{
	int i;

	for (i = 0; i < length; i++)
			free(strings[i]);

	free(strings);
}

/**
 * @brief Internal function which supports checking and reporting errors
 * spawned by app-control API functions.
 * @param[in] err_code The error code spawned by app-control API function.
 * @param[in] err_message The message to be logged in case err_code differs from
 * APP_CONTROL_ERROR_NONE.
 * @return This function returns 'true' if err_code is equal to APP_CONTROL_ERROR_NONE,
 * otherwise 'false' is returned.
 */
static bool _report_app_control_error(int err_code, const char *err_message)
{
	if (err_code != APP_CONTROL_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "%s. Err = %d", err_message, err_code);

	return (err_code == APP_CONTROL_ERROR_NONE);
}
