/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <notification.h>
#include <stdio.h>
#include "$(appName).h"
#include "notification/notifications.h"
#include "notification/notification-defines.h"

#define NOFIFICATIONS_COUNT 8

enum notification_item {NOTIFICATION = 0, NOTIFICATION_NORMAL = 1, NOTIFICATION_BY_APP_CONTROL = 2, NOTIFICATION_BY_APP_ID = 4};

static void _notify_cb(notification_data *notify_data);
static void _notify_normal_cb(notification_data *notify_data);
static void _notify_by_app_control_cb(notification_data *notify_data);
static void _remove_notification_cb(notification_data *notify_data);
static void _notify_by_app_id_cb(notification_data *notify_data);
static void _remove_notification_by_app_id_cb(notification_data *notify_data);
static void _set_badge_number_by_app_id_cb(notification_data *notify_data);
static void _remove_badge_number_by_app_id_cb(notification_data *notify_data);
static notification_h _create_notification(const char *icon_path, const char *title_text, const char *content_text, const char *sound_path);
static bool _delete_notification(enum notification_item type);
static bool _delete_notification_items(void);
static bool _delete_notification_by_app_id(void);

static struct noti_info {
	launch_cb launch_func_cb;
	badge_cb badge_func_cb;
	notification_data noti[NOFIFICATIONS_COUNT];
} s_info = {
	.launch_func_cb = NULL,
	.badge_func_cb = NULL,
	.noti = {{ "Notify", NULL, 0, 0, _notify_cb, },
				{ "Notify(normal)", NULL, 0, 0, _notify_normal_cb, },
				{ "NotifyByAppControl", NULL, 0, 0, _notify_by_app_control_cb, },
				{ "RemoveNotification", NULL, 0, 0, _remove_notification_cb, },
				{ "NotifyByAppId", NULL, 0, 0, _notify_by_app_id_cb, },
				{ "RemoveNotificationByAppId", NULL, 0, 0, _remove_notification_by_app_id_cb, },
				{ "SetBadgeNumberByAppId", NULL, 0, 0, _set_badge_number_by_app_id_cb, },
				{ "RemoveBadgeNumberByAppId", NULL, 0, 0, _remove_badge_number_by_app_id_cb, },},
};

/**
 * @brief Assigns the callback functions.
 * @param[in] launch_func The callback function to be invoked on notification post.
 * @param[in] badge_func The callback function to be invoked on badge counter modification request.
 */
void notification_set_callbacks(launch_cb launch_func, badge_cb badge_func)
{
	s_info.launch_func_cb = launch_func;
	s_info.badge_func_cb = badge_func;
}

/**
 * @brief Obtains the list of all supported notifications.
 * @param[out] size The number is supported notifications.
 * @return This function returns an array of all the supported notifications.
 */
notification_data *notification_list_get(int *size)
{
	*size = NOFIFICATIONS_COUNT;
	return s_info.noti;
}

/**
 * @brief Deletes all the posted notifications.
 */
void delete_all_notifications(void)
{
	_delete_notification(NOTIFICATION);
	_delete_notification(NOTIFICATION_NORMAL);
	_delete_notification(NOTIFICATION_BY_APP_CONTROL);
	_delete_notification(NOTIFICATION_BY_APP_ID);
}

/**
 * @brief Internal callback function which posts a common notification.
 * This callback function is invoked when the user selects "Notify" item
 * from the list in "Notification" tab.
 * @param[in] notify_data Pointer to the notification data.
 */
static void _notify_cb(notification_data *notify_data)
{
	if (!notify_data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "notify_cb() failed. Invalid input argument.");
		return;
	}

	_delete_notification_items();

	notify_data->notification = _create_notification(MAIN_MENU_PATH, title_text, alert_text, NULL);
	if (notify_data->notification) {
		if (s_info.launch_func_cb)
			s_info.launch_func_cb(notify_data->notification, app_message_text, PACKAGE, NULL, NULL, NULL, NULL, NULL);
		notification_post(notify_data->notification);
	}

	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br>%s<br><br>%s<br>%s<br>", result_message,
			(notify_data->notification) ? result_message_success_text : result_message_failure_text,
			notify_message_text, alert_text, launch_argument_text,
			app_message_text);
}

/**
 * @brief Internal callback function which posts a normal notification.
 * This callback function is invoked when the user selects "Notify(normal)" item
 * from the list in "Notification" tab.
 * @param[in] notify_data Pointer to the notification data.
 */
static void _notify_normal_cb(notification_data *notify_data)
{
	int size_of_buffer = 0;
	char *full_thumbnail_path = NULL;
	unsigned int count = 0;

	if (!notify_data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "notify_normal_cb() failed. Invalid input argument.");
		return;
	}

	_delete_notification_items();

	notify_data->notification = _create_notification(ICON2_PATH, title_text, alert_text, SOUND_PATH);
	if (notify_data->notification) {
		if (s_info.launch_func_cb)
			s_info.launch_func_cb(notify_data->notification, app_message_text, PACKAGE, NULL, NULL, NULL, NULL, NULL);

		size_of_buffer = strlen(THUMBNAIL_PATH) + strlen(get_shared_res_path()) + 1;
		full_thumbnail_path = calloc(sizeof(char), size_of_buffer);
		if (full_thumbnail_path) {
			snprintf(full_thumbnail_path, size_of_buffer, "%s%s", get_shared_res_path(), THUMBNAIL_PATH);
			notification_set_image(notify_data->notification, NOTIFICATION_IMAGE_TYPE_BACKGROUND, full_thumbnail_path);
			free(full_thumbnail_path);
		}

		notification_set_text(notify_data->notification, NOTIFICATION_TEXT_TYPE_INFO_2, second_length, NULL, NOTIFICATION_VARIABLE_TYPE_NONE);
		notification_set_text(notify_data->notification, NOTIFICATION_TEXT_TYPE_INFO_SUB_2, line_text, NULL, NOTIFICATION_VARIABLE_TYPE_NONE);
		notification_post(notify_data->notification);
		if (s_info.badge_func_cb)
			s_info.badge_func_cb(&count, true);
	}

	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br>%s<br><br>%s<br>%s<br><br>%s<br>%d<br>", result_message,
			(notify_data->notification) ? result_message_success_text : result_message_failure_text,
			notify_message_text, alert_text, launch_argument_text,
			app_message_text, badge_is_text, count);
}

/**
 * @brief Internal callback function which posts a app-control notification.
 * This callback function is invoked when the user selects "NotifyByAppControl" item
 * from the list in "Notification" tab.
 * @param[in] notify_data Pointer to the notification data.
 */
static void _notify_by_app_control_cb(notification_data *notify_data)
{
	const char *array_result[] = { result_1_text, result_2_text, NULL };
	const char *array_key[] = { SERVICE_DATA_TEXT, SERVICE_DATA_TO, NULL };

	if (!notify_data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "notify_by_app_control_cb() failed. Invalid input argument.");
		return;
	}

	_delete_notification_items();

	notify_data->notification = _create_notification(ICON2_PATH, notification_app, app_control_to_me, NULL);
	if (notify_data->notification) {
		if (s_info.launch_func_cb)
			s_info.launch_func_cb(notify_data->notification, app_message_text, PACKAGE, uri_text, mime_text, operation_id_text, array_result, array_key);
		notification_post(notify_data->notification);
	}

	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br>%s<br><br>%s<br>%s<br><br>%s<br>%s<br>",
			result_message, (notify_data->notification) ? result_message_success_text : result_message_failure_text,
			notify_message_text, app_control_to_me, operation_id_is_text,
			operation_id_text, uri_is_text, uri_text);
}

/**
 * @brief Internal callback function which removes the notification.
 * This callback function is invoked when the user selects "RemoveNotification" item
 * from the list in "Notification" tab.
 * @param[in] notify_data Pointer to the notification data.
 */
static void _remove_notification_cb(notification_data *notify_data)
{
	bool is_success;

	if (!notify_data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "remove_notification_cb() failed. Invalid input argument.");
		return;
	}

	is_success = _delete_notification_items();
	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br>", result_message,
			(is_success) ? result_message_success_text : result_message_failure_text, remove_text);
}

/**
 * @brief Internal callback function which posts the notification by application ID.
 * This callback function is invoked when the user selects "NotifyByAppId" item
 * from the list in "Notification" tab.
 * @param[in] notify_data Pointer to the notification data.
 */
static void _notify_by_app_id_cb(notification_data *notify_data)
{
	if (!notify_data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "notify_by_app_id_cb() failed. Invalid input argument.");
		return;
	}

	_delete_notification_by_app_id();

	notify_data->notification = _create_notification(ICON2_PATH, title_text, notify_with_request, NULL);
	if (notify_data->notification) {
		if (s_info.launch_func_cb)
			s_info.launch_func_cb(notify_data->notification, app_message_text, SETTING_PKGNAME, NULL, NULL, NULL, NULL, NULL);
		notification_post(notify_data->notification);
	}

	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br>%s<br><br>%s<br>%s<br>", result_message,
			(notify_data->notification) ? result_message_success_text : result_message_failure_text,
			app_id_is_text, SETTING_PKGNAME, icon_path_is_text, ICON2_PATH);
}

/**
 * @brief Internal callback function which removes the notification posted by application ID.
 * This callback function is invoked when the user selects "RemoveNotificationByAppId" item
 * from the list in "Notification" tab.
 * @param[in] notify_data Pointer to the notification data.
 */
static void _remove_notification_by_app_id_cb(notification_data *notify_data)
{
	bool is_success;

	if (!notify_data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "remove_notification_by_app_id_cb() failed. Invalid input argument.");
		return;
	}

	is_success = _delete_notification_by_app_id();
	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br><br>%s<br>%s<br>", result_message,
			(is_success) ? result_message_success_text : result_message_failure_text,
			remove_text, target_app_id_text, SETTING_PKGNAME);
}

/**
 * @brief Internal callback function which sets the badge counter by application ID.
 * This callback function is invoked when the user selects "SetBadgeNumberByAppId" item
 * from the list in "Notification" tab.
 * @param[in] notify_data Pointer to the notification data.
 */
static void _set_badge_number_by_app_id_cb(notification_data *notify_data)
{
	unsigned int count = 0;
	bool is_success = false;

	if (!notify_data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "set_badge_number_by_app_id_cb() failed. Invalid input argument.");
		return;
	}

	if (s_info.badge_func_cb)
		is_success = s_info.badge_func_cb(&count, true);

	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br>%d<br><br>%s<br>%s<br>", result_message,
			(is_success) ? result_message_success_text : result_message_failure_text,
			badge_is_text, count, app_id_is_text, PACKAGE);
}

/**
 * @brief Internal callback function which removes the badge by application ID.
 * This callback function is invoked when the user selects "RemoveBadgeNumberByAppId" item
 * from the list in "Notification" tab.
 * @param[in] notify_data Pointer to the notification data.
 */
static void _remove_badge_number_by_app_id_cb(notification_data *notify_data)
{
	unsigned int count = 0;
	bool is_success = false;

	if (!notify_data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "remove_badge_number_by_app_id_cb() failed. Invalid input argument.");
		return;
	}

	if (s_info.badge_func_cb)
		is_success = s_info.badge_func_cb(&count, false);

	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br> <br>%s<br>%d<br> <br>%s<br>%s<br>", result_message,
			(is_success) ? result_message_success_text : result_message_failure_text, badge_is_text, count, app_id_is_text, PACKAGE);
}

/**
 * @brief Internal function which creates the notification.
 * @param[in] icon_path The path to the icon file to be added to the notification.
 * @param[in] title_text The text to be set as a notification's title.
 * @param[in] content_text The text to be set as a notification's content.
 * @param[in] sound_path The path to the sound file to be played when notification arrives.
 * @return The notification handle is returned or NULL on error.
 */
static notification_h _create_notification(const char *icon_path, const char *title_text, const char *content_text, const char *sound_path)
{
	int size_of_buffer = 0;
	char *full_icon_path = NULL;
	char *full_sound_path = NULL;

	notification_h notify = notification_create(NOTIFICATION_TYPE_NOTI);
	if (!notify) {
		dlog_print(DLOG_ERROR, LOG_TAG, "notification_create(NOTIFICATION_TYPE_NOTI) failed.");
		return NULL;
	}

	if (icon_path) {
		size_of_buffer = strlen(icon_path) + strlen(get_shared_res_path()) + 1;
		full_icon_path = calloc(sizeof(char), size_of_buffer);
		if (full_icon_path) {
			snprintf(full_icon_path, size_of_buffer, "%s%s", get_shared_res_path(), icon_path);
			notification_set_image(notify, NOTIFICATION_IMAGE_TYPE_ICON, full_icon_path);
			free(full_icon_path);
		}
	}

	if (sound_path) {
		size_of_buffer = strlen(sound_path) + strlen(get_shared_res_path()) + 1;
		full_sound_path = calloc(sizeof(char), size_of_buffer);
		if (full_sound_path) {
			snprintf(full_sound_path, size_of_buffer, "%s%s", get_shared_res_path(), sound_path);
			notification_set_sound(notify, NOTIFICATION_SOUND_TYPE_USER_DATA, full_sound_path);
			free(full_sound_path);
		}
	}

	notification_set_text(notify, NOTIFICATION_TEXT_TYPE_TITLE, title_text, NULL, NOTIFICATION_VARIABLE_TYPE_NONE);
	notification_set_text(notify, NOTIFICATION_TEXT_TYPE_CONTENT, content_text, NULL, NOTIFICATION_VARIABLE_TYPE_NONE);

	return notify;
}

/**
 * @brief Internal function which deletes the notification of specified type.
 * @param[in] type The type of notification to be deleted.
 * @return This function returns 'true' on successful notification removal,
 * otherwise 'false' is returned.
 */
static bool _delete_notification(enum notification_item type)
{
	int err = NOTIFICATION_ERROR_NONE;
	notification_data *data = NULL;

	if ((int)type < NOFIFICATIONS_COUNT)
		data = &s_info.noti[type];

	if (data && data->notification) {
		err = notification_delete(data->notification);
		data->notification = NULL;
	}

	return (err == NOTIFICATION_ERROR_NONE);
}

/**
 * @brief Internal function which deletes all the common, normal and app-control
 * notifications.
 * @return This function returns 'true' on successful notifications removal,
 * otherwise 'false' is returned.
 */
static bool _delete_notification_items(void)
{
	bool is_success;

	is_success = _delete_notification(NOTIFICATION);
	is_success = _delete_notification(NOTIFICATION_NORMAL) && is_success;
	is_success = _delete_notification(NOTIFICATION_BY_APP_CONTROL) && is_success;

	return is_success;
}

/**
 * @brief Internal function which deletes the notification posted by application ID.
 * @return This function returns 'true' on successful notification removal,
 * otherwise 'false' is returned.
 */
static bool _delete_notification_by_app_id(void)
{
	return _delete_notification(NOTIFICATION_BY_APP_ID);
}
