/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <notification.h>
#include <badge.h>
#include <math.h>
#include <stdio.h>
#include "$(appName).h"
#include "notification/ongoing-notifications.h"
#include "notification/notification-defines.h"

#define NOFIFICATIONS_COUNT 8

enum ongoing_notification {ONGOING_NOTIFICATION_PERCENT = 0, ONGOING_NOTIFICATION_BYTE = 1, ONGOING_NOTIFICATION_TEXT = 2,
							ONGOING_NOTIFICATION_BY_APP_CONTROL = 4, ONGOING_NOTIFICATION_BY_APP_ID = 6};

static void _ongoing_notification_cb(notification_data *notify_data);
static void _ongoing_notification_byte_cb(notification_data *notify_data);
static void _ongoing_notification_text_cb(notification_data *notify_data);
static void _remove_ongoing_notification_cb(notification_data *notify_data);
static void _ongoing_notification_by_app_control_cb(notification_data *notify_data);
static void _remove_notification_by_app_control_cb(notification_data *notify_data);
static void _ongoing_notification_by_app_id_cb(notification_data *notify_data);
static void _remove_ongoing_notification_by_app_id_cb(notification_data *notify_data);
static notification_h _create_ongoing_notification(const char *image_path, const char *title_text);
static bool _delete_ongoing_notification(enum ongoing_notification type);
static void _notification_content_text_set(notification_h notification, const char *content_text, int press_count);

static struct noti_info {
	launch_cb launch_func_cb;
	badge_cb badge_func_cb;
	notification_data noti[NOFIFICATIONS_COUNT];
} s_info = {
	.launch_func_cb = NULL,
	.badge_func_cb = NULL,
	.noti = {{ "OngoingNotification(%)", NULL,  0, 0, _ongoing_notification_cb, },
				{ "OngoingNotification(byte)", NULL,  0, 0, _ongoing_notification_byte_cb, },
				{ "OngoingNotification(text)", NULL,  0, 0, _ongoing_notification_text_cb, },
				{ "RemoveOngoingNotification", NULL,  0, 0, _remove_ongoing_notification_cb, },
				{ "OngoingNotificationByAppControl", NULL,  0, 0, _ongoing_notification_by_app_control_cb, },
				{ "RemoveNotificationByAppControl", NULL,  0, 0, _remove_notification_by_app_control_cb, },
				{ "OngoingNotificationByAppId", NULL,  0, 0, _ongoing_notification_by_app_id_cb, },
				{ "RemoveNotificationByAppId", NULL,  0, 0, _remove_ongoing_notification_by_app_id_cb, },},
};

/**
 * @brief Assigns the callback functions.
 * @param[in] launch_func The callback function to be invoked on ongoing notification post.
 * @param[in] badge_func The callback function to be invoked on badge counter modification request.
 */
void ongoing_notification_set_callbacks(launch_cb launch_func, badge_cb badge_func)
{
	s_info.launch_func_cb = launch_func;
	s_info.badge_func_cb = badge_func;
}

/**
 * @brief Obtains the list of all supported ongoing notifications.
 * @param[out] size The number is supported ongoing notifications.
 * @return This function returns an array of all the supported ongoing notifications.
 */
const notification_data *ongoing_notification_list_get(int *size)
{
	*size = NOFIFICATIONS_COUNT;
	return s_info.noti;
}

/**
 * @brief Deletes all the posted ongoing notifications.
 */
void delete_all_ongoing_notifications(void)
{
	_delete_ongoing_notification(ONGOING_NOTIFICATION_PERCENT);
	_delete_ongoing_notification(ONGOING_NOTIFICATION_BYTE);
	_delete_ongoing_notification(ONGOING_NOTIFICATION_TEXT);
	_delete_ongoing_notification(ONGOING_NOTIFICATION_BY_APP_CONTROL);
	_delete_ongoing_notification(ONGOING_NOTIFICATION_BY_APP_ID);
}

/**
 * @brief Internal callback function which posts a progress ongoing notification.
 * This callback function is invoked when the user selects "OngoingNotification(%)" item
 * from the list in "Ongoing Notification" tab.
 * @param[in] notify_data Pointer to the notification data.
 */
static void _ongoing_notification_cb(notification_data *notify_data)
{
	char launch_argument[TEXT_SHORT_LEN] = {'\0'};
	bool to_post = false;
	int value;

	if (!notify_data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_ongoing_notification_cb() failed. Invalid input argument.");
		return;
	}

	notify_data->press_count++;
	if (!notify_data->notification) {
		to_post = true;
		notify_data->notification = _create_ongoing_notification(ICON2_PATH, notify_data->name);
		if (!notify_data->notification)
			return;
	}

	notify_data->progress_value += percentage_increment;
	if (notify_data->progress_value > percentage_max)
		notify_data->progress_value = 0;

	notification_set_progress(notify_data->notification, notify_data->progress_value);
	_notification_content_text_set(notify_data->notification, percentage_content_text, notify_data->press_count);

	value = round(percentage_multiplier * notify_data->progress_value);

	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br> <br>%s<br>%s%d<br> <br>%s<br>%s = %d<br> <br>%s<br>%d<br>", result_message,
			(notify_data->notification) ? result_message_success_text : result_message_failure_text,
					notify_message_text, percentage_content_text, notify_data->press_count, launch_argument_text, app_message_text, value,
					ongoing_activity_progress_text, value);

	snprintf(launch_argument, TEXT_SHORT_LEN, "%s = %d", app_message_text, value);

	if (s_info.launch_func_cb)
		s_info.launch_func_cb(notify_data->notification, launch_argument, PACKAGE, NULL, NULL, NULL, NULL, NULL);

	if (to_post)
		notification_post(notify_data->notification);
	else
		notification_update(notify_data->notification);
}

/**
 * @brief Internal callback function which posts a byte ongoing notification.
 * This callback function is invoked when the user selects "OngoingNotification(byte)" item
 * from the list in "Ongoing Notification" tab.
 * @param[in] notify_data Pointer to the notification data.
 */
static void _ongoing_notification_byte_cb(notification_data *notify_data)
{
	char launch_argument[TEXT_SHORT_LEN] = {'\0'};
	bool to_post = false;

	if (!notify_data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_ongoing_notification_byte_cb() failed. Invalid input argument.");
		return;
	}

	notify_data->press_count++;
	if (!notify_data->notification) {
		to_post = true;
		notify_data->notification = _create_ongoing_notification(ICON2_PATH, notify_data->name);
		if (!notify_data->notification)
			return;
	}

	if (notify_data->progress_value > byte_max)
		notify_data->progress_value = 0;

	notification_set_size(notify_data->notification, notify_data->progress_value);
	_notification_content_text_set(notify_data->notification, byte_content_text, notify_data->press_count);

	snprintf(launch_argument, TEXT_SHORT_LEN, "%s = %d", app_message_text, (int)round(notify_data->progress_value));

	if (s_info.launch_func_cb)
		s_info.launch_func_cb(notify_data->notification, launch_argument, PACKAGE, NULL, NULL, NULL, NULL, NULL);

	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br>%s%d<br> <br>%s<br>%s = %d<br><br>%s<br>%d<br>", result_message,
			(notify_data->notification) ? result_message_success_text : result_message_failure_text,
			notify_message_text, byte_content_text, notify_data->press_count, launch_argument_text, app_message_text, (int)round(notify_data->progress_value),
			ongoing_activity_progress_text, (int)round(notify_data->progress_value));

	notify_data->progress_value += byte_increment;
	if (to_post)
		notification_post(notify_data->notification);
	else
		notification_update(notify_data->notification);
}

/**
 * @brief Internal callback function which posts a text ongoing notification.
 * This callback function is invoked when the user selects "OngoingNotification(text)" item
 * from the list in "Ongoing Notification" tab.
 * @param[in] notify_data Pointer to the notification data.
 */
static void _ongoing_notification_text_cb(notification_data *notify_data)
{
	bool to_post = false;

	if (!notify_data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_ongoing_notification_text_cb() failed. Invalid input argument.");
		return;
	}

	notify_data->press_count++;
	if (!notify_data->notification) {
		to_post = true;
		notify_data->notification = _create_ongoing_notification(ICON2_PATH, notify_data->name);
		if (!notify_data->notification)
			return;
	}

	notification_set_layout(notify_data->notification, NOTIFICATION_LY_ONGOING_EVENT);
	_notification_content_text_set(notify_data->notification, text_content_text, notify_data->press_count);

	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br>%s%d<br><br>%s<br>%s<br>", result_message,
			(notify_data->notification) ? result_message_success_text : result_message_failure_text,
			notify_message_text, text_content_text, notify_data->press_count, launch_argument_text, app_message_text);

	if (s_info.launch_func_cb)
		s_info.launch_func_cb(notify_data->notification, app_message_text, PACKAGE, NULL, NULL, NULL, NULL, NULL);

	if (to_post)
		notification_post(notify_data->notification);
	else
		notification_update(notify_data->notification);
}

/**
 * @brief Internal callback function which removes ongoing progress/byte/text notification.
 * This callback function is invoked when the user selects "RemoveOngoingNotification" item
 * from the list in "Ongoing Notification" tab.
 * @param[in] notify_data Pointer to the notification data.
 */
static void _remove_ongoing_notification_cb(notification_data *notify_data)
{
	bool is_success;

	if (!notify_data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_remove_ongoing_notification_cb() failed. Invalid input argument.");
		return;
	}

	is_success = _delete_ongoing_notification(ONGOING_NOTIFICATION_PERCENT);
	is_success = _delete_ongoing_notification(ONGOING_NOTIFICATION_BYTE) && is_success;
	is_success = _delete_ongoing_notification(ONGOING_NOTIFICATION_TEXT) && is_success;

	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br>", result_message,
			(is_success) ? result_message_success_text : result_message_failure_text, remove_ongoing_notification_text);
}

/**
 * @brief Internal callback function which posts a ongoing notification by app-control.
 * This callback function is invoked when the user selects "OngoingNotificationByAppControl" item
 * from the list in "Ongoing Notification" tab.
 * @param[in] notify_data Pointer to the notification data.
 */
static void _ongoing_notification_by_app_control_cb(notification_data *notify_data)
{
	int value;
	bool to_post = false;
	unsigned int count = 0;

	if (!notify_data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_ongoing_notification_by_app_control_cb() failed. Invalid input argument.");
		return;
	}

	notify_data->press_count++;
	if (!notify_data->notification) {
		to_post = true;
		notify_data->notification = _create_ongoing_notification(ICON1_PATH, notify_data->name);
		if (!notify_data->notification)
			return;
	}

	notify_data->progress_value += percentage_increment;
	if (notify_data->progress_value > percentage_max)
		notify_data->progress_value = 0;

	notification_set_progress(notify_data->notification, notify_data->progress_value);
	_notification_content_text_set(notify_data->notification, percentage_content_text, notify_data->press_count);

	if (s_info.badge_func_cb)
		s_info.badge_func_cb(&count, true);

	value = round(percentage_multiplier * notify_data->progress_value);

	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br> <br>%s<br>%s%d<br> <br>%s<br>%s = %d<br> <br>%s<br>%d<br>", result_message,
			(notify_data->notification) ? result_message_success_text : result_message_failure_text,
			notify_message_text, percentage_content_text, notify_data->press_count, launch_argument_text, ongoing_by_app_control_text, value,
			ongoing_activity_progress_text, value);

	if (s_info.launch_func_cb)
		s_info.launch_func_cb(notify_data->notification, ongoing_by_app_control_text, PACKAGE, uri_text, mime_text, operation_id_text, NULL, NULL);

	if (to_post)
		notification_post(notify_data->notification);
	else
		notification_update(notify_data->notification);
}

/**
 * @brief Internal callback function which removes the ongoing notification posted by app-control.
 * This callback function is invoked when the user selects "RemoveNotificationByAppControl" item
 * from the list in "Ongoing Notification" tab.
 * @param[in] notify_data Pointer to the notification data.
 */
static void _remove_notification_by_app_control_cb(notification_data *notify_data)
{
	bool result;

	if (!notify_data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_remove_notification_by_app_control_cb() failed. Invalid input argument.");
		return;
	}

	result = _delete_ongoing_notification(ONGOING_NOTIFICATION_BY_APP_CONTROL);
	badge_remove(PACKAGE);

	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br>", result_message,
			(result) ? result_message_success_text : result_message_failure_text, remove_ongoing_notification_by_app_control_text);
}

/**
 * @brief Internal callback function which posts ongoing notification by application ID.
 * This callback function is invoked when the user selects "OngoingNotificationByAppId" item
 * from the list in "Ongoing Notification" tab.
 * @param[in] notify_data Pointer to the notification data.
 */
static void _ongoing_notification_by_app_id_cb(notification_data *notify_data)
{
	bool to_post = false;

	if (!notify_data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_ongoing_notification_by_app_id_cb() failed. Invalid input argument.");
		return;
	}

	notify_data->press_count++;
	if (!notify_data->notification) {
		to_post = true;
		notify_data->notification = _create_ongoing_notification(ICON1_PATH, notify_data->name);
		if (!notify_data->notification)
			return;
	}

	if (notify_data->progress_value > byte_max)
		notify_data->progress_value = 0;

	notification_set_size(notify_data->notification, notify_data->progress_value);
	_notification_content_text_set(notify_data->notification, byte_content_text,
			notify_data->press_count);

	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br> <br>%s<br>%s%d<br> <br>%s<br>%s = %d<br> <br>%s<br>%d<br>", result_message,
			(notify_data->notification) ? result_message_success_text : result_message_failure_text,
			notify_message_text, byte_content_text, notify_data->press_count, launch_argument_text, app_message_text, (int)round(notify_data->progress_value),
			ongoing_activity_progress_text, (int)round(notify_data->progress_value));

	if (s_info.launch_func_cb)
		s_info.launch_func_cb(notify_data->notification, app_message_text, SETTING_PKGNAME, NULL, NULL, NULL, NULL, NULL);

	notify_data->progress_value += byte_increment;
	if (to_post)
		notification_post(notify_data->notification);
	else
		notification_update(notify_data->notification);
}

/**
 * @brief Internal callback function which removes ongoing notification posted by application ID.
 * This callback function is invoked when the user selects "RemoveNotificationByAppId" item
 * from the list in "Ongoing Notification" tab.
 * @param[in] notify_data Pointer to the notification data.
 */
static void _remove_ongoing_notification_by_app_id_cb(notification_data *notify_data)
{
	bool result;

	if (!notify_data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_ongoing_notification_by_app_id_cb() failed. Invalid input argument.");
		return;
	}

	result = _delete_ongoing_notification(ONGOING_NOTIFICATION_BY_APP_ID);

	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br>", result_message,
			(result) ? result_message_success_text : result_message_failure_text, remove_ongoing_notification_by_app_id_text);
}

/**
 * @brief Internal function which creates an ongoing notification.
 * @param[in] image_path The path to the image file to be added to the notification.
 * @param[in] title_text The text to be set as a notification's title.
 * @return The ongoing notification handle is returned or NULL on error.
 */
static notification_h _create_ongoing_notification(const char *image_path, const char *title_text)
{
	int size_of_buffer = 0;
	char *full_icon_path = NULL;
	notification_h ongoing_notification = notification_create(NOTIFICATION_TYPE_ONGOING);
	if (!ongoing_notification) {
		dlog_print(DLOG_ERROR, LOG_TAG, "notification_create(NOTIFICATION_TYPE_ONGOING) failed.");
		return NULL;
	}

	if (image_path) {
		size_of_buffer = strlen(image_path) + strlen(get_shared_res_path()) + 1;
		full_icon_path = calloc(sizeof(char), size_of_buffer);
		if (full_icon_path) {
			snprintf(full_icon_path, size_of_buffer, "%s%s", get_shared_res_path(), image_path);
			notification_set_image(ongoing_notification, NOTIFICATION_IMAGE_TYPE_ICON, full_icon_path);
			free(full_icon_path);
		}
	}

	notification_set_property(ongoing_notification, NOTIFICATION_PROP_DISABLE_TICKERNOTI);
	notification_set_text(ongoing_notification, NOTIFICATION_TEXT_TYPE_TITLE, title_text, NULL, NOTIFICATION_VARIABLE_TYPE_NONE);

	return ongoing_notification;
}

/**
 * @brief Internal function which deletes the ongoing notification of specified type.
 * @param[in] type The type of the ongoing notification to be deleted.
 * @return This function returns 'true' on successful notification removal,
 * otherwise 'false' is returned.
 */
static bool _delete_ongoing_notification(enum ongoing_notification type)
{
	int err = NOTIFICATION_ERROR_NONE;
	notification_data *data = NULL;

	if ((int)type < NOFIFICATIONS_COUNT)
		data = &s_info.noti[type];

	if (data && data->notification) {
		err = notification_delete(data->notification);
		data->notification = NULL;
		data->press_count = 0;
		data->progress_value = 0;
	}

	return (err == NOTIFICATION_ERROR_NONE);
}

/**
 * @brief Internal function which sets the text content of the ongoing notification.
 * @param[in] notification The notification handle which text content is to be set.
 * @param[in] content_text The text to be set as a content of the ongoing notification.
 * @param[in] press_count The counter to be set in notification text content body.
 */
static void _notification_content_text_set(notification_h notification, const char *content_text, int press_count)
{
	char text[TEXT_SHORT_LEN] = {'\0'};
	snprintf(text, sizeof(text), "%s%d", content_text, press_count);
	notification_set_text(notification, NOTIFICATION_TEXT_TYPE_CONTENT, text, NULL, NOTIFICATION_VARIABLE_TYPE_NONE);
}
