/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <efl_extension.h>
#include <Elementary.h>
#include "$(appName).h"
#include "notification/notifications.h"
#include "notification/ongoing-notifications.h"
#include "notification/notification-common.h"
#include "view.h"
#include "view_defines.h"

typedef enum {NT_NONE, NT_NOTIFICATION, NT_ONGOING_NOTIFICATION} notification_t;

static struct view_info {
	Evas_Object *win;
	Evas_Object *conform;
	Evas_Object *navi;
} s_info = {
	.win = NULL,
	.conform = NULL,
	.navi = NULL,
};

static void _delete_win_request_cb(void *data, Evas_Object *obj, void *event_info);
static void _navi_back_cb(void *data, Evas_Object *obj, void *event_info);
static void _toolbar_tab_changed_cb(void *data, Evas_Object *obj, void *event_info);
static void _noti_list_item_selected_cb(void *data, Evas_Object *obj, void *event_info);
static void _get_app_resource(const char *edj_file_in, char *edj_path_out);
static bool _create_main_layout(void);

/**
 * @brief Creates essential objects: window, conformant and layout.
 */
Eina_Bool view_create(void *user_data)
{
	/* Create the window */
	s_info.win = view_create_win(PACKAGE);
	if (s_info.win == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create a window.");
		return EINA_FALSE;
	}

	/* Create the conformant */
	s_info.conform = view_create_conformant(s_info.win);
	if (s_info.conform == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create a conformant");
		return EINA_FALSE;
	}

	if (!view_create_background(s_info.conform, "elm.swallow.bg"))
		return EINA_FALSE;

	s_info.navi = view_create_naviframe(s_info.conform, "elm.swallow.content");
	if (!s_info.navi)
		return EINA_FALSE;

	if (!_create_main_layout())
		return EINA_FALSE;

	/* Show the window after main view is set up */
	evas_object_show(s_info.win);

	return EINA_TRUE;
}

/**
 * @brief Creates a basic window named package.
 * @param[in] pkg_name Name of the window
 */
Evas_Object *view_create_win(const char *pkg_name)
{
	Evas_Object *win = NULL;

	/*
	 * Window
	 * Create and initialize elm_win.
	 * elm_win is mandatory to manipulate the window.
	 */
	win = elm_win_util_standard_add(pkg_name, pkg_name);
	elm_win_conformant_set(win, EINA_TRUE);
	elm_win_indicator_mode_set(win, ELM_WIN_INDICATOR_SHOW);
	elm_win_autodel_set(win, EINA_TRUE);

	evas_object_smart_callback_add(win, "delete,request", _delete_win_request_cb, NULL);

	return win;
}

/**
 * @brief Creates a conformant without indicator for mobile app.
 * @param[in] win The object to which you want to set this conformant
 * Conformant is mandatory for base GUI to have proper size
 */
Evas_Object *view_create_conformant(Evas_Object *win)
{
	/*
	 * Conformant
	 * Create and initialize elm_conformant.
	 * elm_conformant is mandatory for base GUI to have proper size
	 * when indicator or virtual keypad is visible.
	 */
	Evas_Object *conform = NULL;

	if (win == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "window is NULL.");
		return NULL;
	}

	conform = elm_conformant_add(win);
	evas_object_size_hint_weight_set(conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(win, conform);

	evas_object_show(conform);

	return conform;
}

/**
 * @brief Creates a layout object for parent object based on provided EDJE script.
 * @param[in] parent The parent object for layout object.
 * @param[in] edj_file_name The relative path to the layout EDJE script.
 * @param[in] edj_group The name of the group to be loaded from the EDJE script.
 * @return The function returns layout object if it was created successfully,
 * otherwise 'NULL' is returned.
 */
Evas_Object *view_create_layout(Evas_Object *parent, const char *edj_file_name, const char *edj_group)
{
	char edj_path[PATH_MAX] = {0, };
	Evas_Object *layout = NULL;

	if (!parent || !edj_file_name || !edj_group) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong input arguments.");
		return NULL;
	}

	_get_app_resource(edj_file_name, edj_path);

	layout = elm_layout_add(parent);
	if (!layout) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_layout_add() failed.");
		return NULL;
	}

	elm_layout_theme_set(layout, "layout", "application", "default");

	if (!elm_layout_file_set(layout, edj_path, edj_group)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_layout_file_set() failed.");
		evas_object_del(layout);
		return NULL;
	}

	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	return layout;
}

/**
 * @brief Creates a background for parent object and sets it to designated part.
 * @param[in] parent The parent object for background.
 * @param[in] part_name The name of the part within parent object
 * where the background is to be set,
 * @return This function returns a background object if is was successfully created,
 * otherwise NULL is returned.
 */
Evas_Object *view_create_background(Evas_Object *parent, const char *part_name)
{
	Evas_Object *bg = elm_bg_add(parent);
	if (!bg) {
		dlog_print(DLOG_ERROR, LOG_TAG, "elm_bg_add() failed.");
		return NULL;
	}

	evas_object_size_hint_weight_set(bg, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_show(bg);

	elm_object_part_content_set(parent, part_name, bg);

	return bg;
}

/**
 * @brief Creates a naviframe for parent object and sets it to designated part.
 * @param[in] parent The parent object for naviframe.
 * @param[in] part_name The name of the part within parent object
 * where the naviframe is to be set,
 * @return This function returns a naviframe object if is was successfully created,
 * otherwise NULL is returned.
 */
Evas_Object *view_create_naviframe(Evas_Object *parent, const char *part_name)
{
	Evas_Object *navi = elm_naviframe_add(parent);
	if (!navi) {
		dlog_print(DLOG_ERROR, LOG_TAG, "elm_naviframe_add() failed.");
		return NULL;
	}

	elm_naviframe_prev_btn_auto_pushed_set(navi, EINA_FALSE);

	eext_object_event_callback_add(navi, EEXT_CALLBACK_BACK, _navi_back_cb, NULL);
	elm_object_part_content_set(s_info.conform, part_name, navi);

	return navi;
}

/**
 * @brief Creates a toolbar for parent object.
 * @param[in] parent The parent object for toolbar.
 * @return This function returns a toolbar object if is was successfully created,
 * otherwise NULL is returned.
 */
Evas_Object *view_create_toolbar(Evas_Object *parent)
{
	Evas_Object *toolbar = elm_toolbar_add(parent);
	if (!toolbar) {
		dlog_print(DLOG_ERROR, LOG_TAG, "elm_toolbar_add() failed.");
		return NULL;
	}

	elm_toolbar_shrink_mode_set(toolbar, ELM_TOOLBAR_SHRINK_EXPAND);
	elm_toolbar_transverse_expanded_set(toolbar, EINA_TRUE);
	elm_toolbar_select_mode_set(toolbar, ELM_OBJECT_SELECT_MODE_ALWAYS);
	elm_object_style_set(toolbar, "tabbar");

	evas_object_show(toolbar);

	return toolbar;
}

/**
 * @brief Creates a list for parent object.
 * @param[in] parent The parent object for a list.
 * @return This function returns a list object if is was successfully created,
 * otherwise NULL is returned.
 */
Evas_Object *view_create_list(Evas_Object *parent)
{
	Evas_Object *list = elm_list_add(parent);
	if (!list) {
		dlog_print(DLOG_ERROR, LOG_TAG, "elm_list_add() failed.");
		return NULL;
	}

	elm_list_go(list);
	evas_object_show(list);

	return list;
}

/**
 * @brief Creates a new layout object to display the notification and sets it
 * to the naviframe object. The previously set layout is prior dismissed.
 * @param[in] notify_data The notification details to be displayed.
 * @return This function returns 'true' if the layout was successfully displayed,
 * otherwise NULL is returned.
 */
bool view_display_notification(notification_data *notify_data)
{
	static Evas_Object *layout = NULL;

	if (layout)
		evas_object_del(layout);

	layout = view_create_layout(s_info.navi, EDJ_NOTI, GRP_NOTI);
	if (!layout)
		return false;

	elm_naviframe_item_push(s_info.navi, "Notify result", NULL, NULL, layout, NULL);

	elm_object_part_text_set(layout, PART_NOTI_TITLE, notify_data->name);
	elm_object_part_text_set(layout, PART_NOTI_TEXT_CONTENT, notify_data->result_text);

	return true;
}

/**
 * @brief Rises the main window.
 */
void view_rise_window(void)
{
	elm_win_raise(s_info.win);
}

/**
 * @brief Destroys window and frees its resources.
 */
void view_destroy(void)
{
	if (s_info.win == NULL)
		return;

	evas_object_del(s_info.win);
}

/**
 * @brief Internal callback function invoked when the main window needs to be destroyed.
 * @param[in] data The user data passed to the evas_object_smart_callback_add() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] event_info The structure containing the information on this event.
 */
static void _delete_win_request_cb(void *data, Evas_Object *obj, void *event_info)
{
	ui_app_exit();
}

/**
 * @brief Internal callback function invoked when the HW Back button is pressed.
 * @param[in] data The user data passed to the eext_object_event_callback_add() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] event_info The structure containing the information on this event.
 */
static void _navi_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	if (elm_naviframe_top_item_get(s_info.navi) == elm_naviframe_bottom_item_get(s_info.navi))
		elm_win_lower(s_info.win);
	else
		elm_naviframe_item_pop(s_info.navi);
}

/**
 * @brief Internal callback function invoked when the toolbar's tab is changed.
 * Internally, this function creates a new content for selected toolbar's tab.
 * @param[in] data The user data passed to the elm_toolbar_item_append() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] event_info The structure containing the information on this event.
 */
static void _toolbar_tab_changed_cb(void *data, Evas_Object *obj, void *event_info)
{
	static notification_t current_noti_type = NT_NONE;
	notification_t noti_type = (notification_t)data;
	const notification_data *notification_list = NULL;
	Elm_Object_Item *item;
	Evas_Object *list;
	int size = 0;
	int i;

	if (current_noti_type == noti_type)
		return;

	item = elm_naviframe_top_item_get(s_info.navi);
	list = elm_object_item_part_content_unset(item, "default");
	if (list)
		evas_object_del(list);

	list = view_create_list(s_info.navi);
	if (!list) {
		current_noti_type = NT_NONE;
		return;
	}

	elm_object_item_part_content_set(item, "default", list);

	switch (noti_type) {
	case NT_NOTIFICATION:
		notification_list = notification_list_get(&size);
		break;
	case NT_ONGOING_NOTIFICATION:
		notification_list = ongoing_notification_list_get(&size);
		break;
	default:
		current_noti_type = NT_NONE;
		break;
	}

	for (i = 0; i < size; i++)
		elm_list_item_append(list, notification_list[i].name, NULL, NULL, _noti_list_item_selected_cb, &notification_list[i]);

	current_noti_type = noti_type;
}

/**
 * @brief Internal callback function invoked when the notification list's item
 * is selected by the user.
 * Internally, this function creates a new layout to display notification details.
 * @param[in] data The user data passed to the elm_list_item_append() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] event_info The structure containing the information on this event.
 */
static void _noti_list_item_selected_cb(void *data, Evas_Object *obj, void *event_info)
{
	notification_data *notification_info = (notification_data *)data;

	elm_list_item_selected_set(event_info, EINA_FALSE);

	if (notification_info->callback)
		notification_info->callback(notification_info);

	view_display_notification(notification_info);
}

/**
 * @brief Internal function which creates fully qualified path to the provided resource file.
 * @param[in] edj_file_in The file name.
 * @param[out] edj_path_out The fully qualified path to the edj_file_in file.
 */
static void _get_app_resource(const char *edj_file_in, char *edj_path_out)
{
	char *res_path = app_get_resource_path();
	if (res_path) {
		snprintf(edj_path_out, PATH_MAX, "%s%s", res_path, edj_file_in);
		free(res_path);
	}
}

/**
 * @brief Internal function which creates the main layout of the application.
 * The naviframe is used as a container for toolbar and genlist objects.
 * @return This function returns 'true' if the main layout was successfully created,
 * otherwise 'false' is returned.
 */
static bool _create_main_layout(void)
{
	Elm_Object_Item *item;
	Evas_Object *toolbar;

	toolbar = view_create_toolbar(s_info.navi);
	if (!toolbar)
		return false;

	item = elm_naviframe_item_push(s_info.navi, NULL, NULL, NULL, NULL, "tabbar/notitle");
	elm_object_item_part_content_set(item, "tabbar", toolbar);

	item = elm_toolbar_item_append(toolbar, NULL, "Notification", _toolbar_tab_changed_cb, (void *)NT_NOTIFICATION);
	elm_toolbar_item_append(toolbar, NULL, "Ongoing notification", _toolbar_tab_changed_cb, (void *)NT_ONGOING_NOTIFICATION);

	elm_toolbar_item_selected_set(item, EINA_TRUE);

	return true;
}
