/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <Eina.h>
#include <stddef.h>
#include <math.h>
#include <app_preference.h>
#include <app_common.h>
#include "$(appName).h"
#include "data.h"

#define BOOL_STR_TRUE "TRUE"
#define BOOL_STR_FALSE "FALSE"

static struct data_info {
	Eina_List *preferences;
	preference_cb pref_func;
} s_info = {
	.preferences = NULL,
	.pref_func = NULL,
};

static bool _existing_preference_item_cb(const char *key, void *user_data);
static void _preference_changed_cb(const char *key, void *user_data);
static void _add_key_value(const char *key, void *value, pref_value_type_t type);
static void _delete_preferences_list(void);
static void _delete_preference(const char *key);
static bool _string_to_int(const char *string, int *value);
static bool _string_to_double(const char *string, double *value);
static bool _string_to_bool(const char *string, bool *value);
static void _set_preference_changed_callback(const char *key);

/**
 * @brief Initialization function for data module.
 */
void data_initialize(void)
{
	int ret;

	ret = preference_foreach_item(_existing_preference_item_cb, NULL);
	if (ret != PREFERENCE_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "preference_foreach_item() failed. Err = %d.", ret);
		return;
	}
}

/**
 * @brief Finalization function for data module.
 */
void data_finalize(void)
{
	_delete_preferences_list();
}

/**
 * @brief Sets the data callback function.
 * @param[in] func The callback function invoked when the property is successfully
 * acquired and packed into the key-value type structure.
 */
void data_set_callback(preference_cb func)
{
	s_info.pref_func = func;
}

/**
 * @brief Function adds new preference if the one with given key does not exists,
 * otherwise the preference's value is updated.
 * @param[in] key The preference's key name.
 * @param[in] value The value of the preference with given name. This argument is of
 * string type, which is converted to the valid type base on the 'type' argument.
 * @param[in] type The type of the preference's value.
 * @return This function returns 'true' if the preference was successfully added/updated,
 * otherwise 'false' is returned.
 */
bool data_update_preference(const char *key, const char *value, pref_value_type_t type)
{
	int i_value = 0;
	double d_value = 0.0;
	bool b_value = false;
	void *ptr_value = NULL;
	const char *func_name = NULL;
	int ret;

	switch (type) {
	case PREF_INTEGER:
		if (!_string_to_int(value, &i_value))
			return false;

		ret = preference_set_int(key, i_value);
		ptr_value = (void *)&i_value;
		func_name = "preference_set_int";
		break;
	case PREF_DOUBLE:
		if (!_string_to_double(value, &d_value))
			return false;

		ret = preference_set_double(key, d_value);
		ptr_value = (void *)&d_value;
		func_name = "preference_set_double";
		break;
	case PREF_BOOL:
		if (!_string_to_bool(value, &b_value))
			return false;

		ret = preference_set_boolean(key, b_value);
		ptr_value = (void *)&b_value;
		func_name = "preference_set_boolean";
		break;
	case PREF_STRING:
		ret = preference_set_string(key, value);
		ptr_value = (void *)value;
		func_name = "preference_set_str";
		break;
	default:
		dlog_print(DLOG_WARN, LOG_TAG, "data_update_preference() failed. Unknown value type.");
		return false;
	}

	if (ret != PREFERENCE_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "%s() failed. Err = %d.", func_name, ret);
		return false;
	}

	_set_preference_changed_callback(key);
	_add_key_value(key, ptr_value, type);

	return true;
}

/**
 * @brief Function removes all the application's preferences.
 * @return This function returns 'true' if all the preferences was successfully removed,
 * otherwise 'false' is returned.
 */
bool data_remove_preferences(void)
{
	int ret = preference_remove_all();
	if (ret != PREFERENCE_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "preference_remove_all() failed. Err = %d", ret);
		return false;
	}

	_delete_preferences_list();

	dlog_print(DLOG_INFO, LOG_TAG, "All preferences removed successfully");

	return true;
}

/**
 * @brief Function removes the application's preference with given name.
 * @param[in] key The name of the preference to be removed.
 * @return This function returns 'true' if the preferences with given name was successfully removed,
 * otherwise 'false' is returned.
 */
bool data_remove_preference(const char *key)
{
	int ret;

	ret = preference_unset_changed_cb(key);
	if (ret != PREFERENCE_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "preference_unset_changed_cb() failed.");

	ret = preference_remove(key);
	if (ret != PREFERENCE_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "preference_remove() failed.");
		return false;
	}

	_delete_preference(key);
	dlog_print(DLOG_INFO, LOG_TAG, "Preference '%s' removed successfully", key);

	return true;
}

/**
 * @brief Internal callback function invoked by preference_foreach_item() function
 * on application's start-up during all the existing preferences enumeration.
 * This function is responsible for building internal list of all the preferences
 * associated with this application.
 * Internally, the preference is represented by a key name with assigned value
 * of specified type. This callback function provides only the key name with
 * unknown value. Thus one has to guess what is the type of the value associated
 * with the provided key name. This is performed by subsequent calls of the
 * following functions: preference_get_int(), preference_get_double(),
 * preference_get_string(), preference_get_boolean(). If the getter function
 * returns PREFERENCE_ERROR_INVALID_PARAMETER, then it means that the preference's
 * value is of some different type. If the getter function returns PREFERENCE_ERROR_NONE,
 * then it means that the preference value and type type is acquired successfully.
 * @return This function returns 'true' only to continue the preferences enumeration.
 */
static bool _existing_preference_item_cb(const char *key, void *user_data)
{
	int ret;
	int i_value = 0;
	double d_value = 0.0;
	bool b_value = false;
	char *s_value = NULL;

	ret = preference_get_int(key, &i_value);
	if (ret == PREFERENCE_ERROR_NONE) {
		_set_preference_changed_callback(key);
		_add_key_value(key, (void *)&i_value, PREF_INTEGER);
		return true;
	} else if (ret != PREFERENCE_ERROR_INVALID_PARAMETER) {
		dlog_print(DLOG_ERROR, LOG_TAG, "preference_get_int() failed. Err = %d.", ret);
		return true;
	}

	ret = preference_get_double(key, &d_value);
	if (ret == PREFERENCE_ERROR_NONE) {
		_set_preference_changed_callback(key);
		_add_key_value(key, (void *)&d_value, PREF_DOUBLE);
		return true;
	} else if (ret != PREFERENCE_ERROR_INVALID_PARAMETER) {
		dlog_print(DLOG_ERROR, LOG_TAG, "preference_get_double() failed. Err = %d.", ret);
		return true;
	}

	ret = preference_get_string(key, &s_value);
	if (ret == PREFERENCE_ERROR_NONE) {
		_set_preference_changed_callback(key);
		_add_key_value(key, (void *)s_value, PREF_STRING);
		free(s_value);
		return true;
	} else if (ret != PREFERENCE_ERROR_INVALID_PARAMETER) {
		dlog_print(DLOG_ERROR, LOG_TAG, "preference_get_string() failed. Err = %d.", ret);
		return true;
	}

	ret = preference_get_boolean(key, &b_value);
	if (ret == PREFERENCE_ERROR_NONE) {
		_set_preference_changed_callback(key);
		_add_key_value(key, (void *)&b_value, PREF_BOOL);
		return true;
	} else if (ret != PREFERENCE_ERROR_INVALID_PARAMETER) {
		dlog_print(DLOG_ERROR, LOG_TAG, "preference_get_boolean() failed. Err = %d.", ret);
		return true;
	}

	return true;
}

/**
 * @brief Internal callback function attached to each preference with
 * preference_set_changed_cb() function invoked on its value change.
 * @param[in] key The name of the preference.
 * @param[in] user_data The user data passed to the preference_set_changed_cb()
 * function.
 */
static void _preference_changed_cb(const char *key, void *user_data)
{
	dlog_print(DLOG_INFO, LOG_TAG, "Preferences changed: %s", key);
}

/**
 * @brief Function responsible for key-value structure creation for the preference
 * internal representation. This structure is added to the internal list.
 * @param[in] key The name of the preference.
 * @param[in] value The value associated with the preference.
 * @param[in] type The type of the value associated with the preference.
 */
void _add_key_value(const char *key, void *value, pref_value_type_t type)
{
	int val_size = 0;
	key_value_t *kv = (key_value_t *)malloc(sizeof(key_value_t));

	if (!kv) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_add_key_value() failed. Memory allocation error.");
		return;
	}

	switch (type) {
	case PREF_INTEGER:
		val_size = sizeof(int);
		break;
	case PREF_DOUBLE:
		val_size = sizeof(double);
		break;
	case PREF_BOOL:
		val_size = sizeof(bool);
		break;
	case PREF_STRING:
		val_size = strlen((char *)value) + 1;
		break;
	default:
		dlog_print(DLOG_WARN, LOG_TAG, "Unknown value type");
		return;
	}

	kv->key = strdup(key);
	kv->value_type = type;

	kv->value = (void *)malloc(val_size);
	memcpy(kv->value, value, val_size);

	s_info.preferences = eina_list_append(s_info.preferences, (void *)kv);
	if (!s_info.preferences) {
		dlog_print(DLOG_ERROR, LOG_TAG, "eina_list_append() failed. Key-value pair addition error.");
		return;
	}

	if (s_info.pref_func)
		s_info.pref_func(kv);
}

/**
 * @brief Internal function responsible for the preferences list consisting of
 * key-value pair structures deletion.
 */
static void _delete_preferences_list(void)
{
	key_value_t *kv = NULL;

	EINA_LIST_FREE(s_info.preferences, kv) {
		free(kv->key);
		free(kv->value);
		free(kv);
	}
}

/**
 * @brief Internal function responsible for deletion of an item with given key
 * from the preferences list consisting of key-value pair structures.
 * @param[in] key The preference name to be removed from an internal preferences list.
 */
static void _delete_preference(const char *key)
{
	Eina_List *l;
	key_value_t *kv = NULL;
	key_value_t *kv_to_rm = NULL;

	EINA_LIST_FOREACH(s_info.preferences, l, kv) {
		if (strncasecmp(kv->key, key, strlen(kv->key)) != 0 ||
			strlen(kv->key) != strlen(key))
			continue;
		kv_to_rm = kv;
		break;
	}

	if (kv_to_rm) {
		s_info.preferences = eina_list_remove(s_info.preferences, (const void *)kv_to_rm);
		free(kv_to_rm->key);
		free(kv_to_rm->value);
		free(kv_to_rm);
	}
}

/**
 * @brief Internal function which performs conversion of string to the integer value
 * with result checking.
 * Internally, for string conversion, the atoi() function is used which returns 0
 * on conversion error. In order to avoid returned error value and string == "0" mismatch,
 * an additional input string verification is performed.
 * @param[in] string The string to be converted to integer value.
 * @param[out] value The value converted from input string.
 * @return The function returns 'true' if the string conversion is successful,
 * otherwise 'false' is returned.
 */
static bool _string_to_int(const char *string, int *value)
{
	*value = atoi(string);

	if (*value == 0)
		return (strlen(string) == 1 && string[0] == '0');

	return true;
}

/**
 * @brief Internal function which performs conversion of string to the double value
 * with result checking.
 * Internally, for string conversion, the atof() function is used which returns 0.0
 * on conversion error. In order to avoid returned error value and string == "0.0" mismatch,
 * an additional input string verification is performed.
 * @param[in] string The string to be converted to double value.
 * @param[out] value The value converted from input string.
 * @return The function returns 'true' if the string conversion is successful,
 * otherwise 'false' is returned.
 */
static bool _string_to_double(const char *string, double *value)
{
	*value = atof(string);

	if (fabs(*value) < 0.0001)
		return ((strlen(string) == 1 && string[0] == '0') ||
				(strlen(string) == 3 && !strncmp(string, "0.0", 3)));

	return true;
}

/**
 * @brief Internal function which performs conversion of string to the boolean value.
 * Internally, for conversion, the simple strings comparison is performed.
 * @param[in] string The string to be converted to boolean value.
 * @param[out] value The value converted from input string.
 * @return The function returns 'true' if the string conversion is successful,
 * otherwise 'false' is returned.
 */
static bool _string_to_bool(const char *string, bool *value)
{
	if (strncasecmp(string, BOOL_STR_TRUE, strlen(string)) == 0 &&
		strlen(string) == strlen(BOOL_STR_TRUE))
		*value = true;
	else if (strncasecmp(string, BOOL_STR_FALSE, strlen(string)) == 0 &&
				strlen(string) == strlen(BOOL_STR_FALSE))
		*value = false;
	else
		return false;

	return true;
}

/**
 * @brief Internal function which attaches the preference's change callback
 * function.
 * @param[in] key The name of the preference to which the callback function
 * is attached.
 */
static void _set_preference_changed_callback(const char *key)
{
	int ret = preference_set_changed_cb(key, _preference_changed_cb, NULL);
	if (ret != PREFERENCE_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "preference_set_changed_cb() failed. Err %d.", ret);
}
