/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <efl_extension.h>
#include "$(appName).h"
#include "view.h"
#include "view_defines.h"

#define REMOVE_ALL_PREFS_CAPTION "Remove all"
#define UPDATE_PREF_CAPTION "Update"
#define ICON_DELETE_FILE_NAME "images/icon_delete.png"
#define DELETE_IMAGE_SIZE_MIN 25


typedef struct _preftype {
	char *caption;
	pref_value_type_t preftype;
} preftype_s;

preftype_s preftypes[PREF_MAX] = {
	{ .caption = "Integer", .preftype = PREF_INTEGER, },
	{ .caption = "Double", .preftype = PREF_DOUBLE, },
	{ .caption = "Bool", .preftype = PREF_BOOL, },
	{ .caption = "String", .preftype = PREF_STRING, },
};

static struct view_info {
	Evas_Object *win;
	Evas_Object *conform;
	Evas_Object *main_layout;
	Evas_Object *key_entry;
	Evas_Object *value_entry;
	Evas_Object *type_selector;
	Evas_Object *prefs_list;
	Elm_Genlist_Item_Class *itc;
	Eina_List *genlist_items;
	pref_value_type_t selected_type;
	preference_update_cb update_func;
	preference_remove_all_cb remove_all_func;
	preference_remove_cb remove_func;
} s_info = {
	.win = NULL,
	.conform = NULL,
	.main_layout = NULL,
	.key_entry = NULL,
	.value_entry = NULL,
	.type_selector = NULL,
	.prefs_list = NULL,
	.genlist_items = NULL,
	.selected_type = PREF_INTEGER,
	.update_func = NULL,
	.remove_all_func = NULL,
	.remove_func = NULL,
};

static void _delete_win_request_cb(void *data, Evas_Object *obj, void *event_info);
static void _layout_back_cb(void *data, Evas_Object *obj, void *event_info);
static void _update_click_cb(void *data, Evas_Object *obj, void *event_info);
static void _remove_all_click_cb(void *data, Evas_Object *obj, void *event_info);
static void _genlist_item_image_clicked_cb(void *data, Evas_Object *obj, void *event_info);
static void _type_selected_cb(void *data, Evas_Object *obj, void *event_info);
static char *_get_item_label_cb(void *data, Evas_Object *obj, const char *part);
static Evas_Object *_get_item_content_cb(void *data, Evas_Object *obj, const char *part);
static void _get_app_resource(const char *edj_file_in, char *edj_path_out);
static Eina_Bool _create_main_layout(void);
static Eina_Bool _create_pref_edit_panel_layout(void);
static Eina_Bool _create_pref_buttons_panel_layout(void);
static Eina_Bool _create_pref_list_panel_layout(void);
static Evas_Object *_create_radio_group(Evas_Object *parent, const char *target_part_name);
static bool _find_genlist_item(const char *key, Elm_Object_Item **item);

/**
 * @brief Creates essential objects: window, conformant and layout.
 */
Eina_Bool view_create(void)
{
	/* Create the window */
	s_info.win = view_create_win(PACKAGE);
	if (s_info.win == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create a window.");
		return EINA_FALSE;
	}

	/* Create the conformant */
	s_info.conform = view_create_conformant(s_info.win);
	if (s_info.conform == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create a conformant");
		return EINA_FALSE;
	}

	if (!_create_main_layout())
		return false;

	if (!_create_pref_edit_panel_layout())
		return false;

	if (!_create_pref_buttons_panel_layout())
		return false;

	if (!_create_pref_list_panel_layout())
		return false;

	/* Show the window after main view is set up */
	evas_object_show(s_info.win);

	return EINA_TRUE;
}

/**
 * @brief Sets the view callback functions.
 * @param[in] update_func The callback function invoked when the user clicks the 'Update' button.
 * @param[in] remove_all_func The callback function invoked when the user clicks the 'Remove all' button.
 * @param[in] remove_func The callback function invoked when the user clicks the 'x' button on the preferences list.
 */
void view_set_callbacks(preference_update_cb update_func, preference_remove_all_cb remove_all_func, preference_remove_cb remove_func)
{
	s_info.update_func = update_func;
	s_info.remove_all_func = remove_all_func;
	s_info.remove_func = remove_func;
}

/**
 * @brief Creates a basic window named package
 * @param[in] pkg_name Name of the window
 * @return The function returns window object if it was created successfully,
 * otherwise 'NULL' is returned.
 */
Evas_Object *view_create_win(const char *pkg_name)
{
	Evas_Object *win = NULL;

	/*
	 * Window
	 * Create and initialize elm_win.
	 * elm_win is mandatory to manipulate the window.
	 */
	win = elm_win_util_standard_add(pkg_name, pkg_name);
	elm_win_conformant_set(win, EINA_TRUE);
	elm_win_indicator_mode_set(win, ELM_WIN_INDICATOR_SHOW);
	elm_win_indicator_opacity_set(win, ELM_WIN_INDICATOR_OPAQUE);
	elm_win_autodel_set(win, EINA_TRUE);

	evas_object_smart_callback_add(win, "delete,request", _delete_win_request_cb, NULL);

	return win;
}

/**
 * @brief Creates a conformant object for parent object.
 * @param[in] parent The parent object for conformant object.
 * @return The function returns conformant object if it was created successfully,
 * otherwise 'NULL' is returned.
 */
Evas_Object *view_create_conformant(Evas_Object *parent)
{
	Evas_Object *conform = NULL;

	if (!parent) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong input arguments.");
		return NULL;
	}

	conform = elm_conformant_add(parent);
	if (!conform) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_conformant_add() failed.");
		return NULL;
	}

	evas_object_size_hint_weight_set(conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(parent, conform);
	evas_object_show(conform);

	return conform;
}

/**
 * @brief Creates a layout object for parent object based on provided EDJE script.
 * @param[in] parent The parent object for layout object.
 * @param[in] edj_file_name The relative path to the layout EDJE script.
 * @param[in] edj_group The name of the group to be loaded from the EDJE script.
 * @param[in] target_part_name The EDJE part's name where the layout is to be set.
 * @return The function returns layout object if it was created successfully,
 * otherwise 'NULL' is returned.
 */
Evas_Object *view_create_layout(Evas_Object *parent, const char *edj_file_name, const char *edj_group, const char *target_part_name)
{
	char edj_path[PATH_MAX] = {0, };
	Evas_Object *layout = NULL;

	if (!parent || !edj_file_name || !edj_group) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong input arguments.");
		return NULL;
	}

	_get_app_resource(edj_file_name, edj_path);

	layout = elm_layout_add(parent);
	if (!layout) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_layout_add() failed.");
		return NULL;
	}

	if (!elm_layout_file_set(layout, edj_path, edj_group)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_layout_file_set() failed.");
		evas_object_del(layout);
		return NULL;
	}

	if (target_part_name)
		elm_object_part_content_set(parent, target_part_name, layout);
	else
		elm_object_content_set(parent, layout);

	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	return layout;
}

/**
 * @brief Creates an entry object and sets it to the parent object.
 * @param[in] parent The parent object for entry object.
 * @param[in] target_part_name The name of the EDJE part where the entry object has to be set.
 * @return The function returns entry object if it was created successfully,
 * otherwise 'NULL' is returned.
 */
Evas_Object *view_create_entry(Evas_Object *parent, const char *target_part_name)
{
	Evas_Object *entry = elm_entry_add(parent);
	if (!entry) {
		dlog_print(DLOG_ERROR, LOG_TAG, "elm_entry_add() failed.");
		return NULL;
	}

	elm_entry_line_wrap_set(entry, ELM_WRAP_NONE);
	elm_entry_single_line_set(entry, EINA_TRUE);
	evas_object_size_hint_weight_set(entry, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(entry, EVAS_HINT_FILL, 0.5);
	elm_object_part_content_set(parent, target_part_name, entry);
	evas_object_show(entry);

	return entry;
}

/**
 * @brief Creates a button object and sets it to the parent object.
 * @param[in] parent The parent object for button object.
 * @param[in] target_part_name The name of the EDJE part where the button object has to be set.
 * @param[in] caption The caption to be displayed on the button.
 * @param[in] on_click_cb The callback function's handler to be invoked on button click.
 * @return The function returns button object if it was created successfully,
 * otherwise 'NULL' is returned.
 */
Evas_Object *view_create_button(Evas_Object *parent, const char *target_part_name, const char *caption, Evas_Smart_Cb on_click_cb)
{
	Evas_Object *button = elm_button_add(parent);
	if (!button) {
		dlog_print(DLOG_ERROR, LOG_TAG, "elm_button_add() failed.");
		return NULL;
	}

	elm_object_part_content_set(parent, target_part_name, button);
	elm_object_text_set(button, caption);

	evas_object_smart_callback_add(button, "clicked", on_click_cb, NULL);

	evas_object_show(button);

	return button;
}

/**
 * @brief Creates a genlist object and sets it to the parent object.
 * @param[in] parent The parent object for genlist object.
 * @param[in] target_part_name The name of the EDJE part where the genlist object has to be set.
 * @return The function returns genlist object if it was created successfully,
 * otherwise 'NULL' is returned.
 */
Evas_Object *view_create_genlist(Evas_Object *parent, const char *target_part_name)
{
	Evas_Object *list = elm_genlist_add(parent);
	if (!list) {
		dlog_print(DLOG_ERROR, LOG_TAG, "elm_genlist_add() failed.");
		return NULL;
	}

	evas_object_size_hint_weight_set(list, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_object_part_content_set(parent, target_part_name, list);
	evas_object_show(list);

	s_info.itc = elm_genlist_item_class_new();
	if (!s_info.itc) {
		evas_object_del(list);
		dlog_print(DLOG_ERROR, LOG_TAG, "elm_genlist_item_class_new() failed.");
		return NULL;
	}

	s_info.itc->item_style = "double_label";
	s_info.itc->func.text_get = _get_item_label_cb;
	s_info.itc->func.content_get = _get_item_content_cb;
	s_info.itc->func.state_get = NULL;
	s_info.itc->func.del = NULL;

	return list;
}

/**
 * @brief Function updates the preferences list with the given key-value structure.
 * If the preference list already contains an item representing given key-value structure,
 * then the list item is updated. Otherwise new item is appended.
 * @param[in] key_value The key-value structure representing a preference to be added.
 * @return Function returns 'true' is an item is successfully updated,
 * otherwise 'false' is returned.
 */
bool view_update_item(key_value_t *key_value)
{
	Elm_Object_Item *item = NULL;

	if (_find_genlist_item(key_value->key, &item)) {
		key_value_t *item_key_value = elm_object_item_data_get(item);

		if (!item_key_value)
			return false;

		elm_object_item_data_set(item, (void *)key_value);
		elm_genlist_item_update(item);

		return true;
	}

	item = elm_genlist_item_append(s_info.prefs_list, s_info.itc, (void *)key_value,
									NULL, ELM_GENLIST_ITEM_NONE, NULL, NULL);
	if (!item) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_genlist_item_append() failed.");
		return false;
	}

	s_info.genlist_items = eina_list_append(s_info.genlist_items, item);

	return true;
}

/**
 * @brief Destroys window and frees its resources.
 */
void view_destroy(void)
{
	if (s_info.win == NULL)
		return;

	evas_object_del(s_info.win);
	elm_genlist_item_class_free(s_info.itc);
}

/**
 * @brief Internal callback function invoked when the main window needs to be destroyed.
 * @param[in] data The user data passed to the evas_object_smart_callback_add() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] event_info The structure containing the information on this event.
 */
static void _delete_win_request_cb(void *data, Evas_Object *obj, void *event_info)
{
	ui_app_exit();
}

/**
 * @brief Internal callback function invoked on HW Back button press.
 * @param[in] data The user data passed to the eext_object_event_callback_add() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] event_info The structure containing the information on this event.
 */
static void _layout_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	/* Let window go to hide state. */
	elm_win_lower(s_info.win);
}

/**
 * @brief Internal callback function invoked on 'Update' button click.
 * This function is responsible for user input data acquisition. Once the
 * data is acquired, the main module is notified via the update callback
 * about the preference update request.
 * @param[in] data The user data passed to the evas_object_smart_callback_add() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] event_info The structure containing the information on this event.
 */
static void _update_click_cb(void *data, Evas_Object *obj, void *event_info)
{
	const char *key;
	const char *value;

	if (!s_info.update_func) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_update_click_cb() failed. Callback function not assigned");
		return;
	}

	key = elm_entry_entry_get(s_info.key_entry);
	value = elm_entry_entry_get(s_info.value_entry);
	if (!key || strlen(key) == 0 || !value || strlen(value) == 0) {
		dlog_print(DLOG_WARN, LOG_TAG, "Input data is not complete.");
		return;
	}

	s_info.update_func(key, value, s_info.selected_type);
}

/**
 * @brief Internal callback function invoked on 'Remove all' button click.
 * This function is responsible for main module notification about the
 * all preferences removal request sent via the remove all callback.
 * @param[in] data The user data passed to the evas_object_smart_callback_add() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] event_info The structure containing the information on this event.
 */
static void _remove_all_click_cb(void *data, Evas_Object *obj, void *event_info)
{
	if (!s_info.remove_all_func) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_remove_all_click_cb() failed. Callback function not assigned");
		return;
	}

	if (s_info.remove_all_func())
		elm_genlist_clear(s_info.prefs_list);
}

/**
 * @brief Internal callback function invoked on 'x' button click on the preferences list.
 * This function is responsible for main module notification about the
 * selected preference removal request sent via the remove callback.
 * @param[in] data The user data passed to the evas_object_smart_callback_add() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] event_info The structure containing the information on this event.
 */
static void _genlist_item_image_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
	char *key_name;
	Elm_Object_Item *item = NULL;
	key_value_t *key_val = (key_value_t *)data;

	if (!s_info.remove_func) {
		dlog_print(DLOG_ERROR, LOG_TAG, "_genlist_item_image_clicked_cb() failed. Callback function not assigned");
		return;
	}

	key_name = strdup(key_val->key);
	if (_find_genlist_item(key_name, &item) && s_info.remove_func(key_name))
		elm_object_item_del(item);

	free(key_name);
}

/**
 * @brief Internal callback function invoked on selection of a radio button representing preference type.
 * @param[in] data The user data passed to the evas_object_smart_callback_add() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] event_info The structure containing the information on this event.
 */
static void _type_selected_cb(void *data, Evas_Object *obj, void *event_info)
{
	preftype_s *preftype = (preftype_s *)data;

	s_info.selected_type = preftype->preftype;
}

/**
 * @brief Internal callback function invoked on list's item display.
 * This function is responsible for text display and it is assigned to the func.text_get
 * handler of genlist item's class structure.
 * The item's area consists if two labels:
 * - main/top label where the key's value type is displayed,
 * - sub/bottom label where the key name and assigned value is displayed.
 * @param[in] data The user data passed to the elm_genlist_item_append() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] part The name of the item's part being rendered.
 * @return The text string to be displayed is returned.
 */
static char *_get_item_label_cb(void *data, Evas_Object *obj, const char *part)
{
	char buff_main[256] = {0,};
	char *buff_sub = NULL;
	key_value_t *key_val = (key_value_t *)data;

	switch (key_val->value_type) {
	case PREF_INTEGER:
		snprintf(buff_main, sizeof(buff_main), "%s = %d", key_val->key, *((int *)key_val->value));
		buff_sub = "type: INTEGER";
		break;
	case PREF_DOUBLE:
		snprintf(buff_main, sizeof(buff_main), "%s = %.3f", key_val->key, *((double *)key_val->value));
		buff_sub = "type: DOUBLE";
		break;
	case PREF_BOOL:
		if (*((bool *)key_val->value))
			snprintf(buff_main, sizeof(buff_main), "%s = TRUE", key_val->key);
		else
			snprintf(buff_main, sizeof(buff_main), "%s = FALSE", key_val->key);
		buff_sub = "type: BOOLEAN";
		break;
	case PREF_STRING:
		snprintf(buff_main, sizeof(buff_main), "%s = %s", key_val->key, (char *)key_val->value);
		buff_sub = "type: STRING";
		break;
	default:
		snprintf(buff_main, sizeof(buff_main), "%s = ???", key_val->key);
		buff_sub = "type: UNKNOWN";
	}

	if (!strcmp((char *)part, "elm.text"))
		return strdup(buff_main);
	else if (!strcmp((char *)part, "elm.text.sub"))
		return strdup(buff_sub);

	return NULL;
}

/**
 * @brief Internal callback function invoked on list's item display.
 * This function is responsible for non-text objects handling and it is assigned
 * to the func.content_get handler of genlist item's class structure.
 * In the content area of an genlist item, the 'x' image is displayed.
 * @param[in] data The user data passed to the elm_genlist_item_append() function.
 * @param[in] obj The object invoking this callback function.
 * @param[in] part The name of the item's part being rendered.
 * @return The text string to be displayed is returned.
 */
static Evas_Object *_get_item_content_cb(void *data, Evas_Object *obj, const char *part)
{
	Evas_Object *image;
	char file_path[PATH_MAX] = {0,};

	if (!strcmp(part, "elm.swallow.icon")) {
		image = elm_image_add(obj);
		if (!image) {
			dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_image_add() failed.");
			return NULL;
		}

		_get_app_resource(ICON_DELETE_FILE_NAME, file_path);
		if (!elm_image_file_set(image, file_path, NULL)) {
			evas_object_del(image);
			dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_image_file_set() failed.");
			return NULL;
		}

		evas_object_size_hint_min_set(image, DELETE_IMAGE_SIZE_MIN, DELETE_IMAGE_SIZE_MIN);
		evas_object_size_hint_weight_set(image, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

		evas_object_smart_callback_add(image, "clicked", _genlist_item_image_clicked_cb, data);

		return image;
	}

	return NULL;
}

/**
 * @brief Internal function which creates fully qualified path to the provided resource file.
 * @param[in] edj_file_in The file name.
 * @param[out] edj_path_out The fully qualified path to the edj_file_in file.
 */
static void _get_app_resource(const char *edj_file_in, char *edj_path_out)
{
	char *res_path = app_get_resource_path();
	if (res_path) {
		snprintf(edj_path_out, PATH_MAX, "%s%s", res_path, edj_file_in);
		free(res_path);
	}
}

/**
 * @brief Internal function which creates the main layout.
 * @return The function returns 'EINA_TRUE' if the main layout was created successfully,
 * otherwise 'EINA_FALSE' is returned.
 */
static Eina_Bool _create_main_layout(void)
{
	s_info.main_layout = view_create_layout(s_info.conform, EDJ_FILE_NAME_MAIN, GROUP_MAIN, NULL);
	if (!s_info.main_layout)
		return EINA_FALSE;

	eext_object_event_callback_add(s_info.main_layout, EEXT_CALLBACK_BACK, _layout_back_cb, NULL);

	return EINA_TRUE;
}

/**
 * @brief Internal function which creates the preference edit panel layout.
 * @return The function returns 'EINA_TRUE' if the preference edit panel layout was created successfully,
 * otherwise 'EINA_FALSE' is returned.
 */
static Eina_Bool _create_pref_edit_panel_layout(void)
{
	Evas_Object *layout;

	layout = view_create_layout(s_info.main_layout, EDJ_FILE_NAME_PREF_EDIT_PANEL, GROUP_PREF_EDIT_PANEL, PART_MAIN_PREF_EDIT_PANEL);
	if (!layout)
		return EINA_FALSE;

	s_info.key_entry = view_create_entry(layout, PART_PREF_EDIT_PANEL_KEY_PANEL_ENTRY);
	if (!s_info.key_entry)
		return EINA_FALSE;

	s_info.value_entry = view_create_entry(layout, PART_PREF_EDIT_PANEL_VALUE_PANEL_ENTRY);
	if (!s_info.value_entry)
		return EINA_FALSE;

	s_info.type_selector = _create_radio_group(layout, PART_PREF_EDIT_PANEL_TYPE_PANEL_ENTRY);
	if (!s_info.type_selector)
		return EINA_FALSE;

	return EINA_TRUE;
}

/**
 * @brief Internal function which creates the buttons panel layout.
 * @return The function returns 'EINA_TRUE' if the buttons panel layout was created successfully,
 * otherwise 'EINA_FALSE' is returned.
 */
static Eina_Bool _create_pref_buttons_panel_layout(void)
{
	Evas_Object *layout;
	Evas_Object *button;

	layout = view_create_layout(s_info.main_layout, EDJ_FILE_NAME_PREF_BUTTONS_PANEL, GROUP_PREF_BUTTONS_PANEL, PART_MAIN_PREF_BUTTONS_PANEL);
	if (!layout)
		return EINA_FALSE;

	button = view_create_button(layout, PART_PREF_BUTTONS_PANEL_LEFT_BUTTON, REMOVE_ALL_PREFS_CAPTION, _remove_all_click_cb);
	if (!button)
		return EINA_FALSE;

	button = view_create_button(layout, PART_PREF_BUTTONS_PANEL_RIGHT_BUTTON, UPDATE_PREF_CAPTION, _update_click_cb);
	if (!button)
		return EINA_FALSE;

	return EINA_TRUE;
}

/**
 * @brief Internal function which creates the list panel layout.
 * @return The function returns 'EINA_TRUE' if the list panel layout was created successfully,
 * otherwise 'EINA_FALSE' is returned.
 */
static Eina_Bool _create_pref_list_panel_layout(void)
{
	s_info.prefs_list = view_create_genlist(s_info.main_layout, PART_MAIN_PREF_LIST_PANEL);
	if (!s_info.prefs_list)
		return EINA_FALSE;

	return EINA_TRUE;
}

/**
 * @brief Creates radio buttons group.
 * @param[in] parent The parent container for the radio buttons group.
 * @param[in] target_part_name The part's name where the radio group is to be set.
 * @return Function returns radio buttons group container (box) object on successful creation,
 * otherwise NULL is returned.
 */
static Evas_Object *_create_radio_group(Evas_Object *parent, const char *target_part_name)
{
	int i;
	char buff[256] = {0,};
	Evas_Object *group = NULL;
	Evas_Object *radio = NULL;
	Evas_Object *box = elm_box_add(parent);
	if (!box) {
		dlog_print(DLOG_ERROR, LOG_TAG, "elm_box_add() failed.");
		return NULL;
	}

	elm_box_horizontal_set(box, EINA_FALSE);
	evas_object_show(box);

	for (i = 0; i < PREF_MAX; i++) {
		if (!group) {
			group = radio = elm_radio_add(box);
		} else {
			radio = elm_radio_add(group);
			elm_radio_group_add(radio, group);
		}

		if (!radio)
			continue;

		snprintf(buff, sizeof(buff), "<font color=#e1e1e1>%s</font>", preftypes[i].caption);
		elm_object_text_set(radio, buff);

		evas_object_smart_callback_add(radio, "changed", _type_selected_cb, (void *)&preftypes[i]);
		elm_radio_state_value_set(radio, i);
		evas_object_size_hint_align_set(radio, EVAS_HINT_FILL, EVAS_HINT_FILL);
		elm_box_pack_end(box, radio);
		evas_object_show(radio);
	}

	elm_object_part_content_set(parent, target_part_name, box);

	return box;
}

/**
 * @brief Internal function which finds an genlist's item based on the provided preference key name.
 * All the genlist items are stored internally in a Eina_List object. Each genlist item
 * has a key-value structure attached in the data area. The search is performed by iteration
 * over all the list items, stored internally, and verification if attached key-value structure
 * contains the preference key name being searched.
 * @param[in] key The preference key name to find.
 * @param[out] item The item found by key name.
 * @return This function returns 'true' if the item is found,
 * otherwise 'false' is returned.
 */
static bool _find_genlist_item(const char *key, Elm_Object_Item **item)
{
	key_value_t *kv = NULL;
	Eina_List *l;

	EINA_LIST_FOREACH(s_info.genlist_items, l, *item) {
		if (!(*item))
			continue;

		kv = elm_object_item_data_get(*item);
		if (!kv)
			continue;

		if (strncmp(kv->key, key, strlen(kv->key)) == 0 &&
			strlen(kv->key) == strlen(key))
			return true;
	}

	return false;
}
