/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <Elementary.h>
#include <efl_extension.h>
#include <math.h>
#include <widget_viewer_evas.h>
#include <widget_errno.h>
#include "view.h"
#include "data.h"
#include "$(appName).h"
#include "view_defines.h"

#define MAIN_EDJ "edje/main.edj"
#define WIDGET_CONTAINER_SIZE 100
#define WIDGET_SIZES_COUNT 13
#define WIDGETS_PER_PAGE 7
#define LONGPRESS_TIMEOUT 2.0

static struct view_info {
	Evas_Object *win;
	Evas_Object *layout;
	Evas_Object *widget;
	Ecore_Timer *longpress_timer;
	int w;
	int h;
	int widget_selected;
	int total_widgets;
	int current_page;
	bool is_widget_displayed;
} s_info = {
	.win = NULL,
	.layout = NULL,
	.widget = NULL,
	.longpress_timer = NULL,
	.w = 0,
	.h = 0,
	.widget_selected = 0,
	.total_widgets = 0,
	.current_page = 0,
	.is_widget_displayed = false,
};

static void _win_delete_request_cb(void *data , Evas_Object *obj , void *event_info);
static void _win_hide_request_cb(void *data, Evas_Object *obj, void *event_info);
static char *_create_resource_path(const char *file_name);
static bool _create_widget_view(void);
static Eina_Bool _rotary_handler_cb(void *data, Eext_Rotary_Event_Info *ev);
static void _edje_debug_message_cb(void *data, Evas_Object *obj, Edje_Message_Type type, int id, void *msg);
static void _create_page_icons(Evas_Object *layout);
static void _fill_page_icons(int page);
static void _set_higlighted_widget(void);
static void _widget_name_clicked_cb(void *data, Evas_Object *obj, const char *emission, const char *source);
static void _widget_page_change_btn_clicked_cb(void *data, Evas_Object *obj, const char *emission, const char *source);
static void _show_widget_evas(int index);
static bool _widget_viewer_init(void);
static Eina_Bool _longpress_timer_cb(void *data);
static void _set_view_mode(void);

/*
 * @brief Create Essential Object window, conformant and layout
 */
Eina_Bool view_create(void)
{
	/* Create window */
	s_info.win = view_create_win(PACKAGE);
	if (s_info.win == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create a window.");
		return EINA_FALSE;
	}

	elm_win_screen_size_get(s_info.win, NULL, NULL, &s_info.w, &s_info.h);

	if (!_create_widget_view())
		return EINA_FALSE;

	if (!_widget_viewer_init())
		return EINA_FALSE;

	eext_object_event_callback_add(s_info.win, EEXT_CALLBACK_BACK, _win_hide_request_cb, NULL);
	eext_rotary_event_handler_add(_rotary_handler_cb, NULL);

	/* Show window after main view is set up */
	evas_object_show(s_info.win);

	return EINA_TRUE;
}

/*
 * @brief Make a basic window named package
 * @param[in] pkg_name Name of the window
 */
Evas_Object *view_create_win(const char *pkg_name)
{
	Evas_Object *win = NULL;

	/*
	 * Window
	 * Create and initialize elm_win.
	 * elm_win is mandatory to manipulate window
	 */
	win = elm_win_util_standard_add(pkg_name, pkg_name);
	elm_win_conformant_set(win, EINA_TRUE);
	elm_win_autodel_set(win, EINA_TRUE);

	/* Rotation setting */
	if (elm_win_wm_rotation_supported_get(win)) {
		int rots[4] = { 0, 90, 180, 270 };
		elm_win_wm_rotation_available_rotations_set(win, (const int *)(&rots), 4);
	}

	evas_object_smart_callback_add(win, "delete,request", _win_delete_request_cb, NULL);

	return win;
}

/*
 * @brief Makes and sets a layout to the part
 * @param[parent]: Object to which you want to set this layout
 * @param[file_path]: File path of EDJ will be used
 * @param[group_name]: Group name in EDJ that you want to set to layout
 * @param[part_name]: Part name to which you want to set this layout
 */
Evas_Object *view_create_layout_for_part(Evas_Object *parent, char *file_path, char *group_name, char *part_name)
{
	Evas_Object *layout = NULL;

	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "parent is NULL.");
		return NULL;
	}

	layout = elm_layout_add(parent);
	elm_layout_file_set(layout, file_path, group_name);
	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	evas_object_show(layout);

	elm_object_part_content_set(parent, part_name, layout);

	return layout;
}

/*
 * @brief Destroys window and frees its resources.
 */
void view_destroy(void)
{
	widget_viewer_evas_fini();

	if (s_info.win == NULL)
		return;

	evas_object_del(s_info.win);
	widget_viewer_evas_fini();
}

/**
 * @brief Initialize the widget viewer
 * @return true on success, false on fail
 */
bool _widget_viewer_init(void)
{
	if (widget_viewer_evas_init(s_info.win) != WIDGET_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to initialize widget viewer.");
		return false;
	}

	return true;
}

/*
 * @brief Internal callback function to be invoked on main window close.
 * @param[in] data Custom data passed to the callback attachment function.
 * @param[in] obj Object to which the handler is attached.
 * @param[in] event_info The information on the event occurred.
 */
static void _win_delete_request_cb(void *data , Evas_Object *obj , void *event_info)
{
	ui_app_exit();
}

/*
 * @brief Internal callback function to be invoked on main window hide.
 * @param[in] data Custom data passed to the callback attachment function.
 * @param[in] obj Object to which the handler is attached.
 * @param[in] event_info The information on the event occurred.
 */
static void _win_hide_request_cb(void *data, Evas_Object *obj, void *event_info)
{
	if (s_info.is_widget_displayed) {
		s_info.is_widget_displayed = false;
		evas_object_del(s_info.widget);
		s_info.widget = NULL;
	} else {
		elm_win_lower(s_info.win);
	}
}

/*
 * @brief Creates path to the given resource file by concatenation of the basic resource path and the given file_name.
 * @param[in] file_name File name or path relative to the resource directory.
 * @return: The absolute path to the resource with given file_name is returned.
 */
static char *_create_resource_path(const char *file_name)
{
	static char res_path_buff[PATH_MAX] = {0,};
	char *res_path = NULL;

	res_path = app_get_resource_path();
	if (res_path == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to get resource path.");
		return NULL;
	}

	snprintf(res_path_buff, PATH_MAX, "%s%s", res_path, file_name);
	free(res_path);

	return &res_path_buff[0];
}

/*
 * @brief Creates the base layout.
 * @return: The Evas_Object of the layout created.
 */
static Evas_Object *_create_layout(void)
{
	char *edj_path = NULL;

	edj_path = _create_resource_path(MAIN_EDJ);

	Evas_Object *layout = view_create_layout_for_part(s_info.win, edj_path, MAIN_GRP, "default");
	if (!layout)
		return NULL;

	evas_object_size_hint_align_set(layout, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_size_hint_min_set(layout, s_info.w, s_info.h);
	elm_win_resize_object_add(s_info.win, layout);
	elm_layout_signal_callback_add(layout, SIGNAL_NAME_CLICKED, "", _widget_name_clicked_cb, NULL);

	elm_layout_signal_callback_add(layout, SIGNAL_NEXT_CLICKED, "", _widget_page_change_btn_clicked_cb, (void *)true);
	elm_layout_signal_callback_add(layout, SIGNAL_PREV_CLICKED, "", _widget_page_change_btn_clicked_cb, (void *)false);

	s_info.total_widgets = data_get_widget_count();
	_create_page_icons(layout);

	evas_object_show(layout);

	return layout;
}

/**
 * @brief Set the main layout to display the first page of the widget list
 */
static void _initialize_pages(void)
{
	Edje_Message_Int msg = {0,};

	msg.val = ceil((float)s_info.total_widgets / WIDGETS_PER_PAGE);
	edje_object_message_send(elm_layout_edje_get(s_info.layout), EDJE_MESSAGE_INT, MSG_ID_PAGE_COUNT, &msg);

	msg.val = 0;
	edje_object_message_send(elm_layout_edje_get(s_info.layout), EDJE_MESSAGE_INT, MSG_ID_PAGE_SET, &msg);

	_fill_page_icons(0);
	_set_higlighted_widget();
	_set_view_mode();
}

/*
 * @brief Creates the application's layout with widgets preview container.
 * @return: The Evas_Object of the layout created.
 */
static bool _create_widget_view(void)
{
	s_info.layout = _create_layout();
	if (!s_info.layout) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create widgets view layout.");
		return false;
	}

	_initialize_pages();
	edje_object_message_handler_set(elm_layout_edje_get(s_info.layout), _edje_debug_message_cb, NULL);

	return true;
}

/**
 * @brief Callback invoked when a widget icon is pressed
 * @param data User data
 * @param e The evas canvas where the icon is drawn
 * @param obj The icon object
 * @param event_info Event information
 */
static void _icon_pressed_cb(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	int icon_selected = (int)data;
	s_info.widget_selected = s_info.current_page * WIDGETS_PER_PAGE + icon_selected;
	_set_higlighted_widget();

	s_info.longpress_timer = ecore_timer_add(LONGPRESS_TIMEOUT, _longpress_timer_cb, NULL);
}

/**
 * @brief Callback invoked when an widget icon is released
 * @param data User data
 * @param e The evas canvas where the icon is drawn
 * @param obj The icon
 * @param event_info Event info
 */
static void _icon_released_cb(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	if (s_info.longpress_timer) {
		ecore_timer_del(s_info.longpress_timer);
		s_info.longpress_timer = NULL;
	}
}

/**
 * @brief Creates an image object for every swallow in the main layout.
 * @param layout The parent layout.
 */
static void _create_page_icons(Evas_Object *layout)
{
	int i = 0;
	char swallow_name[MAXNAMLEN] = {0,};
	Evas_Object *icon = NULL;

	for (i = 0; i < WIDGETS_PER_PAGE; ++i) {
		icon = elm_icon_add(layout);
		if (!icon) {
			dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] icon == NULL", __FILE__, __LINE__);
			continue;
		}

		evas_object_size_hint_weight_set(icon, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
		evas_object_size_hint_align_set(icon, EVAS_HINT_FILL, EVAS_HINT_FILL);

		snprintf(swallow_name, MAXNAMLEN, "%s%d", PART_ICON_NAME, i);
		if (!elm_layout_content_set(layout, swallow_name, icon)) {
			dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] Failed to set icon as swallow: %s", __FILE__, __LINE__, swallow_name);
			continue;
		}

		evas_object_event_callback_add(icon, EVAS_CALLBACK_MOUSE_DOWN, _icon_pressed_cb, (void *)i);
		evas_object_event_callback_add(icon, EVAS_CALLBACK_MOUSE_UP, _icon_released_cb, NULL);
	}
}

/**
 * @brief Fills the layout with icons based on the current page
 * @param page Current page
 */
static void _fill_page_icons(int page)
{
	int widget_index = 0;
	int i = 0;
	char swallow_name[MAXNAMLEN];
	Evas_Object *icon = NULL;
	char *icon_name = NULL;

	for (i = 0; i < WIDGETS_PER_PAGE; ++i) {
		widget_index = page * WIDGETS_PER_PAGE + i;

		snprintf(swallow_name, MAXNAMLEN, "%s%d", PART_ICON_NAME, i);
		icon = elm_layout_content_get(s_info.layout, swallow_name);

		if (widget_index >= s_info.total_widgets) {
			elm_image_file_set(icon, _create_resource_path(IMAGE_ICON_BG), NULL);
		} else {
			icon_name = data_get_icon(widget_index);
			if (icon_name) {
				elm_image_file_set(icon, icon_name, NULL);
				free(icon_name);
			}
		}
	}
}

/**
 * @brief Sets the current page and sets the widget icons.
 * @param new_page - Page to switch the main view to.
 */
void _set_page(int new_page)
{
	 Edje_Message_Int msg = {0,};

	if (new_page == s_info.current_page)
		return;

	s_info.current_page = new_page;

	msg.val = new_page;
	edje_object_message_send(elm_layout_edje_get(s_info.layout), EDJE_MESSAGE_INT, MSG_ID_PAGE_SET, &msg);
	_fill_page_icons(new_page);
}

/**
 * @brief Sets the current page in the main view based on the selected widget.
 */
static void _page_changed(void)
{
	int new_page = floor((float) s_info.widget_selected / WIDGETS_PER_PAGE);
	_set_page(new_page);
}

/**
 * @brief Sets the selected widget
 * @param ev Rotary event info. Used to determine which widget should be selected as the current one.
 */
static void _set_selected_widget(Eext_Rotary_Event_Info *ev)
{
	if (ev->direction == EEXT_ROTARY_DIRECTION_CLOCKWISE)
		s_info.widget_selected++;
	else
		s_info.widget_selected--;

	if (s_info.widget_selected >= s_info.total_widgets)
		s_info.widget_selected = 0;
	else if (s_info.widget_selected < 0)
		s_info.widget_selected = s_info.total_widgets - 1;
}

/**
 * @brief Sets the highlight position and the displayed widget name.
 */
static void _set_higlighted_widget(void)
{
	char *widget_name = NULL;
	Edje_Message_Int msg = {0,};
	int widget_on_page = s_info.widget_selected % WIDGETS_PER_PAGE;

	msg.val = widget_on_page;
	edje_object_message_send(elm_layout_edje_get(s_info.layout), EDJE_MESSAGE_INT, MSG_ID_WIDGET_SELECTED_SET, &msg);

	widget_name = data_get_name(s_info.widget_selected);
	elm_layout_text_set(s_info.layout, PART_WIDGET_NAME, widget_name);
	free(widget_name);
}

/**
 * @brief Callback invoked when the rotary is used
 * @param data User data
 * @param ev event info
 * @return Check the 'eext_rotary_event_handler_add()' description.
 */
static Eina_Bool _rotary_handler_cb(void *data, Eext_Rotary_Event_Info *ev)
{
	_set_selected_widget(ev);
	_page_changed();
	_set_higlighted_widget();

	return EINA_TRUE;
}

/**
 * @brief Callback invoked to display the edje debug messages
 * @param data User data
 * @param obj The layout object
 * @param type The message type
 * @param id The message id
 * @param msg The message content
 */
static void _edje_debug_message_cb(void *data, Evas_Object *obj, Edje_Message_Type type, int id, void *msg)
{
	Edje_Message_String *msg_str = NULL;

	if (type != EDJE_MESSAGE_STRING || id != MSG_ID_DEBUG)
		return;

	msg_str = (Edje_Message_String *)msg;
	dlog_print(DLOG_INFO, LOG_TAG, "[EDJE] %s", msg_str->str);
}

/**
 * @brief Callback invoked when the icon name is clicked
 * @param data User data
 * @param obj The layout
 * @param emission The emitted signal
 * @param source The emission source
 */
static void _widget_name_clicked_cb(void *data, Evas_Object *obj, const char *emission, const char *source)
{
	_show_widget_evas(s_info.widget_selected);
}

/**
 * @brief Callback invoked when the page next / page previous buttons are clicked
 * @param data The user data. Set to true when the user clicks the next button and to false when the user clicks the previous button
 * @param obj The layout object
 * @param emission The emitted signal
 * @param source  The signal's source
 */
static void _widget_page_change_btn_clicked_cb(void *data, Evas_Object *obj, const char *emission, const char *source)
{
	bool is_next = (bool)data;
	int last_page = s_info.total_widgets / WIDGETS_PER_PAGE;
	int new_page = s_info.current_page;

	if (s_info.total_widgets <= WIDGETS_PER_PAGE)
		return;

	if (!is_next) {
		if (new_page == 0) {
			new_page = last_page;
			s_info.widget_selected = s_info.total_widgets - 1;
		} else {
			new_page--;
			s_info.widget_selected = (new_page + 1) * WIDGETS_PER_PAGE - 1;
		}
	} else {
		if (new_page == last_page)
			new_page = 0;
		else
			new_page++;

		s_info.widget_selected = new_page * WIDGETS_PER_PAGE;
	}


	_set_page(new_page);
	_set_higlighted_widget();
}

/**
 * @brief Displays the selected widget
 * @param index The selected widget's index
 */
static void _show_widget_evas(int index)
{
	char *widget_id = data_get_widget_id(index);
	if (!widget_id) {
		dlog_print(DLOG_WARN, LOG_TAG, "[%s:%d] widget_id == NULL", __FILE__, __LINE__);
		return;
	}

	s_info.widget = widget_viewer_evas_add_widget(s_info.layout, widget_id, NULL, WIDGET_VIEWER_EVAS_DEFAULT_PERIOD);
	if (!s_info.widget) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] widget == NULL; error message: %s", __FILE__, __LINE__, get_error_message(get_last_result()));
		return;
	}

	s_info.is_widget_displayed = true;
	evas_object_resize(s_info.widget, s_info.w, s_info.h);
	evas_object_show(s_info.widget);
}

/**
 * @brief Callback invoked when the user long press a icon
 * @param data User data
 * @return True - repeat the countdown. False - end the countdown.
 */
static Eina_Bool _longpress_timer_cb(void *data)
{
	s_info.longpress_timer = NULL;
	_show_widget_evas(s_info.widget_selected);

	return ECORE_CALLBACK_CANCEL;
}

/**
 * @brief Sets the main view display mode based on the widget count
 */
static void _set_view_mode(void)
{
	if (s_info.total_widgets == 0) {
		elm_layout_signal_emit(s_info.layout, SIGNAL_MODE_EMPTY, "");
		elm_layout_text_set(s_info.layout, PART_WIDGET_NAME, "No widgets found");
	} else if (s_info.total_widgets <= WIDGETS_PER_PAGE) {
		elm_layout_signal_emit(s_info.layout, SIGNAL_MODE_SINGLE_PAGE, "");
	}
}
