/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <Elementary.h>
#include <widget_errno.h>
#include <widget_service.h>
#include <app_manager.h>
#include "$(appName).h"
#include "data.h"

typedef struct widget_data_s {
	char *pkg_id;
	char *widget_id;
} widget_data;

static struct data_info {
	Eina_List *installed_widgets;
} s_info = {
	.installed_widgets = NULL,
};

static int _widget_list_cb(const char *pkgid, const char *widget_id, int is_prime, void *data);

/*
 * @brief Obtains all the available widgets and adds them to the options view.
 * @return: If the function succeeds, 'true' is returned,
 * otherwise 'false' is returned.
 */
bool data_refresh_widgets(void)
{
	int ret;
	widget_data *data;

	EINA_LIST_FREE(s_info.installed_widgets, data) {
		free(data->pkg_id);
		free(data->widget_id);
	}

	ret = widget_service_get_widget_list(_widget_list_cb, NULL);
	if (ret != WIDGET_ERROR_NONE)
		dlog_print(DLOG_WARN, LOG_TAG, "Error obtaining widgets list or no widgets installed: %s", get_error_message(ret));

	return (ret == WIDGET_ERROR_NONE);
}

/**
 * @brief Gets the count of available widgets.
 * @return The widget count.
 */
int data_get_widget_count(void)
{
	return eina_list_count(s_info.installed_widgets);
}

/**
 * @brief Gets the path to the widget's icon. If the widget does not provide an icon,
 * the widget's application icon is acquired.
 * @param index The widget's index in the widgets list.
 * @return Icon's path or NULL on error.
 */
char *data_get_icon(int index)
{
	int ret;
	char *icon_path;
	app_info_h app_info = NULL;
	widget_data *wd = eina_list_nth(s_info.installed_widgets, index);
	if (!wd) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] wd == NULL. Index: %d", __FILE__, __LINE__, index);
		return NULL;
	}

	icon_path = widget_service_get_icon(wd->pkg_id, NULL);
	if (!icon_path) {
		ret = app_info_create(wd->pkg_id, &app_info);
		if (ret != APP_MANAGER_ERROR_NONE) {
			dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] function app_info_create() failed.", __FILE__, __LINE__);
			return NULL;
		}

		ret = app_info_get_icon(app_info, &icon_path);
		if (ret != APP_MANAGER_ERROR_NONE)
			dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] function app_info_get_icon() failed.", __FILE__, __LINE__);

		app_info_destroy(app_info);
	}

	if (!icon_path)
		return NULL;

	return strdup(icon_path);
}

/**
 * @brief Gets the name of a widget with the given index.
 * @param index Widget's index.
 * @return A string containing the widget's name.
 */
char *data_get_name(int index)
{
	char *name;
	widget_data *wd = eina_list_nth(s_info.installed_widgets, index);
	if (!wd) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] wd == NULL", __FILE__, __LINE__);
		return NULL;
	}

	name = widget_service_get_name(wd->widget_id, NULL);

	if (!name)
		return NULL;

	return strdup(name);
}

/**
 * @brief Gets the widget's id based on the widget's index in the widget list.
 * @param index The widget's index.
 * @return The widget's id or NULL on error.
 */
char *data_get_widget_id(int index)
{
	widget_data *wd = eina_list_nth(s_info.installed_widgets, index);
	if (!wd) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] wd == NULL", __FILE__, __LINE__);
		return NULL;
	}

	return wd->widget_id;
}

/**
 * @brief Creates a widget_data structure structure based on the widget's pkg_id and id.
 * @param pkg_id Widget's package id.
 * @param widget_id Widget's id.
 * @return The created structure or NULL on error.
 */
static widget_data *_create_widget_info(const char *pkg_id, const char *widget_id)
{
	widget_data *wd;

	if (!pkg_id || !widget_id) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] invalid input argument.", __FILE__, __LINE__);
		return NULL;
	}

	wd = (widget_data *)calloc(1, sizeof(widget_data));
	if (!wd) {
		dlog_print(DLOG_ERROR, LOG_TAG, "[%s:%d] failed to allocate memory for widget data.", __FILE__, __LINE__);
		return NULL;
	}

	wd->pkg_id = strdup(pkg_id);
	wd->widget_id = strdup(widget_id);

	return wd;
}

/*
 * @brief Internal callback function to be invoked on available widgets information acquisition.
 * @param[in] pkgid The identifier of the package containing the acquired widget.
 * @param[in] widget_id The identifier of the acquired widget.
 * @param[in] is_prime The flag denoting prime widget.
 * @param[in] data The user data passed to the widget_service_get_widget_list() function
 * @return: If the function succeeds, the WIDGET_ERROR_NONE is returned,
 * otherwise error code is returned.
 */
static int _widget_list_cb(const char *pkgid, const char *widget_id, int is_prime, void *data)
{
	widget_data *wgt;

	if (widget_service_get_nodisplay(widget_id))
		return WIDGET_ERROR_NONE;

	wgt = _create_widget_info(pkgid, widget_id);
	if (wgt)
		s_info.installed_widgets = eina_list_append(s_info.installed_widgets, wgt);

	return WIDGET_ERROR_NONE;
}
