/*
* Copyright (c) 2011 Samsung Electronics Co., Ltd All Rights Reserved
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

#include <helper.h>
#include <log.h>
#include <view.h>

bool _playlist_item_cb(const char *index, mc_metadata_h metadata, void *user_data);

/*
 * @brief Function to deallocate.
 * @param[in] data The data to deallocate
 */
void _free_list(gpointer data)
{
	SAFE_FREE(data);
}

/*
 * @brief Function to deallocate playlist.
 * @param[in] data The playlist structure to deallocate
 */
void _free_playlist_list(gpointer data)
{
	unsigned int i = 0;
	playlist_s *item = (playlist_s *)data;

	SAFE_FREE(item->index);
	for (i = 0; i < LAST_META_ITEM; i++)
		SAFE_FREE(item->metadata[i]);

	SAFE_FREE(item);
}

/*
 * @brief Called when updating status of the media controller server.
 * @param[in] server_name The app_id of the media controller server which sent the notification.
 * @param[in] state The state of the updated media controller server
 * @param[in] user_data        The user data passed from the mc_client_set_server_updated_cb() function
 */
void _server_updated_cb(const char *server_name, mc_server_state_e state, void *user_data)
{
	appdata_s *ad = (appdata_s *)user_data;
	unsigned int i = 0;
	char *server = NULL;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	DEBUG_ENTER();

	/* get server applications list using mc_client_foreach_server() */
	mcs_get_server_list(ad);

	/* if server application is activated, update views for server application */
	if (state == MC_SERVER_STATE_ACTIVATE) {
		if (ad->mc_data.server_info.server_name == NULL) {
			ad->mc_data.server_info.server_name = g_strdup(server_name);
			mcs_update_view_latest_server(ad);
			mcs_update_view_server_list(ad);
		}
	/* if server application is deactivated, update views */
	} else if (state == MC_SERVER_STATE_DEACTIVATE) {
		/* remove playlist on view */
		for (i = 0; i < ad->mc_data.server_num; i++) {
			server = g_list_nth_data(ad->mc_data.server_list, i);
			if (g_strcmp0(server_name, server) == 0)
				ad->mc_data.server_list = g_list_remove_all(ad->mc_data.server_list, server);
		}
		/* if deactivated server application is viewing currently, change view to first server application */
		if (strcmp(ad->mc_data.server_info.server_name, server_name) == 0) {
			SAFE_FREE(ad->mc_data.server_info.server_name);
			ad->mc_data.server_info.server_name = g_list_nth_data(ad->mc_data.server_list, 0);
		}
		/* update server applications view */
		mcs_update_view_server_list(ad);
		/* update playlist view */
		mcs_remove_playlist_view(ad);
	}

	DEBUG_LEAVE();
}

/*
 * @brief Called when updating the playback information of the media controller server
 * @param[in] server_name The app_id of the media controller server which sent the notification.
 * @param[in] playback The playback information of the updated media controller server.
 * @param[in] user_data        The user data passed from the mc_client_set_playback_updated_cb() function
 */
void _playback_updated_cb(const char *server_name, mc_playback_h playback, void *user_data)
{
	appdata_s *ad = (appdata_s *)user_data;
	int ret = MEDIA_CONTROLLER_ERROR_NONE;
	mc_playback_states_e state = MC_PLAYBACK_STATE_NONE;
	unsigned long long position = 0;
	char *playlist_name = NULL;
	char *playlist_index = NULL;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	DEBUG_ENTER();

	if (strcmp(ad->mc_data.server_info.server_name, server_name) != 0) {
		LOG_ERR("No current server");
		DEBUG_LEAVE();
		return;
	}

	SAFE_FREE(ad->mc_data.server_info.playlist_index);

	/* get the playback state from mc_playback_h */
	ret = mc_client_get_playback_state(playback, &state);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_get_playback_state is failed (%d)", ret);

	/* get the playback position from mc_playback_h */
	ret = mc_client_get_playback_position(playback, &position);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_get_playback_position is failed (%d)", ret);

	/* get the playlist name and the playlist index from mc_playback_h */
	ret = mc_client_get_playlist_item_info(playback, &playlist_name, &playlist_index);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_get_playback_position is failed (%d)", ret);

	/* update the playback information to the appdata */
	ad->mc_data.server_info.playback_state = state;
	ad->mc_data.server_info.playback_position = position;
	ad->mc_data.server_info.playlist_name = g_strdup(playlist_name);
	ad->mc_data.server_info.playlist_index = g_strdup(playlist_index);
	/* update playback view */
	mcs_update_view_playback(ad);

	SAFE_FREE(playlist_name);
	SAFE_FREE(playlist_index);

	DEBUG_LEAVE();
}

/*
 * @brief Called when updating the metadata of the media controller server.
 * @param[in] server_name The app_id of the media controller server which sent the notification.
 * @param[in] metadata The metadata of the updated media controller server.
 * @param[in] user_data        The user data passed from the mc_client_set_metadata_updated_cb() function
 */
void _metadata_updated_cb(const char *server_name, mc_metadata_h metadata, void *user_data)
{
	appdata_s *ad = (appdata_s *)user_data;
	unsigned int i = 0;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	DEBUG_ENTER();

	if (strcmp(ad->mc_data.server_info.server_name, server_name) != 0) {
		LOG_ERR("No current server");
		DEBUG_LEAVE();
		return;
	}

	/* get the metadata information and update it to the appdata */
	for (i = 0; i < LAST_META_ITEM; i++) {
		SAFE_FREE(ad->mc_data.server_info.metadata[i]);
		mc_metadata_get(metadata, i, &ad->mc_data.server_info.metadata[i]);
	}
	/* update metadata view */
	mcs_update_view_metadata(ad);

	DEBUG_LEAVE();
}

/*
 * @brief Called when updating the shuffle mode of the media controller server.
 * @param[in] server_name The app_id of the media controller server which sent the notification.
 * @param[in] mode The shuffle mode of the updated media controller server
 * @param[in] user_data        The user data passed from the mc_client_set_shuffle_mode_updated_cb() function
 */
void _shuffle_updated_cb(const char *server_name, mc_shuffle_mode_e shuffle_mode, void *user_data)
{
	appdata_s *ad = (appdata_s *)user_data;
	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	DEBUG_ENTER();

	if (strcmp(ad->mc_data.server_info.server_name, server_name) != 0) {
		LOG_ERR("No current server");
		DEBUG_LEAVE();
		return;
	}

	ad->mc_data.server_info.shuffle = shuffle_mode;
	/* update playback view */
	mcs_update_view_playback(ad);

	DEBUG_LEAVE();
}

/*
 * @brief Called when updating the repeat mode of the media controller server.
 * @param[in] server_name The app_id of the media controller server which sent the notification.
 * @param[in] mode The repeat mode of the updated media controller server
 * @param[in] user_data        The user data passed from the mc_client_set_repeat_mode_updated_cb() function
 */
void _repeat_updated_cb(const char *server_name, mc_repeat_mode_e repeat_mode, void *user_data)
{
	appdata_s *ad = (appdata_s *)user_data;
	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	DEBUG_ENTER();

	if (strcmp(ad->mc_data.server_info.server_name, server_name) != 0) {
		LOG_ERR("No current server");
		DEBUG_LEAVE();
		return;
	}

	ad->mc_data.server_info.repeat = repeat_mode;
	/* update playback view */
	mcs_update_view_playback(ad);

	DEBUG_LEAVE();
}

/*
 * @brief Called when updating the playlist of the media controller server.
 * @param[in] server_name The app_id of the media controller server which sent the notification.
 * @param[in] mode The playlist update mode of the updated media controller server
 * @param[in] playlist_name Updated playlist name.
 * @param[in] playlist        The handle of the media controller playlist.
 * @param[in] user_data The user data passed from the mc_client_set_playlist_updated_cb() function
 */
void _playlist_updated_cb(const char *server_name, mc_playlist_update_mode_e mode, const char *playlist_name, mc_playlist_h playlist, void *user_data)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;
	appdata_s *ad = (appdata_s *)user_data;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	DEBUG_ENTER();

	if (strcmp(ad->mc_data.server_info.server_name, server_name) != 0) {
		LOG_ERR("No current server");
		DEBUG_LEAVE();
		return;
	}

	/* update playlist */
	if (mode == MC_PLAYLIST_UPDATED) {
		LOG_INFO("playlist is updated.");
		/* get playlist information */
		SAFE_FREE(ad->mc_data.server_info.playlist_name);
		ad->mc_data.server_info.playlist_name = g_strdup(playlist_name);

		/* remove old playlist */
		if (ad->mc_data.server_info.playlist) {
			g_list_free_full(ad->mc_data.server_info.playlist, _free_playlist_list);
			ad->mc_data.server_info.playlist = NULL;
		}

		ret = mc_playlist_foreach_item(playlist, _playlist_item_cb, ad);
		RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to get playlist.(%d)", ret);
	/* remove playlist */
	} else if (mode == MC_PLAYLIST_REMOVED) {
		LOG_INFO("playlist is removed.");
	/* invalid update mode */
	} else {
		LOG_ERR("Invalid update mode");
	}

	/* update playlist view */
	mcs_update_playlist_view(ad);

	DEBUG_LEAVE();
}

int _compare_str(const void *user_data, const void *data)
{
	return g_strcmp0((char *)user_data, (char *)data);
}

/*
 * @brief Called when the result of the command from the server is received.
 * @param[in] server_name    The app_id of the media controller server which sent the reply
 * @param[in] request_id     The id of the command request
 * @param[in] result_code    The result code of the action
 * @param[in] data The extra data
 * @param[in] user_data        The user data passed from the mc_client_set_cmd_reply_received_cb() function
 */
void _cmd_reply_received_cb(const char *server_name, const char *request_id, int result_code, bundle *data, void *user_data)
{
	appdata_s *ad = (appdata_s *)user_data;
	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	DEBUG_ENTER();

	if (strcmp(ad->mc_data.server_info.server_name, server_name) != 0) {
		LOG_ERR("No current server");
		DEBUG_LEAVE();
		return;
	}

	/* check valid request_id for my requests */
	if (g_list_find_custom(ad->mc_data.req_ids, request_id, _compare_str) == NULL) {
		LOG_ERR("Invalid request_id(%s) was received from %s", request_id, server_name);
		return;
	}

	ad->mc_data.req_ids = g_list_remove(ad->mc_data.req_ids, request_id);
	ad->mc_data.cmd_result = result_code;
	/* update custom command reply view on UI */
	mcs_update_view_command_reply(ad);

	DEBUG_LEAVE();
}

/*
 * @brief Called when receiving custom event of media controller servers.
 * @param[in] server_name The app_id of the subscribed media controller server.
 * @param[in] request_id     The id of the custom event request
 * @param[in] event          The name of the custom event
 * @param[in] data           The data can include other information associated with the event
 * @param[in] user_data        The user data passed from the mc_client_set_custom_event_received_cb() function
 */
void _custom_event_received_cb(const char *server_name, const char *request_id, const char *event_name, bundle *data, void *user_data)
{
	int result_code = MEDIA_CONTROLLER_ERROR_NONE;
	appdata_s *ad = user_data;
	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	DEBUG_ENTER();

	if (strcmp(ad->mc_data.server_info.server_name, server_name) != 0) {
		LOG_ERR("No current server");
		DEBUG_LEAVE();
		return;
	}

	/* do something for custom command */
	SAFE_FREE(ad->mc_data.custom_event);
	ad->mc_data.custom_event = g_strdup(event_name);
	ad->mc_data.custom_event_result = result_code;

	if (result_code == MEDIA_CONTROLLER_ERROR_NONE) {
		/* update custom event view on UI */
		mcs_update_view_custom_event(ad);
	}
	/* send the event reply to the server application */
	mcs_send_event_reply(ad, server_name, request_id, result_code);

	DEBUG_LEAVE();
}

/*
 * @brief Called when requesting the list of created servers.
 * @param[in] server_name The app_id of the created media controller server.
 * @param[in] user_data        The user data passed from the mc_client_foreach_server() function
 */
bool _activated_server_cb(const char *server_name, void *user_data)
{
	GList **server_list = (GList **)user_data;

	if (server_name != NULL) {
		if (g_list_find_custom(*server_list, server_name, _compare_str))
			return true;

		*server_list = g_list_append(*server_list, g_strdup(server_name));
	}

	return true;
}

/*
 * @brief Called for every playlist item in the playlist.
 * @param[in] index The ID of the playlist member.
 * @param[in] metadata The handle to metadata of the playlist item.
 * @param[in] user_data The user data passed from the foreach function
 */
bool _playlist_item_cb(const char *index, mc_metadata_h metadata, void *user_data)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;
	unsigned int i = 0;

	appdata_s *ad = (appdata_s *)user_data;
	/* check valid appdata */
	RETVM_IF((ad == NULL), false, "Invalid appdata_s");

	DEBUG_ENTER();

	playlist_s *item = (playlist_s *)calloc(1, sizeof(playlist_s));
	if (item == NULL)
		return false;

	/* get metadata information for playlist item */
	item->index = g_strdup(index);
	for (i = 0; i < LAST_META_ITEM; i++) {
		ret = mc_metadata_get(metadata, i, &item->metadata[i]);
		RETVM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), false, "Fail to get %s metadata.(%d)", label_metadata[i], ret);
	}

	/* add a playlist item to appdata */
	ad->mc_data.server_info.playlist = g_list_append(ad->mc_data.server_info.playlist, item);

	DEBUG_LEAVE();

	return true;
}

/*
 * @brief Called for every playlist.
 * @param[in] playlist        The handle of the media controller playlist.
 * @param[in] user_data The user data passed from the foreach function
 */
bool _playlist_cb(mc_playlist_h playlist, void *user_data)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;
	appdata_s *ad = (appdata_s *)user_data;
	/* check valid appdata */
	RETVM_IF((ad == NULL), false, "Invalid appdata_s");

	DEBUG_ENTER();

	SAFE_FREE(ad->mc_data.server_info.playlist_name);

	/* remove old playlist from appdata */
	if (ad->mc_data.server_info.playlist) {
		g_list_free_full(ad->mc_data.server_info.playlist, _free_playlist_list);
		ad->mc_data.server_info.playlist = NULL;
	}

	/* get playlist name */
	ret = mc_playlist_get_name(playlist, &ad->mc_data.server_info.playlist_name);
	RETVM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), false, "Fail to get playlist name.(%d)", ret);

	/* get playlist items */
	ret = mc_playlist_foreach_item(playlist, _playlist_item_cb, ad);
	RETVM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), false, "Fail to get playlist.(%d)", ret);

	DEBUG_LEAVE();

	return false;
}

/*
 * @brief Creates a media controller client.
 * @param[out] ad The structure of app data include the handle of the media controller client
 */
void mcs_create_client(appdata_s *ad)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	/* create media controller handle, it will be kept during running application */
	ret = mc_client_create((mc_client_h *)&ad->mc_data.handle);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to create server.(%d)", ret);
}

/*
 * @brief Destroys a media controller client.
 * @param[in] ad The structure of app data
 */
void mcs_destroy_client(appdata_s *ad)
{
	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");
	mc_client_h handle = (mc_client_h)ad->mc_data.handle;
	/* check valid media controller handle */
	RETM_IF((handle == NULL), "Invalid handle");

	/* free playlist for _playlist_item_cb */
	g_list_free_full(ad->mc_data.server_info.playlist, _free_playlist_list);
	/* free client_name for _activated_client_cb */
	g_list_free_full(ad->mc_data.server_list, _free_list);
	/* free requst_id for  */
	g_list_free_full(ad->mc_data.req_ids, _free_list);

	/* destroy media controller handle */
	mc_client_destroy(handle);
}

/*
 * @brief Function to set callback to use
 * @param[in] ad The structure of app data
 */
void mcs_set_callback(appdata_s *ad)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");
	mc_client_h handle = (mc_client_h)ad->mc_data.handle;
	/* check valid media controller handle */
	RETM_IF((handle == NULL), "Invalid handle");

	/* register callback */
	/* set the callback to receive application state from server applications */
	ret = mc_client_set_server_updated_cb(handle, _server_updated_cb, ad);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_set_playback_updated_cb is failed (%d)", ret);

	/* set the callback to receive updated playback information from server applications */
	ret = mc_client_set_playback_updated_cb(handle, _playback_updated_cb, ad);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_set_playback_updated_cb is failed (%d)", ret);

	/* set the callback to receive updated metadata information from server applications */
	ret = mc_client_set_metadata_updated_cb(handle, _metadata_updated_cb, ad);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_set_shuffle_mode_updated_cb is failed (%d)", ret);

	/* set the callback to receive updated shuffle mode from server applications */
	ret = mc_client_set_shuffle_mode_updated_cb(handle, _shuffle_updated_cb, ad);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_set_shuffle_mode_updated_cb is failed (%d)", ret);

	/* set the callback to receive updated repeat mode from server applications */
	ret = mc_client_set_repeat_mode_updated_cb(handle, _repeat_updated_cb, ad);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_set_repeat_mode_updated_cb is failed (%d)", ret);

	/* set the callback to receive updated playlist information from server applications */
	ret = mc_client_set_playlist_updated_cb(handle, _playlist_updated_cb, ad);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_set_playlist_updated_cb is failed (%d)", ret);

	/* set the callback to receive command reply from server applications */
	ret = mc_client_set_cmd_reply_received_cb(handle, _cmd_reply_received_cb, ad);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_set_cmd_reply_received_cb is failed (%d)", ret);

	/* set the callback to receive custom event from server applications */
	ret = mc_client_set_custom_event_received_cb(handle, _custom_event_received_cb, ad);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_set_custom_event_received_cb is failed (%d)", ret);
}

/*
 * @brief Function to unset callback
 * @param[in] ad The structure of app data
 */
void mcs_unset_callback(appdata_s *ad)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");
	mc_client_h handle = (mc_client_h)ad->mc_data.handle;
	/* check valid media controller handle */
	RETM_IF((handle == NULL), "Invalid handle");

	/* unregister callback */

	/* unset the callback not to receive playback information from server applications */
	ret = mc_client_unset_playback_updated_cb(handle);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_unset_playback_updated_cb is failed (%d)", ret);

	/* unset the callback not to receive metadata information from server applications */
	ret = mc_client_unset_metadata_updated_cb(handle);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_unset_metadata_updated_cb is failed (%d)", ret);

	/* unset the callback not to receive shuffle mode from server applications */
	ret = mc_client_unset_shuffle_mode_updated_cb(handle);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_unset_shuffle_mode_updated_cb is failed (%d)", ret);

	/* unset the callback not to receive repeat mode from server applications */
	ret = mc_client_unset_repeat_mode_updated_cb(handle);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_unset_repeat_mode_updated_cb is failed (%d)", ret);

	/* unset the callback not to receive playlist information from server applications */
	ret = mc_client_unset_playlist_updated_cb(handle);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_unset_playlist_updated_cb is failed (%d)", ret);

	/* unset the callback not to receive command reply from server applications */
	ret = mc_client_unset_cmd_reply_received_cb(handle);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_unset_cmd_reply_received_cb is failed (%d)", ret);

	/* unset the callback not to receive command event from server applications */
	ret = mc_client_unset_custom_event_received_cb(handle);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_unset_custom_event_received_cb is failed (%d)", ret);
}

/*
 * @brief Gets the latest media controller server info.
 * @param[in] ad The structure of app data
 */
void mcs_get_latest_server(appdata_s *ad)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;
	mc_server_state_e state;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");
	mc_client_h handle = (mc_client_h)ad->mc_data.handle;
	/* check valid media controller handle */
	RETM_IF((handle == NULL), "Invalid handle");

	/* get latest server name */
	ret = mc_client_get_latest_server_info(handle, &ad->mc_data.latest_server_name, &state);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to get latest server information (%d)", ret);
	ad->mc_data.latest_server_state = state;
}

/*
 * @brief Retrieves all created servers.
 * @param[in] ad The structure of app data
 */
void mcs_get_server_list(appdata_s *ad)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");
	mc_client_h handle = (mc_client_h)ad->mc_data.handle;
	/* check valid media controller handle */
	RETM_IF((handle == NULL), "Invalid handle");

	/* get server list for sending custom event */
	ret = mc_client_foreach_server(handle, _activated_server_cb, &ad->mc_data.server_list);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to get client list.(%d)", ret);

	if (ad->mc_data.server_list)
		ad->mc_data.server_num = g_list_length(ad->mc_data.server_list);
}

/*
 * @brief Gets server information.
 * @param[in] ad The structure of app data
 * @param[in] server_name    The app_id of the media controller server
 */
void mcs_get_information(appdata_s *ad, const char *server_name)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;
	mc_playback_h playback = NULL;
	mc_metadata_h metadata = NULL;
	unsigned int i = 0;

	DEBUG_ENTER();

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");
	RETM_IF((server_name == NULL), "Invalid server_name");
	mc_client_h handle = (mc_client_h)ad->mc_data.handle;
	/* check valid media controller handle */
	RETM_IF((handle == NULL), "Invalid handle");

	/* get information for server application */

	/* get playback information handle */
	ret = mc_client_get_server_playback_info(handle, server_name, &playback);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_get_server_playback_info is failed (%d)", ret);
	/* get playback state information from handle */
	ret = mc_client_get_playback_state(playback, &ad->mc_data.server_info.playback_state);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_get_playback_state is failed (%d)", ret);
	/* get playback position information from handle */
	ret = mc_client_get_playback_position(playback, &ad->mc_data.server_info.playback_position);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_get_playback_position is failed (%d)", ret);

	/* get playlist item information */
	ret = mc_client_get_playlist_item_info(playback, &ad->mc_data.server_info.playlist_name, &ad->mc_data.server_info.playlist_index);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_get_playlist_item_index is failed (%d)", ret);

	/* get shuffle mode information */
	ret = mc_client_get_server_shuffle_mode(handle, server_name, &ad->mc_data.server_info.shuffle);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_get_server_shuffle_mode is failed (%d)", ret);
	/* get repeat mode information */
	ret = mc_client_get_server_repeat_mode(handle, server_name, &ad->mc_data.server_info.repeat);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_get_server_repeat_mode is failed (%d)", ret);

	/* get metadata information */
	ret = mc_client_get_server_metadata(handle, server_name, &metadata);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_get_server_metadata is failed (%d)", ret);

	for (i = 0; i < LAST_META_ITEM; i++) {
		SAFE_FREE(ad->mc_data.server_info.metadata[i]);
		ret = mc_metadata_get(metadata, i, &ad->mc_data.server_info.metadata[i]);
		RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_metadata_get(%d) is failed (%d)", i, ret);
	}

	/* get playlist information */
	ret = mc_client_foreach_server_playlist(handle, server_name, _playlist_cb, ad);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_get_server_metadata is failed (%d)", ret);

	DEBUG_LEAVE();
}

/*
 * @brief Sends the playback action command to server.
 * @param[in] ad The structure of app data
 * @param[in] server_name    The app_id of the media controller server
 * @param[in] action   The playback action command to send to the media controller server
 */
void mcs_send_playback_action(appdata_s *ad, const char *server_name, int action)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;
	char *request_id = NULL;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");
	RETM_IF((server_name == NULL), "Invalid server_name");
	mc_client_h handle = (mc_client_h)ad->mc_data.handle;
	/* check valid media controller handle */
	RETM_IF((handle == NULL), "Invalid handle");

	/* send the playback action command to the server application */
	ret = mc_client_send_playback_action_cmd(handle, server_name, action, &request_id);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_send_playback_action_cmd is failed (%d)", ret);

	if (request_id) {
		LOG_DBG("%s is requested.", request_id);
		ad->mc_data.req_ids = g_list_append(ad->mc_data.req_ids, request_id);
	} else {
		LOG_ERR("The requested id is not created.");
	}
}

/*
 * @brief Sends the playback position command to server.
 * @param[in] ad The structure of app data
 * @param[in] server_name    The app_id of the media controller server
 * @param[in] position   The position of the playback in milliseconds to send to media controller server.
 */
void mcs_send_playback_position(appdata_s *ad, const char *server_name, unsigned long long position)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;
	char *request_id = NULL;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");
	RETM_IF((server_name == NULL), "Invalid server_name");
	mc_client_h handle = (mc_client_h)ad->mc_data.handle;
	/* check valid media controller handle */
	RETM_IF((handle == NULL), "Invalid handle");

	/* send the playback position command to the server application */
	ret = mc_client_send_playback_position_cmd(handle, server_name, position, &request_id);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_send_playback_position_cmd is failed (%d)", ret);

	if (request_id) {
		LOG_DBG("%s is requested.", request_id);
		ad->mc_data.req_ids = g_list_append(ad->mc_data.req_ids, request_id);
	} else {
		LOG_ERR("The requested id is not created.");
	}
}

/*
 * @brief Sends the playlist command to server.
 * @param[in] ad The structure of app data
 * @param[in] server_name    The app_id of the media controller server
 * @param[in] index    The index of the media in playlist to send to the media controller server
 */
void mcs_send_playlist_cmd(appdata_s *ad, const char *server_name, const char *index)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;
	char *request_id = NULL;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");
	RETM_IF((server_name == NULL), "Invalid server_name");
	mc_client_h handle = (mc_client_h)ad->mc_data.handle;
	/* check valid media controller handle */
	RETM_IF((handle == NULL), "Invalid handle");

	/* send the playlist command to the server application */
	ret = mc_client_send_playlist_cmd(handle, server_name, ad->mc_data.server_info.playlist_name, index, MC_PLAYBACK_ACTION_PLAY, 0, &request_id);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_send_repeat_mode_cmd is failed (%d)", ret);

	if (request_id) {
		LOG_DBG("%s is requested.", request_id);
		ad->mc_data.req_ids = g_list_append(ad->mc_data.req_ids, request_id);
	} else {
		LOG_ERR("The requested id is not created.");
	}
}

/*
 * @brief Sends the shuffle mode command to server.
 * @param[in] ad The structure of app data
 * @param[in] server_name    The app_id of the media controller server
 * @param[in] shuffle_mode   The shuffle mode to send to media controller server
 */
void mcs_send_shuffle(appdata_s *ad, const char *server_name, int shuffle_mode)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;
	char *request_id = NULL;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");
	RETM_IF((server_name == NULL), "Invalid server_name");
	mc_client_h handle = (mc_client_h)ad->mc_data.handle;
	/* check valid media controller handle */
	RETM_IF((handle == NULL), "Invalid handle");

	/* send the shuffle mode command to the server application */
	ret = mc_client_send_shuffle_mode_cmd(handle, server_name, shuffle_mode, &request_id);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_send_shuffle_mode_cmd is failed (%d)", ret);

	if (request_id) {
		LOG_DBG("%s is requested.", request_id);
		ad->mc_data.req_ids = g_list_append(ad->mc_data.req_ids, request_id);
	} else {
		LOG_ERR("The requested id is not created.");
	}
}

/*
 * @brief Sends the repeat mode command to server.
 * @param[in] ad The structure of app data
 * @param[in] server_name    The app_id of the media controller server
 * @param[in] repeat_mode   The repeat mode to send to media controller server
 */
void mcs_send_repeat(appdata_s *ad, const char *server_name, int repeat_mode)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;
	char *request_id = NULL;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");
	RETM_IF((server_name == NULL), "Invalid server_name");
	mc_client_h handle = (mc_client_h)ad->mc_data.handle;
	/* check valid media controller handle */
	RETM_IF((handle == NULL), "Invalid handle");

	/* send the repeat mode command to the server application */
	ret = mc_client_send_repeat_mode_cmd(handle, server_name, repeat_mode, &request_id);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_send_repeat_mode_cmd is failed (%d)", ret);

	if (request_id) {
		LOG_DBG("%s is requested.", request_id);
		ad->mc_data.req_ids = g_list_append(ad->mc_data.req_ids, request_id);
	} else {
		LOG_ERR("The requested id is not created.");
	}
}

/*
 * @brief Sends the custom command to server.
 * @param[in] ad The structure of app data
 * @param[in] server_name    The app_id of the media controller server
 * @param[in] command      The command to be sent
 */
void mcs_send_custom_cmd(appdata_s *ad, const char *server_name, const char *command)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;
	char *request_id = NULL;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");
	RETM_IF((server_name == NULL), "Invalid server_name");
	mc_client_h handle = (mc_client_h)ad->mc_data.handle;
	/* check valid media controller handle */
	RETM_IF((handle == NULL), "Invalid handle");

	/* send the custom command to the server application */
	ret = mc_client_send_custom_cmd(handle, server_name, command, NULL, &request_id);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_send_custom_cmd is failed (%d)", ret);

	if (request_id) {
		LOG_DBG("%s is requested.", request_id);
		ad->mc_data.req_ids = g_list_append(ad->mc_data.req_ids, request_id);
	} else {
		LOG_ERR("The requested id is not created.");
	}
}

/*
 * @brief Replies with the result of the requested event to the media controller server.
 * @param[in] ad The structure of app data
 * @param[in] server_name    The app_id of the media controller server
 * @param[in] request_id    The id of the event request, received in the mc_client_set_custom_event_received_cb() function.
 * @param[in] result_code    The result code of custom event
 */
void mcs_send_event_reply(appdata_s *ad, const char *server_name, const char *request_id, int result_code)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");
	RETM_IF((server_name == NULL), "Invalid server_name");
	RETM_IF((request_id == NULL), "Invalid server_name");
	mc_client_h handle = (mc_client_h)ad->mc_data.handle;
	/* check valid media controller handle */
	RETM_IF((handle == NULL), "Invalid handle");

	/* send the event reply to the server application */
	ret = mc_client_send_event_reply(handle, server_name, request_id, result_code, NULL);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "mc_client_send_repeat_mode_cmd is failed (%d)", ret);

	LOG_DBG("The reply of ""%s"" is sent.", request_id);
}
