/*
* Copyright (c) 2011 Samsung Electronics Co., Ltd All Rights Reserved
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

#include "log.h"
#include "helper.h"
#include "view.h"

/*
 * @brief Called when the server receives the playback action command from the client.
 * @param[in] client_name    The app_id of the media controller client.
 * @param[in] request_id    The request_id of the media controller client.
 * @param[in] action    The received playback action
 * @param[in] user_data        The user data passed from the mc_server_set_playback_action_cmd_received_cb() function
 */
void _pb_action_cmd_received_cb(const char *client_name, const char *request_id, mc_playback_action_e action, void *user_data)
{
	int result_code = MEDIA_CONTROLLER_ERROR_NONE;
	appdata_s *ad = user_data;

	DEBUG_ENTER();

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	/* change playback state */
	/*
	 * convert playback_state due to deprecated enum
	 * typedef enum {
	 * MC_PLAYBACK_ACTION_PLAY = 0,
	 * MC_PLAYBACK_ACTION_PAUSE,
	 * MC_PLAYBACK_ACTION_STOP,
	 * MC_PLAYBACK_ACTION_NEXT,
	 * MC_PLAYBACK_ACTION_PREV,
	 * MC_PLAYBACK_ACTION_FAST_FORWARD,
	 * MC_PLAYBACK_ACTION_REWIND,
	 * MC_PLAYBACK_ACTION_TOGGLE_PLAY_PAUSE,
	 * } mc_playback_action_e;
	 */
	switch (action) {
	case MC_PLAYBACK_ACTION_PLAY:
		ad->mc_data.server_info.pb_state = MC_PLAYBACK_STATE_PLAYING;
		break;
	case MC_PLAYBACK_ACTION_PAUSE:
		ad->mc_data.server_info.pb_state = MC_PLAYBACK_STATE_PAUSED;
		break;
	case MC_PLAYBACK_ACTION_STOP:
		ad->mc_data.server_info.pb_state = MC_PLAYBACK_STATE_STOPPED;
		break;
	case MC_PLAYBACK_ACTION_NEXT:
		ad->mc_data.server_info.pb_state = MC_PLAYBACK_STATE_MOVING_TO_NEXT;
		/* if MC_PLAYBACK_ACTION_NEXT is requested, increase index */
		ad->mc_data.server_info.cur_playing_idx++;
		if (ad->mc_data.server_info.cur_playing_idx == ad->mc_data.server_info.playlist_item_num)
			ad->mc_data.server_info.cur_playing_idx = 0;
		break;
	case MC_PLAYBACK_ACTION_PREV:
		ad->mc_data.server_info.pb_state = MC_PLAYBACK_STATE_MOVING_TO_PREVIOUS;
		/* if MC_PLAYBACK_ACTION_PREV is requested, decrease index */
		ad->mc_data.server_info.cur_playing_idx--;
		if (ad->mc_data.server_info.cur_playing_idx < 0)
			ad->mc_data.server_info.cur_playing_idx = 0;
		break;
	case MC_PLAYBACK_ACTION_FAST_FORWARD:
		ad->mc_data.server_info.pb_state = MC_PLAYBACK_STATE_FAST_FORWARDING;
		break;
	case MC_PLAYBACK_ACTION_REWIND:
		ad->mc_data.server_info.pb_state = MC_PLAYBACK_STATE_REWINDING;
		break;
	case MC_PLAYBACK_ACTION_TOGGLE_PLAY_PAUSE:
		/* if MC_PLAYBACK_ACTION_TOGGLE_PLAY_PAUSE is requested */
		/* if current state is play, the states will be changed to pause */
		if (ad->mc_data.server_info.pb_state == MC_PLAYBACK_STATE_PLAYING)
			ad->mc_data.server_info.pb_state = MC_PLAYBACK_STATE_PAUSED;
		/* if current state is pause, the states will be changed to play */
		else if (ad->mc_data.server_info.pb_state == MC_PLAYBACK_STATE_PAUSED)
			ad->mc_data.server_info.pb_state = MC_PLAYBACK_STATE_PLAYING;
		/* if current state is not play or pause, the error will be returned */
		else
			result_code = MEDIA_CONTROLLER_ERROR_INVALID_PARAMETER;
		break;
	default:
		ad->mc_data.server_info.pb_state = MC_PLAYBACK_STATE_NONE;
		break;
	}

	/* if result_code is not error, UI should be updated here */
	if (result_code == MEDIA_CONTROLLER_ERROR_NONE) {
		/* if playlist exist, update playlist */
		if (ad->mc_data.server_info.playlist) {
			unsigned int i = 0;
			/* gets current playing index */
			playlist_s *item = g_list_nth_data(ad->mc_data.server_info.playlist, ad->mc_data.server_info.cur_playing_idx);

			/* if index of playlist is changed, update metadata of content */
			if (g_strcmp0(ad->mc_data.server_info.playlist_index, item->index) != 0) {
				SAFE_FREE(ad->mc_data.server_info.playlist_index);
				ad->mc_data.server_info.playlist_index = g_strdup(item->index);
				for (i = 0; i < LAST_META_ITEM; i++) {
					SAFE_FREE(ad->mc_data.server_info.metadata[i]);
					ad->mc_data.server_info.metadata[i] = g_strdup(item->metadata[i]);
				}
				mcs_update_view_metadata(ad);
			}
		}
		/* update playback view */
		mcs_update_view_playback(ad);
		/* send updated playback information to client applications */
		mcs_update_playback(ad);
	}

	/* send a replay to client applications */
	mcs_send_cmd_reply(ad, client_name, request_id, result_code);

	DEBUG_LEAVE();
}

/*
 * @brief Called when the server receives the playback action command from the client.
 * @param[in] client_name    The app_id of the media controller client.
 * @param[in] request_id    The request_id of the media controller client.
 * @param[in] action    The received playback action
 * @param[in] user_data        The user data passed from the mc_server_set_playback_action_cmd_received_cb() function
 */
void _pb_position_cmd_received_cb(const char *client_name, const char *request_id, unsigned long long position, void *user_data)
{
	int result_code = MEDIA_CONTROLLER_ERROR_NONE;
	appdata_s *ad = user_data;

	DEBUG_ENTER();

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	unsigned long long duration = strtoull(ad->mc_data.server_info.metadata[META_DURATION], NULL, 10);

	/* change playback position */
	if (duration < position)
		result_code = MEDIA_CONTROLLER_ERROR_INVALID_PARAMETER;

	/* if result_code is not error, UI should be updated here */
	if (result_code == MEDIA_CONTROLLER_ERROR_NONE) {
		ad->mc_data.server_info.pb_position = position;
		/* update playback view */
		mcs_update_view_playback(ad);
		/* send updated playback information to client applications */
		mcs_update_playback(ad);
	}

	/* send a replay to client applications */
	mcs_send_cmd_reply(ad, client_name, request_id, result_code);

	DEBUG_LEAVE();
}

/*
 * @brief Called when the server receives the shuffle mode from the client.
 * @param[in] client_name    The app_id of the media controller client.
 * @param[in] request_id    The request_id of the media controller client.
 * @param[in] shuffle_mode    The received shuffle_mode
 * @param[in] user_data        The user data passed from the mc_server_set_shuffle_mode_cmd_received_cb() function
 */
void _shuffle_cmd_received_cb(const char *client_name, const char *request_id, mc_shuffle_mode_e shuffle_mode, void *user_data)
{
	int result_code = MEDIA_CONTROLLER_ERROR_NONE;
	appdata_s *ad = user_data;

	DEBUG_ENTER();

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	/* change shuffle mode */
	if (ad->mc_data.server_info.shuffle == shuffle_mode)
		result_code = MEDIA_CONTROLLER_ERROR_INVALID_PARAMETER;
	else
		ad->mc_data.server_info.shuffle = shuffle_mode;

	/* if result_code is not error, UI should be updated here */
	if (result_code == MEDIA_CONTROLLER_ERROR_NONE) {
		/* update playback view */
		mcs_update_view_playback(ad);
		/* send updated shuffle information to client applications */
		mcs_update_shuffle(ad);
	}

	/* send a replay to client applications */
	mcs_send_cmd_reply(ad, client_name, request_id, result_code);

	DEBUG_LEAVE();
}

/*
 * @brief Called when the server receives the shuffle mode from the client.
 * @param[in] client_name    The app_id of the media controller client.
 * @param[in] request_id    The request_id of the media controller client.
 * @param[in] repeat_mode    The received repeat mode
 * @param[in] user_data        The user data passed from the mc_server_set_repeat_mode_cmd_received_cb() function
 */
void _repeat_cmd_received_cb(const char *client_name, const char *request_id, mc_repeat_mode_e repeat_mode, void *user_data)
{
	int result_code = MEDIA_CONTROLLER_ERROR_NONE;
	appdata_s *ad = user_data;

	DEBUG_ENTER();

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	/* change repeat mode */
	if (ad->mc_data.server_info.repeat == repeat_mode)
		result_code = MEDIA_CONTROLLER_ERROR_INVALID_PARAMETER;
	else
		ad->mc_data.server_info.repeat = repeat_mode;

	/* if result_code is not error, UI should be updated here */
	if (result_code == MEDIA_CONTROLLER_ERROR_NONE) {
		/* update playback view */
		mcs_update_view_playback(ad);
		/* send updated repeat information to client applications */
		mcs_update_repeat(ad);
	}

	/* send a replay to client applications */
	mcs_send_cmd_reply(ad, client_name, request_id, result_code);

	DEBUG_LEAVE();
}

/*
 * @brief Called when the Server receives custom command from the client.
 * @param[in] client_name    The app_id of the media controller client.
 * @param[in] request_id     The id of the command request
 * @param[in] command    The received command.
 * @param[in] data    The extra data
 * @param[in] user_data        The user data passed from the mc_server_set_custom_cmd_received_cb() function
 */
void _custom_cmd_received_cb(const char *client_name, const char *request_id, const char *command, bundle *data, void *user_data)
{
	int result_code = MEDIA_CONTROLLER_ERROR_NONE;
	appdata_s *ad = user_data;

	DEBUG_ENTER();

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	/* do something for custom command */
	SAFE_FREE(ad->mc_data.custom_cmd);
	ad->mc_data.custom_cmd = g_strdup(command);
	ad->mc_data.custom_cmd_result = result_code;

	/* if result_code is not error, UI should be updated here */
	if (result_code == MEDIA_CONTROLLER_ERROR_NONE) {
		/* update custom command view */
		mcs_update_view_custom_command(ad);
	}

	/* send a replay to client applications */
	mcs_send_cmd_reply(ad, client_name, request_id, result_code);

	DEBUG_LEAVE();
}

/*
 * @brief Called when the Server receives playlist command from the client.
 * @param[in] client_name    The app_id of the media controller client.
 * @param[in] request_id    The request_id of the media controller client.
 * @param[in] playlist_name    The name of the server playlist.
 * @param[in] index    The index of the media in playlist.
 * @param[in] action    The received playback action
 * @param[in] position    The received playback position (milliseconds)
 * @param[in] user_data    The user data passed from the mc_server_set_playlist_cmd_received_cb() function
 */
void _playlist_cmd_received_cb(const char *client_name, const char *request_id, const char *playlist_name, const char *index, mc_playback_action_e action, unsigned long long position, void *user_data)
{
	int result_code = MEDIA_CONTROLLER_ERROR_NONE;
	appdata_s *ad = user_data;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	DEBUG_ENTER();

	LOG_DBG("req_id: %s playlist name: %s index: %s", request_id, playlist_name, index);

	if (g_strcmp0(playlist_name, ad->mc_data.server_info.playlist_name) != 0) {
		result_code = MEDIA_CONTROLLER_ERROR_INVALID_PARAMETER;
		mcs_send_cmd_reply(ad, client_name, request_id, result_code);
		return;
	}

	SAFE_FREE(ad->mc_data.server_info.playlist_index);
	ad->mc_data.server_info.playlist_index = g_strdup(index);

	/* change playback state */
	if (action == MC_PLAYBACK_ACTION_TOGGLE_PLAY_PAUSE) {
		if (ad->mc_data.server_info.pb_state == MC_PLAYBACK_ACTION_PLAY)
			ad->mc_data.server_info.pb_state = MC_PLAYBACK_STATE_PAUSED;
		else if (ad->mc_data.server_info.pb_state == MC_PLAYBACK_STATE_PAUSED)
			ad->mc_data.server_info.pb_state = MC_PLAYBACK_STATE_PLAYING;
		else
			result_code = MEDIA_CONTROLLER_ERROR_INVALID_PARAMETER;
	} else {
		ad->mc_data.server_info.pb_state = action + 1;
	}

	ad->mc_data.server_info.pb_position = position;

	/* should update UI here */
	mcs_update_view_playback(ad);
	mcs_update_view_metadata_by_index(ad, ad->mc_data.server_info.playlist_index);

	mcs_update_playback(ad);
	mcs_update_metadata(ad);

	/* send a replay to client applications */
	mcs_send_cmd_reply(ad, client_name, request_id, result_code);

	DEBUG_LEAVE();
}

int _compare_str(const void *user_data, const void *data)
{
	return g_strcmp0((char *)user_data, (char *)data);
}

/*
 * @brief Called when the result of the event from the client is received.
 * @param[in] client_name    The app_id of the media controller client which sent the reply. It can be used only in the callback. To use outside, make a copy.
 * @param[in] request_id     The id of the event request
 * @param[in] result_code    The result code of the event
 * @param[in] data The extra data
 * @param[in] user_data        The user data passed from the mc_server_set_event_reply_received_cb() function
 */
void _event_reply_received_cb(const char *client_name, const char *request_id, int result_code, bundle *data, void *user_data)
{
	appdata_s *ad = user_data;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	DEBUG_ENTER();

	if (g_list_find_custom(ad->mc_data.req_ids, request_id, _compare_str) == NULL) {
		LOG_ERR("Invalid request_id(%s) was received from %s", request_id, client_name);
		return;
	}

	ad->mc_data.req_ids = g_list_remove(ad->mc_data.req_ids, request_id);
	/* should update UI here */
	mcs_update_view_event_reply(ad);

	DEBUG_LEAVE();
}

/*
 * @brief Called when requesting the list of created clients.
 * @param[in] client_name The app_id of the created media controller client.
 * @param[in] user_data        The user data passed from the mc_client_foreach_server() function
 */
bool _activated_client_cb(const char *client_name, void *user_data)
{
	GList **client_list = (GList **)user_data;

	if (client_name != NULL) {
		if (g_list_find_custom(*client_list, client_name, _compare_str))
			return true;

		*client_list = g_list_append(*client_list, g_strdup(client_name));
	}

	return true;
}

/*
 * @brief Free an element in clients list to use SAFE_FREE
 * @param[in] data The client_name.
 */
void _free_list(gpointer data)
{
	SAFE_FREE(data);
}

/*
 * @brief Creates a media controller server.
 * @param[out] ad The structure of app data include the handle of the media controller server
 */
void mcs_create_server(appdata_s *ad)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	/* create media controller handle, it will be kept during running application */
	ret = mc_server_create((mc_server_h *)&ad->mc_data.handle);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to create server.(%d)", ret);
}

/*
 * @brief Destroys media controller server.
 * @param[in] ad The structure of app data
 */
void mcs_destroy_server(appdata_s *ad)
{
	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");
	mc_server_h handle = (mc_server_h)ad->mc_data.handle;
	/* check valid handle of media controller */
	RETM_IF((handle == NULL), "Invalid handle");

	/* free client_name for _activated_client_cb */
	g_list_free_full(ad->mc_data.client_list, _free_list);
	/* free requst_id for  */
	g_list_free_full(ad->mc_data.req_ids, _free_list);

	/* destroy media controller handle during application termination */
	mc_server_destroy(handle);
}

/*
 * @brief Function to set callback to use
 * @param[in] ad The structure of app data
 */
void mcs_set_callback(appdata_s *ad)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");
	mc_server_h handle = (mc_server_h)ad->mc_data.handle;
	/* check valid handle of media controller */
	RETM_IF((handle == NULL), "Invalid handle");

	/* register callback */

	/* set the callback to receive the playback action command from client */
	ret = mc_server_set_playback_action_cmd_received_cb(handle, _pb_action_cmd_received_cb, ad);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to set action command callback.(%d)", ret);

	/* set the callback to receive the playback position command from client */
	ret = mc_server_set_playback_position_cmd_received_cb(handle, _pb_position_cmd_received_cb, ad);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to set position command callback.(%d)", ret);

	/* set the callback to receive the shuffle mode command from client */
	ret = mc_server_set_shuffle_mode_cmd_received_cb(handle, _shuffle_cmd_received_cb, ad);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to set shuffle command callback.(%d)", ret);

	/* set the callback to receive the repeat mode command from client */
	ret = mc_server_set_repeat_mode_cmd_received_cb(handle, _repeat_cmd_received_cb, ad);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to set repeat command callback.(%d)", ret);

	/* set the callback to receive the custom command from client */
	ret = mc_server_set_custom_cmd_received_cb(handle, _custom_cmd_received_cb, ad);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to set custom command callback.(%d)", ret);

	/* set the callback to receive the playlist command from client */
	ret = mc_server_set_playlist_cmd_received_cb(handle, _playlist_cmd_received_cb, ad);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to set playlist command callback.(%d)", ret);

	/* set the callback to receive the event reply from client */
	ret = mc_server_set_event_reply_received_cb(handle, _event_reply_received_cb, ad);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to set event reply callback.(%d)", ret);
}

/*
 * @brief Function to unset callback
 * @param[in] ad The structure of app data
 */
void mcs_unset_callback(appdata_s *ad)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");
	mc_server_h handle = (mc_server_h)ad->mc_data.handle;
	RETM_IF((handle == NULL), "Invalid handle");

	/* unregister callback */

	/* unset the callback not to receive the playback action command from client */
	ret = mc_server_unset_playback_action_cmd_received_cb(handle);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to unset action command callback.(%d)", ret);

	/* unset the callback not to receive the playback position command from client */
	ret = mc_server_unset_playback_position_cmd_received_cb(handle);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to unset position command callback.(%d)", ret);

	/* unset the callback not to receive the shuffle mode command from client */
	ret = mc_server_unset_shuffle_mode_cmd_received_cb(handle);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to unset shuffle command callback.(%d)", ret);

	/* unset the callback not to receive the repeat mode command from client */
	ret = mc_server_unset_repeat_mode_cmd_received_cb(handle);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to unset repeat command callback.(%d)", ret);

	/* unset the callback not to receive the custom command from client */
	ret = mc_server_unset_custom_cmd_received_cb(handle);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to unset custom command callback.(%d)", ret);

	/* unset the callback not to receive the playlist command from client */
	ret = mc_server_unset_playlist_cmd_received_cb(handle);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to unset playlist command callback.(%d)", ret);

	/* unset the callback not to receive the event reply from client */
	ret = mc_server_unset_event_reply_received_cb(handle);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to unset event reply callback.(%d)", ret);
}

/*
 * @brief Updates the modified playback info.
 * @param[in] ad The structure of app data
 */
void mcs_update_playback(appdata_s *ad)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	mc_server_h handle = (mc_server_h)ad->mc_data.handle;
	/* check valid handle of media controller */
	RETM_IF((handle == NULL), "Invalid handle");

	/* set a playback state before update playback information */
	ret = mc_server_set_playback_state(handle, ad->mc_data.server_info.pb_state);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to set playback state.(%d)", ret);
	/* set a playback position before update playback information */
	ret = mc_server_set_playback_position(handle, ad->mc_data.server_info.pb_position);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to set playback position.(%d)", ret);
	/* set a playback item before update playback information */
	ret = mc_server_set_playlist_item_info(handle, ad->mc_data.server_info.playlist_name, ad->mc_data.server_info.playlist_index);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to set playlist index.(%d)", ret);

	/* update the playback information to send to client applications */
	ret = mc_server_update_playback_info(handle);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to update playback information.(%d)", ret);
}

/*
 * @brief Updates the modified metadata info.
 * @param[in] ad The structure of app data
 */
void mcs_update_metadata(appdata_s *ad)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;
	unsigned int i = 0;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	mc_server_h handle = (mc_server_h)ad->mc_data.handle;
	/* check valid handle of media controller */
	RETM_IF((handle == NULL), "Invalid handle");

	/* set metadata information */
	for (i = 0; i < LAST_META_ITEM; i++) {
		ret = mc_server_set_metadata(handle, i, ad->mc_data.server_info.metadata[i]);
		RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to set %d metadata.(%d)", i, ret);
	}

	/* update the metadata information to send to client applications */
	ret = mc_server_update_metadata(handle);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to update metadata information.(%d)", ret);
}

/*
 * @brief Updates the modified shuffle mode.
 * @param[in] ad The structure of app data
 */
void mcs_update_shuffle(appdata_s *ad)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	mc_server_h handle = (mc_server_h)ad->mc_data.handle;
	/* check valid handle of media controller */
	RETM_IF((handle == NULL), "Invalid handle");

	/* update a shuffle mode information to send to client applications */
	ret = mc_server_update_shuffle_mode(handle, ad->mc_data.server_info.shuffle);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to update shuffle mode information.(%d)", ret);
}

/*
 * @brief Updates the modified repeat mode.
 * @param[in] ad The structure of app data
 */
void mcs_update_repeat(appdata_s *ad)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	mc_server_h handle = (mc_server_h)ad->mc_data.handle;
	/* check valid handle of media controller */
	RETM_IF((handle == NULL), "Invalid handle");

	/* update a repeat mode information to send to client applications */
	ret = mc_server_update_repeat_mode(handle, ad->mc_data.server_info.repeat);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to update repeat mode information.(%d)", ret);
}

/*
 * @brief Registers items to the playlist.
 * @param[in] ad The structure of app data
 */
void mcs_update_playlist(appdata_s *ad)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	mc_server_h handle = (mc_server_h)ad->mc_data.handle;
	/* check valid handle of media controller */
	RETM_IF((handle == NULL), "Invalid handle");
	RETM_IF((ad->mc_data.server_info.playlist == NULL), "Invalid playlist");

	mc_playlist_h playlist = NULL;
	unsigned int i = 0, j = 0;
	unsigned int length = g_list_length(ad->mc_data.server_info.playlist);

	/* create a playlist handle */
	ret = mc_server_create_playlist(handle, ad->mc_data.server_info.playlist_name, &playlist);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to create playlist.(%d)", ret);

	/* add items to playlist */
	for (i = 0; i < length; i++) {
		playlist_s *item = g_list_nth_data(ad->mc_data.server_info.playlist, i);
		if (item == NULL)
			continue;
		for (j = 0; j < LAST_META_ITEM; j++) {
			ret = mc_server_add_item_to_playlist(handle, playlist, item->index, j, item->metadata[j]);
			if (ret != MEDIA_CONTROLLER_ERROR_NONE) {
				LOG_ERR("Fail to add item.(%d)", ret);
				mc_playlist_destroy(playlist);
				return;
			}
		}
	}

	/* update the playlist information to send to client applications */
	ret = mc_server_update_playlist_done(handle, playlist);
	if (ret != MEDIA_CONTROLLER_ERROR_NONE)
		LOG_ERR("Fail to update playlist.(%d)", ret);

	mc_playlist_destroy(playlist);
}

/*
 * @brief Replies the result of the requested command to the client.
 * @param[in] ad The structure of app data
 * @param[in] client_name    The app_id of the media controller client
 * @param[in] request_id    The id of the command request, received in the mc_server_custom_cmd_received_cb() function.
 * @param[in] result_code    The result code of custom command
 */
void mcs_send_cmd_reply(appdata_s *ad, const char *client_name, const char *request_id, int result_code)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	mc_server_h handle = (mc_server_h)ad->mc_data.handle;
	/* check valid handle of media controller */
	RETM_IF((handle == NULL), "Invalid handle");

	ret = mc_server_send_cmd_reply(handle, client_name, request_id, result_code, NULL);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to send the reply of command.(%d)", ret);
}

/*
 * @brief Retrieves all created clients.
 * @param[in] ad The structure of app data
 */
void mcs_get_client_list(appdata_s *ad)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");
	mc_server_h handle = (mc_server_h)ad->mc_data.handle;
	/* check valid handle of media controller */
	RETM_IF((handle == NULL), "Invalid handle");

	/* get client list for sending custom event */
	ret = mc_server_foreach_client(handle, _activated_client_cb, &ad->mc_data.client_list);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to get client list.(%d)", ret);

	ad->mc_data.client_num = g_list_length(ad->mc_data.client_list);
}

/*
 * @brief Sends the customized event with the bundle data.
 * @param[in] ad The structure of app data
 * @param[in] client_name The name of the client which receive the event
 * @param[in] event_name The name of the event
 */
void mcs_send_custom_event(appdata_s *ad, const char *client_name, const char *event_name)
{
	int ret = MEDIA_CONTROLLER_ERROR_NONE;
	char *request_id = NULL;

	/* check valid appdata */
	RETM_IF((ad == NULL), "Invalid appdata_s");

	mc_server_h handle = (mc_server_h)ad->mc_data.handle;
	/* check valid handle of media controller */
	RETM_IF((handle == NULL), "Invalid handle");

	ret = mc_server_send_custom_event(handle, client_name, event_name, NULL, &request_id);
	RETM_IF((ret != MEDIA_CONTROLLER_ERROR_NONE), "Fail to send the reply of command.(%d)", ret);

	ad->mc_data.req_ids = g_list_append(ad->mc_data.req_ids, request_id);
}
