/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <tizen.h>
#include <dlog.h>
#include <app.h>
#include <efl_extension.h>
#include <Elementary.h>

#include "$(appName).h"
#include "view.h"

#include <glib.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "log.h"
#include "helper.h"

static struct view_info {
	Evas_Object *win;
	Evas_Object *conform;
} s_info = {
	.win = NULL,
	.conform = NULL,
};

static void _dummy_callback_cb(void *data, Evas_Object *obj, void *event_info);

/**
 * @brief Creates essential objects: window, conformant and layout.
 */
Eina_Bool view_create(void *user_data)
{
	/* Create the window */
	s_info.win = view_create_win(PACKAGE);
	if (s_info.win == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create a window.");
		return EINA_FALSE;
	}

	/* Create the conformant */
	s_info.conform = view_create_conformant_without_indicator(s_info.win);
	if (s_info.conform == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "failed to create a conformant");
		return EINA_FALSE;
	}

	/* Show the window after main view is set up */
	evas_object_show(s_info.win);
	return EINA_TRUE;
}

/**
 * @brief Creates a basic window named package.
 * @param[in] pkg_name Name of the window
 */
Evas_Object *view_create_win(const char *pkg_name)
{
	Evas_Object *win = NULL;

	/*
	 * Window
	 * Create and initialize elm_win.
	 * elm_win is mandatory to manipulate the window.
	 */
	win = elm_win_util_standard_add(pkg_name, pkg_name);
	elm_win_conformant_set(win, EINA_TRUE);
	elm_win_autodel_set(win, EINA_TRUE);

	/* Rotation setting */
	if (elm_win_wm_rotation_supported_get(win)) {
		int rots[4] = { 0, 90, 180, 270 };
		elm_win_wm_rotation_available_rotations_set(win, (const int *)(&rots), 4);
	}

	evas_object_smart_callback_add(win, "delete,request", _dummy_callback_cb, NULL);

	return win;
}

/**
 * @brief Creates a layout to target parent object with edje file
 * @param[in] parent The object to which you want to add this layout
 * @param[in] file_path File path of EDJ file will be used
 * @param[in] group_name Name of group in EDJ you want to set to
 * @param[in] cb_function The function will be called when back event is detected
 * @param[in] user_data The user data to be passed to the callback functions
 */
Evas_Object *view_create_layout(Evas_Object *parent, const char *file_path, const char *group_name, Eext_Event_Cb cb_function, void *user_data)
{
	Evas_Object *layout = NULL;

	if (parent == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "parent is NULL.");
		return NULL;
	}

	/* Create layout using EDC(an edje file) */
	layout = elm_layout_add(parent);
	elm_layout_file_set(layout, file_path, group_name);

	/* Layout size setting */
	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	if (cb_function)
		eext_object_event_callback_add(layout, EEXT_CALLBACK_BACK, cb_function, user_data);

	evas_object_show(layout);

	return layout;
}

/**
 * @brief Creates a conformant without indicator for wearable app.
 * @param[in] win The object to which you want to set this conformant
 * Conformant is mandatory for base GUI to have proper size
 */
Evas_Object *view_create_conformant_without_indicator(Evas_Object *win)
{
	/*
	 * Conformant
	 * Create and initialize elm_conformant.
	 * elm_conformant is mandatory for base GUI to have proper size
	 * when indicator or virtual keypad is visible.
	 */
	Evas_Object *conform = NULL;

	if (win == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "window is NULL.");
		return NULL;
	}

	conform = elm_conformant_add(win);
	evas_object_size_hint_weight_set(conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(win, conform);

	evas_object_show(conform);

	return conform;
}

/**
 * @brief Destroys window and frees its resources.
 */
void view_destroy(void)
{
	if (s_info.win == NULL)
		return;

	evas_object_del(s_info.win);
}

/**
 * @brief Destroys given layout.
 * @param[in] layout The layout to destroy
 */
void view_destroy_layout(Evas_Object *layout)
{
	evas_object_del(layout);
}

/**
 * @brief Function will be operated when registered event is triggered.
 * @param[in] data The data to be passed to the callback function
 * @param[in] obj The Evas object handle to be passed to the callback function
 * @param[in] event_info The system event information
 */
static void _dummy_callback_cb(void *data, Evas_Object *obj, void *event_info)
{
	/*
	 * Write your code here for smart callback.
	 */
}

Evas_Object* _create_label(Evas_Object *parent, const char *text);

/*
 * @brief Gets the index from the selected text
 * @param[in] selected The selected text
 * @param[in] items    The text list for check the selected parameter
 * @param[in] item_num    The length of the text list for check the selected parameter
 * @return index number
 */
int __get_index_from_item_text(const char *selected, const char **items, unsigned int item_num)
{
	unsigned int i = 0;
	RETVM_IF(selected == NULL, -1, "Invalid selected");
	RETVM_IF(items == NULL, -1, "Invalid items");

	for (i = 0; i < item_num; i++) {
		if (items[i] == NULL)
			continue;
		if (g_strcmp0(selected, _STYLE(items[i])) == 0)
			return (int)i;
	}

	return -1;
}

/*
 * @brief Called when find selected item in the list
 * @param[in] data		The element of the list
 * @param[in] user_data	The selected text
 * @return index number
 */
int __compare_index(const void *data, const void *user_data)
{
	const char *index = (const char *)user_data;
	const playlist_s *item = (const playlist_s *)data;

	LOG_DBG("__compare_index: %s %s", item->index, index);

	return g_strcmp0(index, _STYLE(item->index));
}

/*
 * @brief Gets the item from the selected item in the playlist
 * @param[in] selected The text of the selected object
 * @param[in] list    The playlist to get item
 * @return the structure of the item in playlist
 */
playlist_s* __get_item_from_playlist(const char *selected, GList *list)
{
	GList *item = NULL;
	RETVM_IF(selected == NULL, NULL, "Invalid selected");
	RETVM_IF(list == NULL, NULL, "Invalid list");

	item = g_list_find_custom(list, selected, __compare_index);
	if (item == NULL)
		return NULL;

	return item->data;
}

/*
 * @brief Callback function to be called when server is selected.
 * @param[in] data The structure of app data
 * @param[in] obj    The object to set
 * @param[in] event_info    The selected event information
 */
void _hover_playback_state_selected_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = (appdata_s *)data;
	Elm_Object_Item *it = event_info;
	mc_playback_states_e playback_state = MC_PLAYBACK_STATE_NONE;
	/* check valid appdata */
	RETM_IF(ad == NULL, "Invalid appdata");
	RETM_IF(obj == NULL, "Invalid object");
	RETM_IF(event_info == NULL, "Invalid event");

	int index = __get_index_from_item_text(elm_object_item_text_get(it), hover_items_pb_state, hover_item_num_pb_state);
	LOG_DBG("Selected(state): %d", index);

	/*
	 * convert playback_state due to deprecated enum
	 * typedef enum {
	 * MC_PLAYBACK_STATE_NONE = 0,
	 * MC_PLAYBACK_STATE_PLAYING,
	 * MC_PLAYBACK_STATE_PAUSED,
	 * MC_PLAYBACK_STATE_STOPPED,
	 * MC_PLAYBACK_STATE_MOVING_TO_NEXT = 8,
	 * MC_PLAYBACK_STATE_MOVING_TO_PREVIOUS,
	 * MC_PLAYBACK_STATE_FAST_FORWARDING,
	 * MC_PLAYBACK_STATE_REWINDING,
	 * } mc_playback_states_e;
	 */
	switch (index) {
	case 1:
		playback_state = MC_PLAYBACK_STATE_PLAYING;
		break;
	case 2:
		playback_state = MC_PLAYBACK_STATE_PAUSED;
		break;
	case 3:
		playback_state = MC_PLAYBACK_STATE_STOPPED;
		break;
	case 4:
		playback_state = MC_PLAYBACK_STATE_MOVING_TO_NEXT;
		break;
	case 5:
		playback_state = MC_PLAYBACK_STATE_MOVING_TO_PREVIOUS;
		break;
	case 6:
		playback_state = MC_PLAYBACK_STATE_FAST_FORWARDING;
		break;
	case 7:
		playback_state = MC_PLAYBACK_STATE_REWINDING;
		break;
	case 0:
	default:
		playback_state = MC_PLAYBACK_STATE_NONE;
		break;
	}

	/* set the text to selected item */
	elm_object_text_set(obj, _STYLE(hover_items_pb_state[index]));
	/* check current playlist */
	if (ad->mc_data.server_info.playlist) {
		/* MC_PLAYBACK_STATE_NEXT_FILE(4) is deprecated, it will be removed since tizen 5.5. */
		if (playback_state == MC_PLAYBACK_STATE_MOVING_TO_NEXT) {
			ad->mc_data.server_info.cur_playing_idx++;
			/* go playing item to 1st item circularly */
			if (ad->mc_data.server_info.cur_playing_idx == ad->mc_data.server_info.playlist_item_num)
				ad->mc_data.server_info.cur_playing_idx = 0;
			/* MC_PLAYBACK_STATE_PREV_FILE(5) is deprecated, it will be removed since tizen 5.5. */
		} else if (playback_state == MC_PLAYBACK_STATE_MOVING_TO_PREVIOUS) {
			ad->mc_data.server_info.cur_playing_idx--;
			/* keep playing item to 1st item */
			if (ad->mc_data.server_info.cur_playing_idx < 0)
				ad->mc_data.server_info.cur_playing_idx = 0;
		}
		unsigned int i = 0;
		playlist_s *item = g_list_nth_data(ad->mc_data.server_info.playlist, ad->mc_data.server_info.cur_playing_idx);

		/* set metadata before update view */
		if (g_strcmp0(ad->mc_data.server_info.playlist_index, item->index) != 0) {
			SAFE_FREE(ad->mc_data.server_info.playlist_index);
			ad->mc_data.server_info.playlist_index = g_strdup(item->index);
			for (i = 0; i < LAST_META_ITEM; i++) {
				SAFE_FREE(ad->mc_data.server_info.metadata[i]);
				ad->mc_data.server_info.metadata[i] = g_strdup(item->metadata[i]);
			}
		}
	}
	/* update playback state */
	ad->mc_data.server_info.pb_state = playback_state;

	/* update metadata view */
	mcs_update_view_metadata(ad);
	/* update playback view */
	mcs_update_view_playback(ad);

	/* update playback information to client application */
	mcs_update_playback(ad);
}

/*
 * @brief Callback function to be called when position command is selected.
 * @param[in] data The structure of app data
 * @param[in] obj    The object to set
 * @param[in] event_info    The selected event information
 */
void _hover_playback_position_selected_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = (appdata_s *)data;
	Elm_Object_Item *it = event_info;
	/* check valid appdata */
	RETM_IF(ad == NULL, "Invalid appdata");
	RETM_IF(obj == NULL, "Invalid object");
	RETM_IF(event_info == NULL, "Invalid event");

	/* get the index of selected item in hoversel */
	int index = __get_index_from_item_text(elm_object_item_text_get(it), hover_items_pb_position, hover_item_num_pb_position);
	LOG_DBG("Selected(position): %s (%d)", elm_object_item_text_get(it), index);

	/* set the text to selected item */
	elm_object_text_set(obj, _STYLE(hover_items_pb_position[index]));

	char *end = NULL;
	char *input = g_strdup(hover_items_pb_position[index]);
	if (!input) {
		LOG_ERR("Input is null");
		return;
	}

	/* change text to number */
	unsigned long long ull = strtoull(input, &end, 10);
	if ((end == input) || ('\0' != *end) || ((ULLONG_MAX == ull) && (ERANGE == errno))) {
		LOG_ERR("not a proper number");
		SAFE_FREE(input);
		return;
	}
	SAFE_FREE(input);
	/* update playback position */
	ad->mc_data.server_info.pb_position = ull;
	mcs_update_playback(ad);
}

/*
 * @brief Callback function to be called when playlist is selected.
 * @param[in] data The structure of app data
 * @param[in] obj    The object to set
 * @param[in] event_info    The selected event information
 */
void _hover_playlist_selected_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = (appdata_s *)data;
	Elm_Object_Item *it = event_info;
	playlist_s *item = NULL;
	/* check valid appdata */
	RETM_IF(ad == NULL, "Invalid appdata");
	RETM_IF(obj == NULL, "Invalid object");
	RETM_IF(event_info == NULL, "Invalid event");

	const char *selected = elm_object_item_text_get(it);
	LOG_DBG("Selected(index): %s", selected);

	/* get the index of object item in playlist */
	item = __get_item_from_playlist(selected, ad->mc_data.server_info.playlist);
	if (item == NULL)
		return;

	/* update current playlist index */
	SAFE_FREE(ad->mc_data.server_info.playlist_index);
	ad->mc_data.server_info.playlist_index = g_strdup(item->index);

	elm_object_text_set(obj, _STYLE(ad->mc_data.server_info.playlist_index));

	/* update metadata view */
	mcs_update_view_metadata_by_index(ad, ad->mc_data.server_info.playlist_index);

	/* update playback information to client application */
	mcs_update_playback(ad);
}

/*
 * @brief Callback function to be called when shuffle command is selected.
 * @param[in] data The structure of app data
 * @param[in] obj    The object to set
 * @param[in] event_info    The selected event information
 */
void _hover_shuffle_selected_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = (appdata_s *)data;
	Elm_Object_Item *it = event_info;
	/* check valid appdata */
	RETM_IF(ad == NULL, "Invalid appdata");
	RETM_IF(obj == NULL, "Invalid object");
	RETM_IF(event_info == NULL, "Invalid event");

	/* get the index of selected item in hoversel */
	int index = __get_index_from_item_text(elm_object_item_text_get(it), hover_items_shuffle, hover_item_num_shuffle);
	LOG_DBG("Selected(shuffle): %s (%d)", elm_object_item_text_get(it), index);

	/* set the text to selected item */
	elm_object_text_set(obj, _STYLE(hover_items_shuffle[index]));
	/* update shuffle */
	ad->mc_data.server_info.shuffle = index;

	/* update shuffle information to client application */
	mcs_update_shuffle(ad);
}

/*
 * @brief Callback function to be called when repeat command is selected.
 * @param[in] data The structure of app data
 * @param[in] obj    The object to set
 * @param[in] event_info    The selected event information
 */
void _hover_repeat_selected_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = (appdata_s *)data;
	Elm_Object_Item *it = event_info;
	/* check valid appdata */
	RETM_IF(ad == NULL, "Invalid appdata");
	RETM_IF(obj == NULL, "Invalid object");
	RETM_IF(event_info == NULL, "Invalid event");

	/* get the index of selected item in hoversel */
	int index = __get_index_from_item_text(elm_object_item_text_get(it), hover_items_repeat, hover_item_num_repeat);
	LOG_DBG("Selected(repeat): %s (%d)", elm_object_item_text_get(it), index);

	/* set the text to selected item */
	elm_object_text_set(obj, _STYLE(hover_items_repeat[index]));
	/* update repeat */
	ad->mc_data.server_info.repeat = index;

	/* update repeat information to client application */
	mcs_update_repeat(ad);
}

/*
 * @brief Callback function to be called when custom command is selected.
 * @param[in] data The structure of app data
 * @param[in] obj    The object to set
 * @param[in] event_info    The selected event information
 */
void _hover_event_selected_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = (appdata_s *)data;
	Elm_Object_Item *it = event_info;
	/* check valid appdata */
	RETM_IF(ad == NULL, "Invalid appdata");
	RETM_IF(obj == NULL, "Invalid object");
	RETM_IF(event_info == NULL, "Invalid event");

	/* get the index of object item in hoversel */
	int index = __get_index_from_item_text(elm_object_item_text_get(it), hover_items_event, hover_item_num_event);
	LOG_DBG("Selected(event): %s (%d)", elm_object_item_text_get(it), index);

	/* set the text to selected item */
	elm_object_text_set(obj, _STYLE(hover_items_event[index]));
	/* send command */
	mcs_get_client_list(ad);
	mcs_clear_view_event_reply(ad);
	mcs_send_custom_event(ad, g_list_nth_data(ad->mc_data.client_list, 0), hover_items_event[index]);
}

/*
 * @brief Callback function to get item label in genlist object
 * @param[in] data The structure of app data
 * @param[in] obj    The object to set
 * @param[in] part   The part of object
 */
static char* _genlist_item_label_get(void *data, Evas_Object *obj, const char *part)
{
	playlist_s *item = (playlist_s *)data;
	if (item)
		return strdup(item->metadata[META_TITLE]);
	else
		return NULL;
}

/*
 * @brief Callback function to get item content in genlist object
 * @param[in] data The structure of app data
 * @param[in] obj    The object to set
 * @param[in] part   The part of object
 */
static Evas_Object* _genlist_item_content_get(void *data, Evas_Object *obj, const char *part)
{
	playlist_s *item = (playlist_s *)data;
	if (item) {
		Evas_Object *label = elm_label_add(obj);
		elm_object_text_set(label, g_strdup(item->index));
		evas_object_show(label);

		return label;
	} else {
		return NULL;
	}
}

/*
 * @brief Callback function to release resource when item is removed in genlist object
 * @param[in] data The structure of app data
 * @param[in] obj    The object to set
 */
static void _genlist_item_del(void *data, Evas_Object *obj)
{
	printf("item(%d) is now deleted", (int)data);
}

/*
 * @brief The function to create a table on the parent object.
 * @param[in] parent The object to be set
 */
Evas_Object* _create_table(Evas_Object *parent)
{
	RETVM_IF(parent == NULL, NULL, "Invalid parent");
	Evas_Object *new_table = NULL;

	new_table = elm_table_add(parent);
	evas_object_size_hint_weight_set(new_table, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(new_table, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_table_padding_set(new_table, 1, 1);
	elm_table_homogeneous_set(new_table, EINA_TRUE);
	evas_object_show(new_table);

	return new_table;
}

/*
 * @brief The function to create a label on the parent object.
 * @param[in] parent The object to be set
 * @param[in] text The text to set
 */
Evas_Object* _create_label(Evas_Object *parent, const char *text)
{
	RETVM_IF(parent == NULL, NULL, "Invalid parent");
	Evas_Object *new_label = NULL;

	new_label = elm_label_add(parent);
	elm_object_text_set(new_label, (text) ? text : "-");
//	elm_label_text_style_user_push(new_label, USER_STYLE_LABEL);
	evas_object_size_hint_weight_set(new_label, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(new_label, EVAS_HINT_FILL, 0.5);
	evas_object_show(new_label);

	return new_label;
}

static Elm_Entry_Filter_Limit_Size limit_size = {
		.max_char_count = 8,
		.max_byte_count = 0
};

/*
 * @brief The function to create a entry on the parent object.
 * @param[in] parent The object to be set
 * @param[in] text The created entry name
 * @param[in] layout The created entry layout
 * @param[in] entry_activated_cb The callback when entry is activated
 * @param[in] user_data User data
 */
Evas_Object* _create_entry(Evas_Object *parent, const char *text, Elm_Input_Panel_Layout layout, Evas_Smart_Cb entry_activated_cb, void *user_data)
{
	RETVM_IF(parent == NULL, NULL, "Invalid parent");
	Evas_Object *new_entry = NULL;

	new_entry = elm_entry_add(parent);
	elm_entry_entry_set(new_entry, (text) ? text : "-");
	elm_entry_text_style_user_push(new_entry, USER_STYLE_ENTRY);
	evas_object_size_hint_weight_set(new_entry, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(new_entry, EVAS_HINT_FILL, 0.5);
	elm_entry_single_line_set(new_entry, EINA_TRUE);
	elm_entry_input_panel_layout_set(new_entry, layout);
	elm_entry_markup_filter_append(new_entry, elm_entry_filter_limit_size, &limit_size);
	elm_entry_cursor_end_set(new_entry);
	evas_object_smart_callback_add(new_entry, "activated", entry_activated_cb, user_data);
	evas_object_show(new_entry);

	return new_entry;
}

/*
 * @brief The function to create a combo box on the parent object.
 * @param[in] parent The object to be set
 * @param[in] items The list of items
 * @param[in] hover_item_num The length of the item list
 * @param[in] item_select_cb The callback when item is selected
 * @param[in] user_data User data
 */
Evas_Object* _create_hoversel(Evas_Object *parent, const char **items, unsigned int hover_items_num, Evas_Smart_Cb item_select_cb, void *user_data)
{
	RETVM_IF(parent == NULL, NULL, "Invalid parent");
	unsigned int i = 0;
	const char *item;
	Evas_Object *new_hoversel = NULL;

	new_hoversel = elm_hoversel_add(parent);
	elm_hoversel_hover_parent_set(new_hoversel, parent);
	elm_hoversel_horizontal_set(new_hoversel, EINA_FALSE);
	LOG_DBG("item count: %u", hover_items_num);

	/* add items into hoversel object */
	for (i = 0; i < hover_items_num; i++) {
		item = items[i];
		if (item == NULL)
			continue;
		LOG_DBG("item: %s", item);
		elm_hoversel_item_add(new_hoversel, _STYLE(item), NULL, ELM_ICON_NONE, item_select_cb, user_data);
	}

	/* set the first item as default text */
	elm_object_text_set(new_hoversel, _STYLE(items[0]));
	evas_object_size_hint_weight_set(new_hoversel, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(new_hoversel, EVAS_HINT_FILL, 0.5);
	evas_object_show(new_hoversel);

	return new_hoversel;
}

/*
 * @brief The function to create a combo box on the parent object.
 * @param[in] parent The object to be set
 * @param[in] list The list of items
 * @param[in] item_select_cb The callback when item is selected
 * @param[in] user_data User data
 */
Evas_Object* _create_hoversel_with_playlist(Evas_Object *parent, GList *list, Evas_Smart_Cb item_select_cb, void *user_data)
{
	RETVM_IF(parent == NULL, NULL, "Invalid parent");
	unsigned int i = 0;
	unsigned int hover_items_num = g_list_length(list);
	playlist_s *item;
	Evas_Object *new_hoversel = NULL;

	new_hoversel = elm_hoversel_add(parent);
	elm_hoversel_hover_parent_set(new_hoversel, parent);
	elm_hoversel_horizontal_set(new_hoversel, EINA_FALSE);

	LOG_DBG("item count: %u", hover_items_num);

	/* add items into hoversel object */
	for (i = 0; i < hover_items_num; i++) {
		item = g_list_nth_data(list, i);
		if (item == NULL)
			continue;
		LOG_DBG("item: %s", item->index);
		elm_hoversel_item_add(new_hoversel, _STYLE(item->index), NULL, ELM_ICON_NONE, item_select_cb, user_data);
	}

	/* set the first item as default text */
	if (hover_items_num == 0)
		elm_object_text_set(new_hoversel, _STYLE("Empty"));
	else {
		playlist_s *item = g_list_nth_data(list, 0);
		elm_object_text_set(new_hoversel, _STYLE(item->index));
	}

	evas_object_size_hint_weight_set(new_hoversel, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(new_hoversel, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_show(new_hoversel);

	return new_hoversel;
}

Evas_Object* _create_genlist(Evas_Object *parent, GList *list, Evas_Smart_Cb item_select_cb, void *user_data)
{
	RETVM_IF(parent == NULL, NULL, "Invalid parent");
	unsigned int i = 0;
	unsigned int hover_items_num = g_list_length(list);
	playlist_s *item;
	Evas_Object *new_genlist = NULL;
	Elm_Genlist_Item_Class *itc = elm_genlist_item_class_new();

	new_genlist = elm_genlist_add(parent);
	evas_object_size_hint_weight_set(new_genlist, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(new_genlist, EVAS_HINT_FILL, 0.5);
	evas_object_show(new_genlist);

	/* set the genlist as default value */
	itc->item_style = "default";
	itc->func.text_get = _genlist_item_label_get;
	itc->func.content_get = _genlist_item_content_get;
	itc->func.del = _genlist_item_del;
	LOG_DBG("item count: %u", hover_items_num);

	/* add items into genlist object */
	for (i = 0; i < hover_items_num; i++) {
		item = g_list_nth_data(list, i);
		if (item == NULL)
			continue;
		LOG_DBG("item: %s %s", item->index, item->metadata[META_TITLE]);
		elm_genlist_item_append(new_genlist, /* Genlist object */
								itc, /* Genlist item class */
								(void *)item, /* Item data */
								NULL, /* Parent item */
								ELM_GENLIST_ITEM_NONE, /* Item type */
								item_select_cb, /* Select callback */
								user_data); /* Callback data */
	}

	elm_genlist_item_class_free(itc);

	return new_genlist;
}

/*
 * @brief Creates the main view
 * @param[in] ad The structure of app data
 * @param[in] parent The object of the main view
 */
Evas_Object* mcs_create_content_view(appdata_s *ad, Evas_Object *parent)
{
	RETVM_IF(ad == NULL, NULL, "Invalid appdata");
	RETVM_IF(parent == NULL, NULL, "Invalid parent");
	unsigned int rows = 0;
	unsigned int i = 0;
	unsigned int category = 0;

	DEBUG_ENTER();

	content_view_s *view = &(ad->content_view);
	view->layout = _create_table(parent);

	elm_table_pack(view->layout, _create_label(view->layout, label_category[category++]), 0, rows, 1, 1);

	/* add metadata object */
	for (i = 0; i < LAST_META_ITEM; i++) {
		view->meta_item[i].label = _create_label(view->layout, label_metadata[i]);
		view->meta_item[i].obj = _create_label(view->layout, ad->mc_data.server_info.metadata[i]);

		/* pack the metadata object into table */
		elm_table_pack(view->layout, view->meta_item[i].label, 1, rows + i, 1, 1);
		elm_table_pack(view->layout, view->meta_item[i].obj, 2, rows + i, 1, 1);
	}

	/* add playback object */
	rows = rows + LAST_META_ITEM;
	elm_table_pack(view->layout, _create_label(view->layout, label_category[category++]), 0, rows, 1, 1);

	view->pb_item[PLAYBACK_STATE].label = _create_label(view->layout, "State");
	view->pb_item[PLAYBACK_STATE].obj = _create_hoversel(view->layout, hover_items_pb_state, hover_item_num_pb_state, _hover_playback_state_selected_cb, ad);

	view->pb_item[PLAYBACK_POSITION].label = _create_label(view->layout, "Position");
	view->pb_item[PLAYBACK_POSITION].obj = _create_hoversel(view->layout, hover_items_pb_position, hover_item_num_pb_position, _hover_playback_position_selected_cb, ad);

	view->pb_item[PLAYBACK_INDEX].label = _create_label(view->layout, "Index");
	view->pb_item[PLAYBACK_INDEX].obj = _create_hoversel_with_playlist(view->layout, ad->mc_data.server_info.playlist, _hover_playlist_selected_cb, ad);

	view->pb_item[PLAYBACK_SHUFFLE].label = _create_label(view->layout, "Shuffle");
	view->pb_item[PLAYBACK_SHUFFLE].obj = _create_hoversel(view->layout, hover_items_shuffle, hover_item_num_shuffle, _hover_shuffle_selected_cb, ad);

	view->pb_item[PLAYBACK_REPEAT].label = _create_label(view->layout, "Repeat");
	view->pb_item[PLAYBACK_REPEAT].obj = _create_hoversel(view->layout, hover_items_repeat, hover_item_num_repeat, _hover_repeat_selected_cb, ad);

	/* pack the playback object into table */
	elm_table_pack(view->layout, view->pb_item[PLAYBACK_STATE].label, 1, rows + PLAYBACK_STATE, LABEL_COLS, 1);
	elm_table_pack(view->layout, view->pb_item[PLAYBACK_STATE].obj, LABEL_COLS, rows + PLAYBACK_STATE, 1, 1);
	elm_table_pack(view->layout, view->pb_item[PLAYBACK_POSITION].label, 1, rows + PLAYBACK_POSITION, LABEL_COLS, 1);
	elm_table_pack(view->layout, view->pb_item[PLAYBACK_POSITION].obj, LABEL_COLS, rows + PLAYBACK_POSITION, 1, 1);
	elm_table_pack(view->layout, view->pb_item[PLAYBACK_INDEX].label, 1, rows + PLAYBACK_INDEX, LABEL_COLS, 1);
	elm_table_pack(view->layout, view->pb_item[PLAYBACK_INDEX].obj, LABEL_COLS, rows + PLAYBACK_INDEX, 1, 1);
	elm_table_pack(view->layout, view->pb_item[PLAYBACK_SHUFFLE].label, 1, rows + PLAYBACK_SHUFFLE, LABEL_COLS, 1);
	elm_table_pack(view->layout, view->pb_item[PLAYBACK_SHUFFLE].obj, LABEL_COLS, rows + PLAYBACK_SHUFFLE, 1, 1);
	elm_table_pack(view->layout, view->pb_item[PLAYBACK_REPEAT].label, 1, rows + PLAYBACK_REPEAT, LABEL_COLS, 1);
	elm_table_pack(view->layout, view->pb_item[PLAYBACK_REPEAT].obj, LABEL_COLS, rows + PLAYBACK_REPEAT, 1, 1);

	/* add custom command object */
	rows = rows + LAST_PLAYBACK_ITEM;
	elm_table_pack(view->layout, _create_label(view->layout, label_category[category++]), 0, rows, 1, 1);

	view->custom_item[CUSTOM_EVENT].label = _create_label(view->layout, label_custom[CUSTOM_EVENT]);
	view->custom_item[CUSTOM_EVENT].obj = _create_hoversel(view->layout, hover_items_event, hover_item_num_event, _hover_event_selected_cb, ad);
	view->custom_item[CUSTOM_EVENT_REPLY].label = _create_label(view->layout, label_custom[CUSTOM_EVENT_REPLY]);
	view->custom_item[CUSTOM_EVENT_REPLY].obj = _create_label(view->layout, NULL);
	view->custom_item[CUSTOM_COMMAND].label = _create_label(view->layout, label_custom[CUSTOM_COMMAND]);
	view->custom_item[CUSTOM_COMMAND].obj = _create_label(view->layout, NULL);

	/* pack the custom command object into table */
	elm_table_pack(view->layout, view->custom_item[CUSTOM_EVENT].label, 1, rows + CUSTOM_EVENT, LABEL_COLS, 1);
	elm_table_pack(view->layout, view->custom_item[CUSTOM_EVENT].obj, LABEL_COLS, rows + CUSTOM_EVENT, 1, 1);
	elm_table_pack(view->layout, view->custom_item[CUSTOM_EVENT_REPLY].label, 1, rows + CUSTOM_EVENT_REPLY, LABEL_COLS, 1);
	elm_table_pack(view->layout, view->custom_item[CUSTOM_EVENT_REPLY].obj, LABEL_COLS, rows + CUSTOM_EVENT_REPLY, 1, 1);
	elm_table_pack(view->layout, view->custom_item[CUSTOM_COMMAND].label, 1, rows + CUSTOM_COMMAND, LABEL_COLS, 1);
	elm_table_pack(view->layout, view->custom_item[CUSTOM_COMMAND].obj, LABEL_COLS, rows + CUSTOM_COMMAND, 1, 1);

	/* add playlist object */
	rows = rows + LAST_CUSTOM_ITEM;

	view->pl_item[PLAYLIST_NAME].label = _create_label(view->layout, "Playlist Name");
	view->pl_item[PLAYLIST_NAME].obj = _create_label(view->layout, ad->mc_data.server_info.playlist_name);

	/* pack the static playlist object into table */
	elm_table_pack(view->layout, view->pl_item[PLAYLIST_NAME].label, 0, rows + PLAYLIST_NAME, LABEL_COLS, 1);
	elm_table_pack(view->layout, view->pl_item[PLAYLIST_NAME].obj, LABEL_COLS, rows + PLAYLIST_NAME, 1, 1);

	elm_table_pack(view->layout, _create_label(view->layout, "Index"), 0, rows + PLAYLIST_INDEX, 1, 1);
	elm_table_pack(view->layout, _create_label(view->layout, label_metadata[META_ARTIST]), 1, rows + PLAYLIST_INDEX, 1, 1);
	elm_table_pack(view->layout, _create_label(view->layout, label_metadata[META_TITLE]), 2, rows + PLAYLIST_INDEX, 1, 1);

	/* pack the variable playlist object into table */
	if (ad->mc_data.server_info.playlist_name != NULL) {
		unsigned int length = g_list_length(ad->mc_data.server_info.playlist);
		for (i = 0; i < length; i++) {
			playlist_s *item = g_list_nth_data(ad->mc_data.server_info.playlist, i);

			elm_table_pack(view->layout, _create_label(view->layout, item->index), 0, rows + PLAYLIST_LIST + i, 1, 1);
			elm_table_pack(view->layout, _create_label(view->layout, item->metadata[META_ARTIST]), 1, rows + PLAYLIST_LIST + i, 1, 1);
			elm_table_pack(view->layout, _create_label(view->layout, item->metadata[META_TITLE]), 2, rows + PLAYLIST_LIST + i, 1, 1);
		}
	}

	DEBUG_LEAVE();

	return (view->layout);
}

/*
 * @brief Updates view of playback
 * @param[in] ad The structure of app data
 */
void mcs_update_view_playback(appdata_s *ad)
{
	unsigned int playback_state = 0;

	RETM_IF(ad == NULL, "Invalid appdata");

	DEBUG_ENTER();

	content_view_s *view = &(ad->content_view);

	/*
	 * convert playback_state due to deprecated enum
	 * typedef enum {
	 * MC_PLAYBACK_STATE_NONE = 0,
	 * MC_PLAYBACK_STATE_PLAYING,
	 * MC_PLAYBACK_STATE_PAUSED,
	 * MC_PLAYBACK_STATE_STOPPED,
	 * MC_PLAYBACK_STATE_MOVING_TO_NEXT = 8,
	 * MC_PLAYBACK_STATE_MOVING_TO_PREVIOUS,
	 * MC_PLAYBACK_STATE_FAST_FORWARDING,
	 * MC_PLAYBACK_STATE_REWINDING,
	 * } mc_playback_states_e;
	 */
	switch (ad->mc_data.server_info.pb_state) {
	case MC_PLAYBACK_STATE_PLAYING:
		playback_state = 1;
		break;
	case MC_PLAYBACK_STATE_PAUSED:
		playback_state = 2;
		break;
	case MC_PLAYBACK_STATE_STOPPED:
		playback_state = 3;
		break;
	case MC_PLAYBACK_STATE_MOVING_TO_NEXT:
		playback_state = 4;
		break;
	case MC_PLAYBACK_STATE_MOVING_TO_PREVIOUS:
		playback_state = 5;
		break;
	case MC_PLAYBACK_STATE_FAST_FORWARDING:
		playback_state = 6;
		break;
	case MC_PLAYBACK_STATE_REWINDING:
		playback_state = 7;
		break;
	case 0:
	default:
		playback_state = MC_PLAYBACK_STATE_NONE;
		break;
	}

	/* update playback information on UI */
	elm_object_text_set(view->pb_item[PLAYBACK_STATE].obj, _STYLE(hover_items_pb_state[playback_state]));
	elm_entry_entry_set(view->pb_item[PLAYBACK_POSITION].obj, g_strdup_printf("%llu", ad->mc_data.server_info.pb_position));
	elm_entry_entry_set(view->pb_item[PLAYBACK_INDEX].obj, _STYLE(ad->mc_data.server_info.playlist_index));
	elm_object_text_set(view->pb_item[PLAYBACK_SHUFFLE].obj, _STYLE(hover_items_shuffle[ad->mc_data.server_info.shuffle]));
	elm_object_text_set(view->pb_item[PLAYBACK_REPEAT].obj, _STYLE(hover_items_repeat[ad->mc_data.server_info.repeat]));

	DEBUG_LEAVE();
}

/*
 * @brief Updates view of metadata
 * @param[in] ad The structure of app data
 */
void mcs_update_view_metadata(appdata_s *ad)
{
	RETM_IF(ad == NULL, "Invalid appdata");
	unsigned int i = 0;

	DEBUG_ENTER();

	content_view_s *view = &(ad->content_view);

	/* update metadata information on UI */
	for (i = 0; i < LAST_META_ITEM; i++)
		elm_object_text_set(view->meta_item[i].obj, ad->mc_data.server_info.metadata[i]);

	/* update metadata view */
	mcs_update_metadata(ad);

	DEBUG_LEAVE();
}

/*
 * @brief Updates view of metadata in the playlist
 * @param[in] ad The structure of app data
 * @param[in] index The number of index in the playlist
 */
void mcs_update_view_metadata_by_index(appdata_s *ad, const char *index)
{
	RETM_IF(ad == NULL, "Invalid appdata");
	RETM_IF(index == NULL, "Invalid index");

	DEBUG_ENTER();

	unsigned int length = g_list_length(ad->mc_data.server_info.playlist);
	unsigned int j = 0;

	/* find metadata information by playlist index */
	for (j = 0; j < length; j++) {
		playlist_s *item = g_list_nth_data(ad->mc_data.server_info.playlist, j);

		if (g_strcmp0(ad->mc_data.server_info.playlist_index, item->index) == 0) {
			ad->mc_data.server_info.cur_playing_idx = j;
			unsigned int i = 0;
			for (i = 0; i < LAST_META_ITEM; i++) {
				SAFE_FREE(ad->mc_data.server_info.metadata[i]);
				ad->mc_data.server_info.metadata[i] = g_strdup(item->metadata[i]);
			}
		}
	}

	/* update metadata view */
	mcs_update_view_metadata(ad);

	DEBUG_LEAVE();
}

/*
 * @brief Updates view of custom command
 * @param[in] ad The structure of app data
 */
void mcs_update_view_custom_command(appdata_s *ad)
{
	/* check valid appdata */
	RETM_IF(ad == NULL, "Invalid appdata");

	DEBUG_ENTER();

	content_view_s *view = &(ad->content_view);

	elm_object_text_set(view->custom_item[CUSTOM_COMMAND].obj, ad->mc_data.custom_cmd);

	DEBUG_LEAVE();
}

/*
 * @brief Updates view of event reply
 * @param[in] ad The structure of app data
 */
void mcs_update_view_event_reply(appdata_s *ad)
{
	/* check valid appdata */
	RETM_IF(ad == NULL, "Invalid appdata");

	DEBUG_ENTER();

	content_view_s *view = &(ad->content_view);

	/* update event reply information on UI */
	elm_object_text_set(view->custom_item[CUSTOM_EVENT_REPLY].obj, g_strdup_printf("result_code: %d", ad->mc_data.event_result));

	DEBUG_LEAVE();
}

/*
 * @brief Clears view of event reply
 * @param[in] ad The structure of app data
 */
void mcs_clear_view_event_reply(appdata_s *ad)
{
	/* check valid appdata */
	RETM_IF(ad == NULL, "Invalid appdata");

	content_view_s *view = &(ad->content_view);
	RETM_IF(view->layout == NULL, "Invalid view");

	/* update event information on UI */
	elm_object_text_set(view->custom_item[CUSTOM_EVENT_REPLY].obj, NULL);
}
