/*
 *      Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 *      Licensed under the Flora License, Version 1.1 (the "License");
 *      you may not use this file except in compliance with the License.
 *      You may obtain a copy of the License at
 *
 *              http://floralicense.org/license/
 *
 *      Unless required by applicable law or agreed to in writing, software
 *      distributed under the License is distributed on an "AS IS" BASIS,
 *      WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *      See the License for the specific language governing permissions and
 *      limitations under the License.
 */

/*global pageHistory, pageDetail, util, WEEKDAY, MONTH*/

var pageDaily = (function() {
    var page = document.getElementById('page-daily');
    var scrollTimer = -1;
    var slideText = null;
    var slideTextList = null;

    /**
     * Shows the daily page.
     * @public
     */
    function show() {
        page.style.display = 'block';
    }

    /**
     * Hides the daily page.
     * @public
     */
    function hide() {
        page.style.display = 'none';
    }

    /**
     * Handles scroll end event.
     * @private
     */
    function scrollEndHandler() {
        var center = page.scrollTop + 180 - 83,
            text, len, i;
        len = slideTextList.length;
        if (slideText) {
            if (slideText.offsetTop <= center && slideText.offsetTop + 79 >= center) {
                return;
            } else {
                slideText.classList.remove('slide-text');
                slideText = null;
            }
        }
        for (i = 0; i < len; i++) {
            text = slideTextList[i];
            if (text.offsetTop <= center && text.offsetTop + 79 >= center) {
                text.classList.add('slide-text');
                slideText = text;
            }
        }
    }

    /**
     * Handles scroll event.
     * @private
     */
    function scrollHandler() {
        if (scrollTimer !== -1) {
            clearTimeout(scrollTimer);
        }
        scrollTimer = window.setTimeout(scrollEndHandler, 1000);
    }

    /**
     * activates the detail page.
     * @private
     */
    function activateDetailHandler(e) {
        hide();
        pageDetail.activate(e.currentTarget.key, e.currentTarget.id);
    }

    /**
     * Activates the daily page
     * @public
     */
    function activate() {
        var container = document.getElementById('daily-container'),
            events = util.getEvents(),
            today, year, month, date, len, data, dateElement, noEventElement, eventElement, nameContainerElement, nameElement, nameIcon,
            timeElement, eventDateObj, eventYear, eventMonth, eventDate, eventDay, eventList, key, i, j;

        slideTextList = [];
        pageHistory.push(pageDaily);
        show();

        // Gets today's year, month and date
        today = new Date();
        year = today.getFullYear();
        month = today.getMonth();
        date = today.getDate();

        while (container.hasChildNodes()) {
            container.removeChild(container.firstChild);
        }

        // Generates days that have events from today to 4 weeks later
        for (i = 0; i < 28; i++) {
            eventDateObj = new Date(year, month, date + i);
            eventYear = eventDateObj.getFullYear();
            eventMonth = eventDateObj.getMonth() + 1;
            eventDate = eventDateObj.getDate();
            eventDay = eventDateObj.getDay();

            // Generates key to get event data
            key = '' + eventYear;
            key += eventMonth > 9 ? '-' + eventMonth : '-0' + eventMonth;
            key += eventDate > 9 ? '-' + eventDate : '-0' + eventDate;

            // Gets the event list using the key value
            eventList = events[key];

            if (i === 0 && !eventList) {
                dateElement = document.createElement('div');
                dateElement.classList.add('daily-date');
                dateElement.textContent = WEEKDAY[eventDay] + ', ' + MONTH[eventMonth] + ' ' + eventDate;
                noEventElement = document.createElement('div');
                noEventElement.classList.add('no-event-name');
                noEventElement.textContent = 'No events today';
                container.appendChild(dateElement);
                container.appendChild(noEventElement);
            } else if (eventList) {
                len = eventList.length;
                dateElement = document.createElement('div');
                dateElement.classList.add('daily-date');
                dateElement.textContent = WEEKDAY[eventDay] + ', ' + MONTH[eventMonth] + ' ' + eventDate;
                container.appendChild(dateElement);
                for (j = 0; j < len; j++) {
                    data = eventList[j];
                    eventElement = document.createElement('div');
                    eventElement.classList.add('daily-event');
                    eventElement.id = data.id;
                    eventElement.key = key;
                    nameContainerElement = document.createElement('div');
                    nameContainerElement.classList.add('daily-event-name');
                    nameIcon = document.createElement('span');
                    nameIcon.classList.add('daily-event-name-icon');
                    nameIcon.style.backgroundColor = data.color;
                    nameElement = document.createElement('div');
                    nameElement.classList.add('daily-event-name-text');
                    if (data.declined) {
                        nameElement.classList.add('event-declined');
                    }
                    nameElement.textContent = data.name;
                    nameElement.appendChild(nameIcon);
                    nameContainerElement.appendChild(nameElement);
                    timeElement = document.createElement('div');
                    timeElement.classList.add('daily-event-time');
                    if (data.allDay) {
                        timeElement.textContent = 'All day';
                    } else {
                        timeElement.textContent = util.convertTimeFormat(data.startHours, data.startMinutes, data.endHours, data.endMinutes);
                    }
                    eventElement.appendChild(nameContainerElement);
                    eventElement.appendChild(timeElement);
                    container.appendChild(eventElement);

                    if (nameElement.offsetWidth > 360) {
                        slideTextList.push(nameElement);
                    }
                    eventElement.addEventListener('click', activateDetailHandler);
                }
            }
        }

        scrollHandler();
        page.addEventListener('scroll', scrollHandler);
    }

    return {
        show: show,
        hide: hide,
        activate: activate
    };
})();
