/*
 *      Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 *      Licensed under the Flora License, Version 1.1 (the "License");
 *      you may not use this file except in compliance with the License.
 *      You may obtain a copy of the License at
 *
 *              http://floralicense.org/license/
 *
 *      Unless required by applicable law or agreed to in writing, software
 *      distributed under the License is distributed on an "AS IS" BASIS,
 *      WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *      See the License for the specific language governing permissions and
 *      limitations under the License.
 */

/*global pageDaily, pageHistory, swipedetect, MONTH*/

var pageMonthly = (function() {
    var today = null;
    var currentYear = 0;
    var currentMonth = 0;
    var page = document.getElementById('page-monthly');

    /**
     * Shows the monthly page.
     * @public
     */
    function show() {
        page.style.display = 'block';
    }

    /**
     * Hides the monthly page.
     * @public
     */
    function hide() {
        page.style.display = 'none';
    }

    /**
     * Moves to the next month.
     * @public
     */
    function next() {
        var nextDate = new Date(currentYear, currentMonth);

        activate(nextDate.getFullYear(), nextDate.getMonth() + 1);
    }

    /**
     * Moves to the previous month.
     * @public
     */
    function prev() {
        var prevDate = new Date(currentYear, currentMonth - 2);

        activate(prevDate.getFullYear(), prevDate.getMonth() + 1);
    }

    /**
     * Handles the rotary event.
     * @private
     */
    function rotaryHandler(e) {
        if (pageHistory.getCurrentPage() === pageMonthly) {
            if (e.detail.direction === 'CW') {
                next();
            } else if (e.detail.direction === 'CCW') {
                prev();
            }
        }
    }

    /**
     * Handles the swipe event.
     * @private
     * @param {string} swipedir The direction of swipe event.
     */
    function swipeHandler(swipedir) {
        if (swipedir === 'up') {
            next();
        } else if (swipedir === 'down') {
            prev();
        }
    }

    /**
     * Handles the click event.
     * @private
     */
    function clickHandler() {
        if (today.getFullYear() === currentYear && today.getMonth() + 1 === currentMonth) {
            hide();
            pageDaily.activate();
        } else {
            today = new Date();
            activate(today.getFullYear() , today.getMonth() + 1, today.getDate());
        }
    }

    /**
     * Activates the monthly page.
     * @public
     */
    function activate(year, month) {
        var container = document.getElementById('days-container'),
            days = [],
            activeDate, firstDay, lastDate, count, el, i;

        if (pageHistory.getCurrentPage() !== pageMonthly) {
            pageHistory.push(pageMonthly);
        }
        show();
        // Gets current year and month
        today = new Date();
        currentYear = year ? year : today.getFullYear();
        currentMonth = month ? month : today.getMonth() + 1;

        while (container.hasChildNodes()) {
            container.removeChild(container.firstChild);
        }

        // Sets the title of calendar
        document.getElementById('monthly-title').textContent = MONTH[currentMonth] + ' ' + currentYear;

        // Sets the date to last date of current month
        activeDate = new Date(currentYear, currentMonth, 0);
        lastDate = activeDate.getDate();
        activeDate.setDate(1);
        firstDay = activeDate.getDay();

        for (i = 0; i < firstDay; i++) {
            el = document.createElement('div');

            el.classList.add('days');
            if (i === 0) {
                el.classList.add('sunday');
            }
            container.appendChild(el);
        }

        for (i = 1; i <= lastDate; i++) {
            var day = i + firstDay;

            el = document.createElement('div');
            el.classList.add('days');

            if (day % 7 === 1) {
                el.classList.add('sunday');
            } else if (day % 7 === 0) {
                el.classList.add('saturday');
            }
            el.textContent = i;
            container.appendChild(el);
            days[i] = el;
        }

        count = (42 - (firstDay + lastDate)) % 7;
        for (i = 1; i <= count; i++) {
            el = document.createElement('div');
            el.classList.add('days');
            if (i === count) {
                el.classList.add('saturday');
            }
            container.appendChild(el);
        }

        if (count + firstDay +lastDate === 35) {
            document.getElementById('calendar').style.paddingTop = '19px';
        } else if (count + firstDay + lastDate === 42) {
            document.getElementById('calendar').style.paddingTop = '3px';
        } else if (count + firstDay + lastDate === 28) {
            document.getElementById('calendar').style.paddingTop = '36px';
        }

        if (today.getFullYear() === currentYear && today.getMonth() + 1 === currentMonth) {
            days[today.getDate()].classList.add('today');
        }
    }

    page.addEventListener('click', clickHandler);
    swipedetect(page, swipeHandler);
    window.addEventListener('rotarydetent', rotaryHandler);

    return {
        show: show,
        hide: hide,
        next: next,
        prev: prev,
        activate: activate
    };
})();
