/*
 *      Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 *      Licensed under the Flora License, Version 1.1 (the "License");
 *      you may not use this file except in compliance with the License.
 *      You may obtain a copy of the License at
 *
 *              http://floralicense.org/license/
 *
 *      Unless required by applicable law or agreed to in writing, software
 *      distributed under the License is distributed on an "AS IS" BASIS,
 *      WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *      See the License for the specific language governing permissions and
 *      limitations under the License.
 */

/*exported swipedetect*/

/**
 * Detects the swipe event.
 * @public
 * @param {element} el The element to bind the swipe event.
 * @param {function} callback The function to be called.
 */
function swipedetect(el, callback){

    var touchsurface = el,
        threshold = 100, //required min distance traveled to be considered swipe
        restraint = 100, // maximum distance allowed at the same time in perpendicular direction
        allowedTime = 500, // maximum time allowed to travel that distance
        handleswipe = callback || function(){},
        swipedir, startX, startY, distX, distY, elapsedTime, startTime;

    touchsurface.addEventListener('touchstart', function(e){
        var touchobj = e.changedTouches[0];
        swipedir = 'none';
        startX = touchobj.pageX;
        startY = touchobj.pageY;
        // record time when finger first makes contact with surface
        startTime = new Date().getTime();
    }, false);

    touchsurface.addEventListener('touchmove', function(e){
        // prevent scrolling when inside DIV
        e.preventDefault();
    }, false);

    touchsurface.addEventListener('touchend', function(e){
        var touchobj = e.changedTouches[0];
        // get horizontal dist traveled by finger while in contact with surface
        distX = touchobj.pageX - startX;
        // get vertical dist traveled by finger while in contact with surface
        distY = touchobj.pageY - startY;
         // get time elapsed
        elapsedTime = new Date().getTime() - startTime;
        // first condition for awipe met
        if (elapsedTime <= allowedTime){
            // 2nd condition for horizontal swipe met
            if (Math.abs(distX) >= threshold && Math.abs(distY) <= restraint){
                // if dist traveled is negative, it indicates left swipe
                swipedir = (distX < 0)? 'left' : 'right';
            }
            else if (Math.abs(distY) >= threshold && Math.abs(distX) <= restraint){ // 2nd condition for vertical swipe met
                // if dist traveled is negative, it indicates up swipe
                swipedir = (distY < 0)? 'up' : 'down';
            }
        }
        handleswipe(swipedir);
    }, false);
}
