/*
 *      Copyright (c) 2016 Samsung Electronics Co., Ltd
 *
 *      Licensed under the Flora License, Version 1.1 (the "License");
 *      you may not use this file except in compliance with the License.
 *      You may obtain a copy of the License at
 *
 *              http://floralicense.org/license/
 *
 *      Unless required by applicable law or agreed to in writing, software
 *      distributed under the License is distributed on an "AS IS" BASIS,
 *      WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *      See the License for the specific language governing permissions and
 *      limitations under the License.
 */

/*
 * We must declare global variables to let the validator know the
 * global variables we use.
 * To declare global variables, write global variables along with
 * default values inside a comment on top of each source code.
 * The comment should start with the keyword, global.
 */

/* global WEEKDAY: true*/

(function() {
    /**
     * Convertes time format to string.
     * @private
     * @param {number} startHours The start hours.
     * @param {number} startMinutes The start minutes.
     * @param {number} endHours The end hours.
     * @param {number} endMinutes The end minutes.
     * @return {string} The converted time.
     */
    function convertTimeFormat(startHours, startMinutes, endHours, endMinutes) {
        var startTime = '',
            endTime = '',
            startAmPm = 'AM',
            endAmPm = 'AM';

        if (startHours) {
            // Converts 24 time format to 12 time format.
            if (startHours > 12) {
                startHours = startHours % 12;
                startAmPm = 'PM';
            }
            if (startMinutes < 10) {
                startMinutes = '0' + startMinutes;
            }
            startTime = startHours + ':' + startMinutes + startAmPm;
        }

        if (endHours) {
            // Converts 24 time format to 12 time format.
            if (endHours > 12) {
                endHours = endHours % 12;
                endAmPm = 'PM';
            }
            if (endMinutes < 10) {
                endMinutes = '0' + endMinutes;
            }
            endTime = endHours + ':' + endMinutes + endAmPm;
        }

        return startTime + ' - ' + endTime;
    }

    /**
     * Generates the event item.
     * @private
     * @param {string} color The color of the event.
     * @param {string} name The name of the event.
     * @param {string} time The time of the event.
     * @return {element} The generated item.
     */
    function generateItem(color, name, time) {
        var detail = document.createElement('div'),
            nameContainer = document.createElement('div'),
            nameText = document.createElement('span'),
            nameIcon = document.createElement('img'),
            timeElement;

        detail.classList.add('event-detail');
        nameContainer.classList.add('name');
        nameIcon.classList.add('name-icon');
        nameIcon.src = 'image/account_color_circle_' + color + '.png';
        nameText.classList.add('event-name');
        nameText.textContent = name;
        nameContainer.appendChild(nameIcon);
        nameContainer.appendChild(nameText);
        detail.appendChild(nameContainer);
        if (time) {
            timeElement = document.createElement('div');
            timeElement.classList.add('event-time');
            timeElement.textContent = time;
            detail.appendChild(timeElement);
        }

        return detail;
    }

    /**
     * Sets the events list
     * @private
     */
    function setEvents() {
        var events = getEvents(),
            today = new Date(),
            count = 0,
            allDayCount = 0,
            timeAssignCount = 0,
            eventList, year, month, date, day, list, listItem, time, key, noEvent, item, i;

        // Gets today's year, month, date and day
        year = today.getFullYear();
        month = today.getMonth() + 1;
        date = today.getDate();
        day = today.getDay();

        // Sets the today's date
        document.getElementById('event-date').textContent = date;
        // Sets the today's weekday
        document.getElementById('event-day').textContent = WEEKDAY[day];

        list = document.getElementById('event-list');

        // Generates key to get event data
        key = '' + year;
        key += month > 9 ? '-' + month : '-0' + month;
        key += date > 9 ? '-' + date : '-0' + date;

        // Gets the event list using the key value
        eventList = events[key];
        // Creates event list and adds it to page
        if (eventList) {
            count = eventList.length;
            // Counts the number of all day event and time assign event
            for (i = 0; i < count; i++) {
                item = eventList[i];
                if (item.allDay) {
                    allDayCount++;
                } else {
                    timeAssignCount++;
                }
            }
            if (count <= 2) {
                for (i = 0; i < count; i++) {
                    item = eventList[i];
                    if (item.allDay) {
                        time = 'All day';
                    } else {
                        time = convertTimeFormat(item.startHours, item.startMinutes, item.endHours, item.endMinutes);
                    }
                    listItem = generateItem(item.color, item.name, time);
                    list.appendChild(listItem);
                }
            } else if (allDayCount >= 2 && timeAssignCount >= 1) {
                listItem = generateItem(eventList[0].color, allDayCount + ' all-day events');
                list.appendChild(listItem);
                item = eventList[allDayCount];
                time = convertTimeFormat(item.startHours, item.startMinutes, item.endHours, item.endMinutes);
                listItem = generateItem(item.color, item.name, time);
                list.appendChild(listItem);

                if (timeAssignCount > 1) {
                    document.getElementById('event-more').textContent = '+' + (timeAssignCount - 1);
                }
            } else if (allDayCount >= 3) {
                item = eventList[0];
                listItem = generateItem(item.color, item.name, 'All day');
                list.appendChild(listItem);
                item = eventList[1];
                listItem = generateItem(item.color, (allDayCount - 1) + ' more all-day events');
                listItem.style.paddingLeft = '10px';
                listItem.style.paddingRight = '10px';
                list.appendChild(listItem);
            } else if (allDayCount === 1 && timeAssignCount > 2) {
                item = eventList[0];
                listItem = generateItem(item.color, item.name, 'All day');
                list.appendChild(listItem);
                item = eventList[allDayCount];
                time = convertTimeFormat(item.startHours, item.startMinutes, item.endHours, item.endMinutes);
                listItem = generateItem(item.color, item.name, time);
                list.appendChild(listItem);
                document.getElementById('event-more').textContent = '+' + (timeAssignCount - 1);
            } else if (allDayCount === 0 && timeAssignCount > 2) {
                item = eventList[0];
                time = convertTimeFormat(item.startHours, item.startMinutes, item.endHours, item.endMinutes);
                listItem = generateItem(item.color, item.name, time);
                list.appendChild(listItem);
                item = eventList[1];
                time = convertTimeFormat(item.startHours, item.startMinutes, item.endHours, item.endMinutes);
                listItem = generateItem(item.color, item.name, time);
                list.appendChild(listItem);
                document.getElementById('event-more').textContent = '+' + (timeAssignCount - 2);
            }
        }

        // If event list is not exist, add no event element to page
        if (count === 0) {
            noEvent = document.createElement('div');
            noEvent.classList.add('event-no');
            noEvent.textContent = 'No upcoming event';
            list.appendChild(noEvent);
        }
    }

    /**
     * Gets the event data from the schedule application using preference.
     * @private
     */
    function getEvents() {
        return JSON.parse(tizen.preference.getValue('events'));
    }

    /**
     * Launches the schedule application.
     * @private
     * @param {string} operation The action to be performed by an application control.
     */
    function launchApp(operation) {
        var app = window.tizen.application.getCurrentApplication();
        var appId = app.appInfo.id.substring(0, (app.appInfo.id.lastIndexOf('.')) );
        var appControl = new tizen.ApplicationControl(operation, null, null, null, null, null);
        window.tizen.application.launchAppControl(appControl, appId,
            function() {
                console.log("launch application control succeed");
            },
            function(e) {
                console.log("launch application control failed. reason: " + e.message);
            },
            null);
    }

    /**
     * Handles the back key event
     * @private
     * @param {Object} event The event object.
     */
    function keyEventHandler(event) {
        if (event.keyName === "back") {
            try {
                tizen.application.getCurrentApplication().exit();
            } catch (ignore) {}
        }
    }

    /**
     * Initializes the application
     * @private
     */
    function init() {
        setEvents();

        // Launches the schedule application when the page is clicked
        document.getElementById('page').addEventListener('click', launchApp.bind(this, 'schedule/daily'));
        // Adds eventListener for tizenhwkey
        window.addEventListener('tizenhwkey', keyEventHandler);
    }

    window.onload = init();
}());
