/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*global document, window, tizen, console*/

/**
 * App implementation.
 *
 * @namespace main
 */
(function strict() {
    'use strict';

    /**
     * Reference to HTML element.
     *
     * @type {HTMLElement}
     */
    var directionEl = null,

        /**
         * Reference to HTML element.
         *
         * @type {HTMLElement}
         */
        angleEl = null,

        /**
         * Reference to HTML element.
         *
         * @type {HTMLElement}
         */
        rotationEl = null,

        /**
         * Information about the physical orientation of the device.
         *
         * @type {DeviceOrientationEvent}
         */
        orientationData = null,

        /**
         * Unique index for time interval.
         *
         * @type {number}
         */
        orientationCheckInterval = null,

        /**
         * Counter for orientation check attempts.
         *
         * @type {number}
         */
        orientationCheckAttemptsLeft = 10,

         /**
         * Compass rotation degrees.
         *
         * @type {number}
         */
        rotation = 0;

    /**
     * Obtains angle from rotation.
     *
     * @param {number} value
     * @returns {number}
     */
    function angleFromRotation(value) {
        var angle = -value % 360;

        if (angle < 0) {
            angle += 360;
        }
        return angle;
    }

    /**
     * Handles device orientation data.
     *
     * @memberof main
     * @param {DeviceOrientationEvent} eventData Device orientation event.
     */
    function onDeviceOrientation(eventData) {
        var angle = eventData.alpha,
            lastAngle = angleFromRotation(rotation),
            deltaAngle = 0,
            text = '';

        orientationData = eventData;

        // Check if the angle has changed since the last measurement.
        if (isNaN(angle) || angle === lastAngle) {
            return;
        }

        // find direction from given angle
        if (angle < 68 || angle > 292) {
            text = 'N';
        } else if (angle > 112 && angle < 248) {
            text = 'S';
        }
        if (angle > 22 && angle < 158) {
            text += 'E';
        } else if (angle > 202 && angle < 338) {
            text += 'W';
        }

        // Calculate the rotation of the compass.
        deltaAngle = lastAngle - angle;
        if (deltaAngle > 180) {
            rotation -= 360 - deltaAngle;
        } else if (deltaAngle < -180) {
            rotation += 360 + deltaAngle;
        } else {
            rotation += deltaAngle;
        }

        // Update the direction and angle labels.
        directionEl.innerHTML = text;
        angleEl.innerHTML =
            Math.round(angle) + '<sup>o</sup>';
        // Rotate the compass image.
        rotationEl.style['-webkit-transform'] =
            'rotate(' + rotation + 'deg)';
    }

    /**
     * Exits the application.
     *
     * @memberof main
     */
    function exit() {
        try {
            tizen.application.getCurrentApplication().exit();
        } catch (err) {
            console.error('Error: ', err);
        }
    }

    /**
     * Handles no orientation support case.
     *
     * @memberof main
     */
    function onOrientationNotSupported() {
        window.alert('Device orientation\n is not supported\n on this device.');
        exit();
    }

    /**
     * Checks if orientation is supported by the device.
     *
     * @memberof main
     */
    function checkOrientationSupport() {
        orientationCheckInterval = window.setInterval(
            function checkOrientationSupport() {
                if (orientationData !== null &&
                    orientationData.alpha !== null) {
                    window.clearInterval(orientationCheckInterval);
                } else {
                    orientationCheckAttemptsLeft -= 1;
                    if (orientationCheckAttemptsLeft === 0) {
                        window.clearInterval(orientationCheckInterval);
                        onOrientationNotSupported();
                    }
                }
            }, 200);
    }

    /**
     * Binds events.
     *
     * @memberof main
     */
    function bindEvents() {
        window.addEventListener('tizenhwkey', function onTizenHWKey(e) {
            if (e.keyName === 'back') {
                exit();
            }
        });
        window.addEventListener(
            'deviceorientation',
            onDeviceOrientation,
            false
        );
    }

    /**
     * Initializes the application.
     *
     * @memberof main
     */
    function init() {
        directionEl = document.getElementById('direction');
        angleEl = document.getElementById('angle');
        rotationEl = document.getElementById('rotation');

        bindEvents();
        checkOrientationSupport();
    }

    // initialize application when the document is ready
    window.addEventListener('DOMContentLoaded', init);

}());
