/*
 * Command line interface
 *
 * Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact: 
 * BonYong Lee <bonyong.lee@samsung.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
 *
 */
package org.tizen.cli.command;

import static org.tizen.common.util.CollectionUtil.isEmpty;
import static org.tizen.common.util.StringUtil.isEmpty;

import java.io.IOException;
import java.util.Collection;
import java.util.Iterator;
import java.util.TreeSet;

import org.tizen.common.core.command.ExecutionContext;
import org.tizen.common.verrari.AttributeContainer;
import org.tizen.common.verrari.Condition;
import org.tizen.common.verrari.Condition.Operation;
import org.tizen.common.verrari.Realm;
import org.tizen.common.verrari.SearchableRealm;
import org.tizen.common.verrari.template.TemplateConstants;

public class SearchTemplate
extends AbstractCommand
implements CommandWithParameter
{

    protected void
    search(
        final Realm realm,
        final ExecutionContext context,
        final String[] args
    ) throws IOException
    {
        if ( !( realm instanceof SearchableRealm ) )
        {
            context.getPrompter().error( "Search unsupported realm." );
            return ;
        }
        
        final SearchableRealm searchable = (SearchableRealm) realm;
        final String keyword = args[0];
        final Collection<AttributeContainer> templates = searchable.search(
            new Condition( Operation.CaseInsensitiveContains, "keyword", keyword )
        );

        if ( isEmpty( templates ) )
        {
            context.getPrompter().notify( "No template with '" + keyword  + "'." );
            return ;
        }

        TreeSet<String> sortTree = new TreeSet<String>();
        for ( final AttributeContainer template : templates )
        {
            final StringBuilder buffer = new StringBuilder();
            final String name = template.getAttribute( TemplateConstants.ATTR_NAME );
            final String desc = template.getAttribute( TemplateConstants.ATTR_DESCRIPTION );
            final String version = template.getAttribute( TemplateConstants.ATTR_VERSION );

            buffer.append( name );
            if ( !isEmpty( version ) )
            {
                buffer.append( " ( " );
                buffer.append( version );
                buffer.append( " ) " );
            }
            
            buffer.append( " - " );
            if ( isEmpty( desc ) )
            {
                buffer.append( "undescribed" );
            }
            else
            {
                buffer.append( desc );
            }

            sortTree.add( buffer.toString() );
        }

        // search result notify
        Iterator<String> iter = sortTree.iterator();
        while ( iter.hasNext() ) {
            context.getPrompter().notify( iter.next() + "." );
        }
    }

    /* (non-Javadoc)
     * @see org.tizen.cli.command.CommandWithParameter#process(org.tizen.common.core.command.ExecutionContext, java.lang.String[])
     */
    @Override
    public void process(
        final ExecutionContext context,
        final String[] args
    ) throws IOException
    {
        if ( null == args || 0 == args.length )
        {
            context.getPrompter().error( "A keyword required." );
            return ;
        }
        final Realm realm = (Realm) context.getValue( PluginConstants.REALM );
        search( realm, context, args );
    }
}
