/*
 * Web IDE - Command Line Interface
 *
 * Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact: 
 * BonYong Lee <bonyong.lee@samsung.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
 *
 */
package org.tizen.cli.exec;

import static org.tizen.common.util.StringUtil.EMPTY_STRING;
import static org.tizen.common.util.StringUtil.nvl;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.List;

import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tizen.common.util.FilenameUtil;
import org.tizen.common.util.StringUtil;

/**
 * <p>
 * Help.
 * 
 * Execute to print out usage for command line
 * </p>
 * 
 * @author BonYong Lee{@literal <bonyong.lee@samsung.com>} (S-Core)
 */
public class Help
{
    
    /**
     * Logger for this instance
     */
    protected Logger logger = LoggerFactory.getLogger( getClass() );
    
    /**
     * Program name property key
     */
    protected static final String PROP_PRG = "cli.name";
    
    //TODO Other Main classes already modified syntax like: $(script_name) [options].
    //Therefore, we cannot use syntax using getSyntax()
    private static final String DEFAULT_SYNTAX = (System.getProperty( PROP_PRG ) != null) ?
            FilenameUtil.getTailingPath( System.getProperty( PROP_PRG ), 1 ) : StringUtil.EMPTY_STRING;
    
    /**
     * {@link Options} for Help print-out
     */
    protected final Options opts;
    
    
    /**
     * Syntax for Help print out
     */
    protected String syntax;
    
    /**
     * <p>
     * keyword - help mapping
     * </p>
     */
    protected HashMap<String, String> keyword2help = new HashMap<String, String>();
    
    /**
     * commands for Help print-out
     */
    protected final Options cmds;
    
    /**
     * a command user inputs.
     */
    protected final String inputCmd;
    
    /**
     * a description for usage.
     */
    protected final String usageDescription;
    
    /**
     * Constructor with {@link Options}
     * 
     * @param opts {@link Options}
     */
    public
    Help(
        final Options opts
    )
    {
        this(opts, null, "", "");
    }
    
    /**
     * A constructor with options, commands, input command and usage description.
     * 
     * @param opts
     * @param cmds
     * @param inputCmd
     * @param usageDes
     */
    public
    Help(
            final Options opts, 
            final Options cmds, 
            final String inputCmd, 
            final String usageDes)
    {
        this.opts = opts;
        this.cmds = cmds;
        this.inputCmd = inputCmd;
        this.usageDescription = usageDes;
    }
    
    /**
     * {@link Options} for usage
     * 
     * @return defined {@link Options}
     */
    protected
    Options
    getOptions()
    {
        return opts;
    }
    
    /**
     * Add <code>help</code> contents for <code>keyword</code>
     * 
     * @param keyword keyword
     * @param help contents
     */
    public
    void
    addHelpDetail(
        final String keyword,
        final String help
    )
    {
        keyword2help.put( keyword.toLowerCase(), help );
        
        logger.info( "Help Detail added for {} :\n{}", keyword, help );
    }

    
    /**
     * syntax for usage
     * 
     * @param syntax contents
     */
    public
    void
    setSyntax( String syntax )
    {
        this.syntax = syntax;
    }
    
    /**
     * <p>
     * Return execution command for usage
     * </p>
     * 
     * @return execution command
     */
    public
    String
    getSyntax()
    {
        return nvl(
            this.syntax,
            FilenameUtil.getTailingPath( System.getProperty( PROP_PRG ), 1 ),
            "java " + getClass().getName()
        );
    }

    
    /**
     * <p>
     * Return default usage
     * </p>
     * 
     * @return usage
     */
	public String getHelp()
    {
        StringBuffer helpBuffer = new StringBuffer();
        
        helpBuffer.append("\n");
        getUsage(helpBuffer);
        helpBuffer.append("\n");
        
        boolean cmdDes = getCmds() != null  & StringUtil.isEmpty(this.getInputCmd());
        if(cmdDes) {
            getCommandDescription(helpBuffer);
            helpBuffer.append("\n");
        }
        
        if(getOptions().getOptions().size() > 0) {
            getOptionDescription(helpBuffer);
            helpBuffer.append("\n");
        }
        
        if(cmdDes) {
            getHelpUsage(helpBuffer);
            helpBuffer.append("\n");
        }
        
        return helpBuffer.toString();
    }
    
	/**
	 * <p>
	 * Returns a usage for help.
	 * </p> 
	 * 
	 * @param buffer to store help usage
	 * @return help usage
	 */
    public StringBuffer getHelpUsage(StringBuffer buffer) {
        String syntax = nvl(DEFAULT_SYNTAX, getSyntax());
        
        buffer.append(String.format("For more information on a specific command: '%s <command> -h'" , syntax));
        return buffer;
    }

    /**
     * <p>
     * Returns a command description.
     * </p>
     * 
     * @param buffer to store a command description
     * @return a command description
     */
    public StringBuffer getCommandDescription(StringBuffer buffer) {
        Options cmds = getCmds();
        buffer.append("Commands:\n");
        getOptionDescriptionWithPrefix(cmds, StringUtil.EMPTY_STRING, StringUtil.EMPTY_STRING, buffer);
        
        return buffer;
    }

    /**
     * <p>
     * Returns a option description.
     * </p>
     * 
     * @param buffer to store a option description
     * @return a option description
     */
    public StringBuffer getOptionDescription(StringBuffer buffer) {
        Options opts = getOptions();
        buffer.append("Options:\n");
        getOptionDescriptionWithPrefix(opts, "-", "--", buffer);
        
        return buffer;
    }
    
    /**
     * <p>
     * Returns a option description with option prefix and long option prefix.
     * </p>
     * 
     * @param opts options description will describe about
     * @param optPrefix option prefix
     * @param longOptPrefix long option prefix
     * @param buffer to store a option description
     * @return
     */
    public StringBuffer getOptionDescriptionWithPrefix(Options opts, String optPrefix, String longOptPrefix, StringBuffer buffer) {
        
        if(opts == null || opts.getOptions().size() == 0) {
            return buffer;
        }
        
        final StringWriter writer = new StringWriter();
        final PrintWriter stream = new PrintWriter( writer );
        final HelpFormatter formatter = new HelpFormatter();
        
        formatter.setOptPrefix(optPrefix);
        formatter.setLongOptPrefix(longOptPrefix);
        formatter.printOptions(
                stream,
                formatter.getWidth(),
                opts,
                formatter.getLeftPadding(),
                formatter.getDescPadding()
        );
        
        return buffer.append(writer.toString());
    }
    
    /**
     * Returns a usage.
     * 
     * @param usageBuffer to store usage
     * @return a usage
     */
    public StringBuffer getUsage(StringBuffer usageBuffer) {
        String syntax = nvl(DEFAULT_SYNTAX, getSyntax());
        
        usageBuffer.append(String.format("Usage: %s ", syntax));
        Options opts = getOptions();
        Options cmds = getCmds();
        String inputCmd = getInputCmd();
        
        String description = "";
        if(cmds != null) {
            Option cmd = cmds.getOption(inputCmd);
            
            if(cmd != null) {
                usageBuffer.append(inputCmd + " ");
                description = cmd.getDescription();
            }
            else {
                usageBuffer.append("<command> ");
                description = usageDescription;
            }
        }
        int optionSize = opts.getOptions().size();
        int reqOptSize = opts.getRequiredOptions().size();
        
        if(optionSize > 0 && optionSize != reqOptSize) {
            usageBuffer.append("[options] ");
        }
        
        getRequiredOptionInUsage(usageBuffer);
        usageBuffer.append(String.format("\n%s\n", description));
        return usageBuffer;
    }
    
    private StringBuffer getRequiredOptionInUsage(StringBuffer buffer) {
        Options opts = getOptions();
        @SuppressWarnings("unchecked")
        List<String> requiredOptKeys = opts.getRequiredOptions();
        
        for(String key: requiredOptKeys) {
            Option opt = opts.getOption(key);
            buffer.append(String.format("-%s ", opt.getOpt()));
            
            if(opt.hasArg()) {
                buffer.append(String.format("<%s> ", opt.getArgName()));
            }
        }
        
        return buffer;
    }
    
    public Options getCmds() {
        return cmds;
    }
    
    public String getInputCmd() {
        return inputCmd;
    }

    /**
     * <p>
     * Return detail help for <code>keyword</code>
     * </p>
     * 
     * @param keyword keyword
     * @return detail
     */
    public String
    getHelpDetail(
        final String keyword
    )
    {
        if ( null == keyword )
        {
            return EMPTY_STRING;
        }
        return nvl( keyword2help.get( keyword.toLowerCase() ) );
    }

}
