/*
 * Web IDE - Command Line Interface
 *
 * Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact: 
 * BonYong Lee <bonyong.lee@samsung.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
 *
 */
package org.tizen.cli.exec.template;

import static org.apache.commons.cli.OptionBuilder.withLongOpt;
import static org.tizen.common.util.ArrayUtil.isEmpty;
import static org.tizen.common.util.StringUtil.isEmpty;
import static org.tizen.common.util.StringUtil.nvl;

import java.net.URL;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Options;
import org.tizen.cli.command.CreateTemplate;
import org.tizen.cli.command.PluginConstants;
import org.tizen.cli.command.SearchTemplate;
import org.tizen.cli.exec.AbstractLauncher;
import org.tizen.cli.util.TemplateModelProvider;
import org.tizen.common.config.Preference;
import org.tizen.common.verrari.Realm;
import org.tizen.common.verrari.RealmFactory;
import org.tizen.common.verrari.TemplateContext;

/**
 * Command Line Interface to create from template
 * 
 * @author BonYong Lee{@literal <bonyong.lee@samsung.com>} (S-Core)
 */
public class
ClientMain
extends AbstractLauncher
{
    protected static final String OPT_REALM = "realm";
    
    protected static final String DESC_REALM = "Specify realm for template";
    
    protected static final String OPT_SEARCH = "search";
    
    protected static final String DESC_SEARCH = "Search template with keyword";
    
    
    /**
     * entry point
     * 
     * @param args input argument
     * 
     * @throws Exception If unhandled exception
     */
    public static
    void
    main(
        final String[] args
    )
    throws Exception
    {
        new ClientMain().run( args );
    }
    
    /* (non-Javadoc)
     * @see org.tizen.cli.exec.AbstractLauncher#getOptions()
     */
    @SuppressWarnings("static-access")
    @Override
    protected Options getOptions()
    {
        final Options ret = super.getOptions();
        
        ret.addOption( withLongOpt( OPT_REALM ).withDescription( DESC_REALM ).hasArg().create( OPT_REALM.substring( 0, 1 ) ) );
        ret.addOption( withLongOpt( OPT_SEARCH ).withDescription( DESC_SEARCH ).hasArg().create( OPT_SEARCH.substring( 0, 1 ) ) );
        
        return ret;
    }

    /* (non-Javadoc)
     * @see org.tizen.cli.exec.AbstractLauncher#execute(org.apache.commons.cli.CommandLine)
     */
    @Override
    protected
    void
    execute(
        final CommandLine cmdLine
    )
    throws Exception
    {
        String urlStr = nvl( Preference.getValue( "tizen.template.realm", cmdLine.getOptionValue( OPT_REALM ) ) );
        logger.info( "Realm url :{}", urlStr );
        if ( isEmpty( urlStr ) )
        {
            getPrompter().error( "No realm!! Specify realm with --realm option." );
            return ;
        }
        
        TemplateContext.getInstance().setModelProvider( new TemplateModelProvider( getExecutionContext() ) );
        
        final URL url = new URL( urlStr );
        Realm realm = null;
        try
        {
            realm = RealmFactory.getInstance().create( url );
        }
        catch ( final Throwable e )
        {
            logger.error( "Error: ", e );
            getPrompter().error( "Invalid realm!! Check realm and its contents: " + urlStr );
            return ;
        }
        
        getExecutionContext().setValue( PluginConstants.REALM, realm );
        
        if ( cmdLine.hasOption( OPT_SEARCH ) )
        {
            
            if ( !isEmpty( cmdLine.getArgs() ) )
            {
                printHelp();
                return ;
            }
            final SearchTemplate command = new SearchTemplate();
            command.process( getExecutionContext(), new String[] { cmdLine.getOptionValue( OPT_SEARCH ) } );
            
        }
        else
        {
            if ( isEmpty( cmdLine.getArgs() ) )
            {
                printHelp();
                return ;
            }
            
            final CreateTemplate command = new CreateTemplate();
            command.process( getExecutionContext(), cmdLine.getArgs() );
        }
        
    }

}
