/*
* Web IDE - Command Line Interface
*
* Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
*
* Contact:
* GyeongSeok Seo <gyeongseok.seo@samsung.com>
* BonYong Lee <bonyong.lee@samsung.com>
* 
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
* Contributors:
* - S-Core Co., Ltd
*
*/
package org.tizen.cli.exec;

import java.util.List;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.tizen.common.core.application.InstallPathConfig;
import org.tizen.common.core.command.ExecutionContext;
import org.tizen.common.core.command.Prompter;
import org.tizen.common.util.HostUtil;
import org.tizen.common.util.OSChecker;
import org.tizen.common.util.StringUtil;
import org.tizen.sdblib.IDevice;
import org.tizen.sdblib.SmartDevelopmentBridge;

/**
 * <p>
 * WRTLauncher.
 * 
 * class for simple WRT command.
 * 
 * provide {@link ExecutionContext} for command line
 * </p>
 * 
 * @author GyeongSeok Seo{@literal <gyeongseok.seo@samsung.com>} (S-Core)
 * @author BonYong Lee{@literal <bonyong.lee@samsung.com>} (S-Core)
 */
public class
WRTLauncher
extends AbstractLauncher
{
    /**
     * <p>
     * Option for select device
     * </p>
     */
    protected static final String OPT_DEVICE = "device";

    /**
     * <p>
     * Description for select device
     * 
     * This is printed out in usage
     * </p>
     * 
     * @see #OPT_DEVICE
     */
    protected static final String DESC_DEVICE = "Command to the given device";

    /**
     * <p>
     * Option for wait for the sdb demon connection and command completion.
     * </p>
     */
    protected static final String OPT_TIME = "time";

    /**
     * <p>
     * Description for wait for the sdb demon connection and command completion.
     * 
     * This is printed out in usage
     * </p>
     * 
     * @see #OPT_TIME
     */
    protected static final String DESC_TIME = "Waiting time for sdb daemon connection and command completion (sec)";

    /**
     * <p>
     * Target device
     * 
     * command is running in this device
     * </p>
     */
    protected IDevice device = null;

    /**
     * Get target device
     * 
     * @return device
     */
    public IDevice getDevice() {
        return device;
    }

    /**
     * Set target device
     * 
     * @param device
     */
    public void setDevice(IDevice device) {
        this.device = device;
    }

    /**
     * command and sdb daemon connection wait time
     */
    protected int time = 90;

    /**
     * Get command and sdb daemon connection wait time
     * 
     * @return time
     */
    public int getTime() {
        return time;
    }

    /**
     * Set command and sdb daemon connection wait time
     * 
     * @param time
     */
    public void setTime(int time) {
        this.time = time;
    }

    /**
     * {@link Options} for usage
     * 
     * @return defined {@link Options}
     */
    @SuppressWarnings("static-access")
    protected
    Options
    getOptions()
    {
        final Options opts = super.getOptions();
        opts.addOption( OptionBuilder.hasArg().withLongOpt( OPT_DEVICE ).withDescription( DESC_DEVICE ).create( OPT_DEVICE.substring( 0, 1 ) ) );
        opts.addOption( OptionBuilder.hasOptionalArg().withLongOpt( OPT_TIME ).withDescription( DESC_TIME ).create( OPT_TIME.substring( 0, 1 ) ) );

        return opts;
    }

    protected void startSDB() {
        final Prompter prompter = getPrompter();
        String sdbPath = InstallPathConfig.getSDBPath();
        logger.trace( "sdb lib Path :{}", sdbPath );

        SmartDevelopmentBridge bridge = SmartDevelopmentBridge.createBridge(sdbPath, true);

        if ( null == bridge ) {
            prompter.notify( "Problem occurred while initializing sdb" );
        }
    }

    protected void endSDB() {
        SmartDevelopmentBridge.terminate();
    }

	/**
	 * Terminates the currently running command line interface.<p>
	 * The argument serves as a status code, by convention, a nonzero status code indicates abnormal termination.<br>
	 * Before the exit, this method terminate SDB bridge.
	 * This method calls the System.exit method.
	 * 
	 * @param status exit status.
	 */
    @Override
    protected void exit(int status) {
        endSDB();
        super.exit(status);
    }

    protected String getDeviceNotConnectedMessage() {
        return "Device is not connected. Process is canceled.";
    }
    protected String getDeviceNotSelectedMessage() {
        return "Process is stoped, please select connect device.";
    }

    /* (non-Javadoc)
     * @see org.tizen.cli.exec.AbstractLauncher#execute(org.apache.commons.cli.CommandLine)
     */
    @Override
    @SuppressWarnings("unchecked")
    protected void execute(CommandLine cmdLine) throws Exception {
        startSDB();
        final Prompter prompter = getPrompter();
        final List<String> args = cmdLine.getArgList();
        final TargetDeviceSelector deviceSelector = new TargetDeviceSelector(prompter);
        String time = cmdLine.getOptionValue( OPT_TIME );
        logger.trace( "option input waiting time value:{} sec", time );
        if ( !StringUtil.isEmpty( time ) ) {
            setTime( Integer.parseInt(time) );
            deviceSelector.setSdbtime( getTime()*1000 );
        }
        logger.trace( "waiting time value:{} sec", getTime() );
        logger.trace( "arguments :{}", args );

        logger.trace("check connected devices");
        if ( !deviceSelector.isDeviceConnected() ) {
            prompter.notify( getDeviceNotConnectedMessage() );
            exit( 1 );
        }

        String deviceName = cmdLine.getOptionValue( OPT_DEVICE );
        logger.trace( "Device option :{}", deviceName );

        if ( !StringUtil.isEmpty( deviceName ) ) {
            setDevice( deviceSelector.getDevice( deviceName ) );
            if ( getDevice() == null ) {
                prompter.notify( getDeviceNotSelectedMessage() );
                exit( 1 );
            }
        } else {
            // user not using device select option
            logger.trace( "user can't use device select option" );
            setDevice( deviceSelector.selectDevice() );
        }
    }
}
