/*
* Web IDE - Command Line Interface
*
* Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
*
* Contact:
* GyeongSeok Seo <gyeongseok.seo@samsung.com>
* BonYong Lee <bonyong.lee@samsung.com>
* 
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
* Contributors:
* - S-Core Co., Ltd
*
*/
package org.tizen.cli.exec.uninstall;

import java.text.MessageFormat;

import static org.tizen.cli.exec.LaunchOptionConstants.DESC_FAILED_CANNOT_FIND_CONNECTED_DEVICE;
import static org.tizen.cli.exec.LaunchOptionConstants.DESC_FAILED_CANNOT_FIND_SELECTED_DEVICE;
import static org.tizen.cli.exec.LaunchOptionConstants.OPT_WIDGETID;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.tizen.cli.exec.Help;
import org.tizen.cli.exec.LaunchOptionConstants;
import org.tizen.cli.exec.WRTLauncher;
import org.tizen.common.core.command.Executor;
import org.tizen.common.core.command.Prompter;
import org.tizen.common.sdb.command.message.ErrorMessageType;
import org.tizen.web.zimlaunch.cli.command.CheckInstallCommand;
import org.tizen.web.zimlaunch.cli.command.KillCommand;
import org.tizen.web.zimlaunch.cli.command.UninstallCommand;

/**
 * Command Line Interface for uninstall widget in target
 * 
 * @author GyeongSeok Seo{@literal <gyeongseok.seo@samsung.com>} (S-Core)
 * @author BonYong Lee{@literal <bonyong.lee@samsung.com>} (S-Core)
 */
public class
Main
extends WRTLauncher
{
    protected String FAILED_UNINSTALL = "Failed to uninstall: ";

    /**
     * <p>
     * Description for target widget id
     * 
     * This is printed out in usage
     * </p>
     * 
     * @see #OPT_WIDGETID
     */
    protected static final String DESC_WIDGETID = LaunchOptionConstants.DESC_WIDGETID;

    /**
     * Entry point for cli main
     * 
     * @param args user input parameter
     * 
     * @throws Exception If unhandled exception occur
     */
    public static
    void
    main(
        final String[] args
    )
    throws Exception
    {
        final Main instance = new Main();
        instance.run( args );
    }

    /* (non-Javadoc)
     * @see org.tizen.cli.exec.AbstractLauncher#execute(org.apache.commons.cli.CommandLine)
     */
    @Override
    protected
    void
    execute(
        final CommandLine cmdLine
    )
    throws Exception
    {
        super.execute( cmdLine );

        final Prompter prompter = getPrompter();
        final Executor exec = getExecutor();

        // core operation
        String widgetID = cmdLine.getOptionValue( OPT_WIDGETID );

        // install check
        CheckInstallCommand install_check_command = new CheckInstallCommand( widgetID );
        install_check_command.setDevice( getDevice() );
        install_check_command.setTime( getTime() );
        install_check_command.setPrintOption( false );
        exec.execute( install_check_command );
        if ( !install_check_command.isInstalled() ) {
            prompter.notify(
                    MessageFormat.format( "Uninstall failed, ''{0}'' widget is not installed.", widgetID )
                    );
            exit( 1 );
        }

        // kill
        KillCommand kill_command = new KillCommand( widgetID );
        kill_command.setDevice( getDevice() );
        kill_command.setTime( getTime() );
        exec.execute( kill_command );

        // uninstall
        UninstallCommand uninstall_command = new UninstallCommand( widgetID );
        uninstall_command.setDevice( getDevice() );
        uninstall_command.setTime( getTime() );
        exec.execute( uninstall_command );

        String[] result = uninstall_command.getResult().split("\\s+");
        int returnCode = Integer.parseInt(result[result.length-1]);
        if (returnCode != 0) {
            ErrorMessageType type = ErrorMessageType.getErrorType(returnCode);
            prompter.notify(type.getManagement());
            exit(returnCode);
        }
    }

    /* (non-Javadoc)
     * @see org.tizen.cli.exec.AbstractLauncher#getOptions()
     */
    @Override
    @SuppressWarnings("static-access")
    protected
    Options
    getOptions()
    {
        final Options opts = super.getOptions();
        opts.addOption( OptionBuilder.hasArg().isRequired().withLongOpt( OPT_WIDGETID ).withDescription( DESC_WIDGETID ).create( OPT_WIDGETID.substring( 0, 1 ) ) );

        return opts;
    }

    /* (non-Javadoc)
     * @see org.tizen.cli.exec.WRTLauncher#getDeviceNotConnectedMessage()
     */
    @Override
    protected String getDeviceNotConnectedMessage() {
        return FAILED_UNINSTALL+ DESC_FAILED_CANNOT_FIND_CONNECTED_DEVICE;
    }

    /* (non-Javadoc)
     * @see org.tizen.cli.exec.WRTLauncher#getDeviceNotSelectedMessage()
     */
    @Override
    protected String getDeviceNotSelectedMessage() {
        return FAILED_UNINSTALL+ DESC_FAILED_CANNOT_FIND_SELECTED_DEVICE;
    }

    /* (non-Javadoc)
     * @see org.tizen.cli.exec.AbstractLauncher#createHelp()
     */
    @Override
    protected Help createHelp() {
        Help help = super.createHelp();

        help.setSyntax( help.getSyntax() + getSyntax() );
        return help;
    }

    /* (non-Javadoc)
     * @see org.tizen.cli.exec.AbstractLauncher#getSyntax()
     */
    @Override
    protected
    String
    getSyntax()
    {
        return " [options] --id widget-id";
    }
}
