/*
 * Web IDE - Command Line Interface
 *
 * Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact:
 * Taeyoung Son <taeyoung2.son@samsung.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
 *
 */
package org.tizen.cli.exec.web.build;

import java.io.File;
import java.io.IOException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.apache.tools.ant.BuildException;
import org.tizen.cli.exec.AbstractLauncher;
import org.tizen.cli.exec.Help;
import org.tizen.common.FactoryWithArgument;
import org.tizen.common.builder.BuildProcess;
import org.tizen.common.builder.Builder;
import org.tizen.common.builder.Resource;
import org.tizen.common.builder.ResourceLayer;
import org.tizen.common.builder.core.CopyBuilder;
import org.tizen.common.file.FileHandler;
import org.tizen.common.file.FileHandler.Attribute;
import org.tizen.common.file.Filter;
import org.tizen.common.file.SimpleFileFilter;
import org.tizen.common.file.StandardFileHandler;
import org.tizen.common.file.VirtualFileHandler;
import org.tizen.common.file.filter.WildCardFilterFactory;
import org.tizen.common.util.FileUtil;
import org.tizen.common.util.FilenameUtil;
import org.tizen.web.builder.CssMinifyBuilder;
import org.tizen.web.builder.HybridAppCLIBuilder;
import org.tizen.web.builder.JSMinifyBuilder;
import org.tizen.web.builder.UIFWBuilder;
import org.tizen.web.builder.exception.MinifyException;
import org.tizen.web.common.WebConstant;

/**
 * Command Line Interface for optimization of web resources
 *
 * @author Taeyoung Son{@literal <taeyoung2.son@samsung.com>} (S-Core)
 */
public class
Main
extends AbstractLauncher
{

    // messages
    private static final String SUCCEED_MSG = "Build completed. In sequence you can sign on your build output folder(default:{0}) using the web-signing command.";
    
    private String cwd = ".";
    private String output = ".buildResult"; //default output directory

    private SimpleFileFilter filter = new SimpleFileFilter(true);
    private FactoryWithArgument<Filter, String> filterFactory = new WildCardFilterFactory();

    // optimize resources option
    private static final String OPT_NAME_OPTIMIZE = "optimize";
    private static final String OPT_DESC_OPTIMIZE = "Minify resources (js, css)";

    // exclude ui framework and use target's original library option
    private static final String OPT_NAME_EXCLUDE_UI_FRAMEWORK = "exclude-uifw";
    private static final String OPT_NAME_EXCLUDE_UI_FRAMEWORK_SHORT = "euf";
    private static final String OPT_DESC_EXCLUDE_UI_FRAMEWORK = "If you want to use Tizen UI Framework in target, use -" + OPT_NAME_EXCLUDE_UI_FRAMEWORK_SHORT + " option.\nThis option is only for application using Tizen UI Framework.";

    // exclude ui framework and use target's minified library option
    private static final String OPT_NAME_EXCLUDE_UI_FRAMEWORK_USEMIN = "exclude-uifw-min";
    private static final String OPT_NAME_EXCLUDE_UI_FRAMEWORK_USEMIN_SHORT = "eufm";
    private static final String OPT_DESC_EXCLUDE_UI_FRAMEWORK_USEMIN = "If you want to use minified Tizen UI Framework in target, use -" + OPT_NAME_EXCLUDE_UI_FRAMEWORK_USEMIN_SHORT + " option.";

    // excluding files option
    private static final String OPT_NAME_EXCLUDE = "exclude";
    private static final String OPT_DESC_EXCLUDE = "Add exclude file pattern. By default, following resources will be excluded:\n.build*, .project, .settings, .sdk_delta.info, *.wgt";

    // set output option
    private static final String OPT_NAME_OUTPUT = "output";
    private static final String OPT_DESC_OUTPUT = "Set output directory for build resources. By default, '.buildResult' is used.";

    // set reference project option
    private static final String OPT_NAME_REFERENCE_PROJECT = "ref-prj";
    private static final String OPT_NAME_REFERENCE_PROJECT_SHORT = "rp";
    private static final String OPT_DESC_REFERENCE_PROJECT = "Set reference project.\nThe reference project must be a Tizen Native project. This option is only for hybrid application.";

    // ResourceLayer
    protected static final String RESOURCE_LAYER_START = "start";
    protected static final String RESOURCE_LAYER_END = "end";

    // file filter. If you want to exclude a directory, append '/*'
    private static final String[] DEFAULT_EXCLUDES = {".build"+File.separator+"*"
                                                    , ".project"
                                                    , ".settings" + File.separator + "*"
                                                    , ".sdk_delta.info"
                                                    , "*.wgt"};

    // excluding file list
    private List<String> excludes = new ArrayList<String>();

    /**
     * Entry point for cli main
     * 
     * @param args user cwd parameter
     * 
     * @throws Exception If unhandled exception occur
     */
    public static
    void
    main(
            final String[] args
            )
                    throws Exception
                    {
        final Main instance = new Main();
        instance.run( args );
                    }

    /* (non-Javadoc)
     * @see org.tizen.cli.exec.AbstractLauncher#getOptions()
     */
    @Override
    @SuppressWarnings("static-access")
    protected
    Options
    getOptions()
    {
        final Options opts = super.getOptions();
        opts.addOption( OptionBuilder.withLongOpt( OPT_NAME_OPTIMIZE ).withDescription( OPT_DESC_OPTIMIZE ).create( OPT_NAME_OPTIMIZE.substring( 0, 1 ) ) );
        opts.addOption( OptionBuilder.withLongOpt( OPT_NAME_EXCLUDE_UI_FRAMEWORK ).withDescription( OPT_DESC_EXCLUDE_UI_FRAMEWORK ).create( OPT_NAME_EXCLUDE_UI_FRAMEWORK_SHORT ) );
        opts.addOption( OptionBuilder.withLongOpt( OPT_NAME_EXCLUDE_UI_FRAMEWORK_USEMIN ).withDescription( OPT_DESC_EXCLUDE_UI_FRAMEWORK_USEMIN ).create( OPT_NAME_EXCLUDE_UI_FRAMEWORK_USEMIN_SHORT ) );
        opts.addOption( OptionBuilder.hasArg().withLongOpt( OPT_NAME_EXCLUDE ).withDescription( OPT_DESC_EXCLUDE ).create( OPT_NAME_EXCLUDE.substring( 0, 1 ) ) );
        opts.addOption( OptionBuilder.hasArg().withLongOpt( OPT_NAME_REFERENCE_PROJECT ).withDescription( OPT_DESC_REFERENCE_PROJECT ).create( OPT_NAME_REFERENCE_PROJECT_SHORT ) );
        opts.addOption( OptionBuilder.hasArg().withLongOpt( OPT_NAME_OUTPUT ).withDescription( OPT_DESC_OUTPUT ).create() );

        return opts;
    }

    /* (non-Javadoc)
     * @see org.tizen.cli.exec.AbstractLauncher#execute(org.apache.commons.cli.CommandLine)
     */
    @Override
    @SuppressWarnings("unchecked")
    protected
    void
    execute(
            final CommandLine cmdLine
            )
                    throws Exception
                    {
        final List<String> args = cmdLine.getArgList();
        logger.trace( "arguments :{}", args );

        // input path's argument will be set.
        if (args.size() < 1) {
            printHelp();
            return;
        }

        // set absolute path
        cwd = convertPath(args.get(0));

        logger.debug( "Base directory :{}", cwd );

        // set output path (absolute)
        if (cmdLine.hasOption(OPT_NAME_OUTPUT)) {
            output = convertPath(cmdLine.getOptionValue(OPT_NAME_OUTPUT));
        } else {
            output = cwd+File.separator+output;
        }

        logger.debug( "Output directory :{}", output );

        // If output path exist, the path will be recreated.
        if (getFileHandler().is(output, Attribute.EXISTS)) {
            getFileHandler().removeDirectory( output );
            logger.info( "{} directory deleted", output);
        }

        // Set exclude file list
        initExcludes(cmdLine);

        BuildProcess buildProcess = new BuildProcess();

        // Set start layer
        FileHandler fh = new StandardFileHandler();
        fh.setCurrentWorkingDirectory(cwd);
        ResourceLayer startLayer = new ResourceLayer(RESOURCE_LAYER_START, fh);
        
        // Get resource list for build
        Resource[] resources = getBuildResources(cmdLine
                                                , startLayer
                                                , null
                                                , excludes.toArray(new String[excludes.size()]));

        // Generate web builders to build process
        generateAllBuilders(cmdLine, buildProcess, startLayer);
        if (buildProcess.getLastBuilder() != null) {
            logger.debug("start build process");
            try {
                buildProcess.build(resources);
            } catch (MinifyException e) {
                StringBuffer msg = new StringBuffer();
                msg.append("Optimization failed.\n");
                msg.append("Error: "+ e.getPath()+"("+e.getLineNumber()+"): "+e.getLineSource()+"\n");
                msg.append("Cause: "+e.getMessage()+"\n");
                getPrompter().notify(msg.toString());
                return;
            }

            // check succeeding status.
            checkProcessComplete( buildProcess );
        }
    }

    /**
     * Check whether this process have no problem.
     * If have problems, BuildException will be occurred.
     * 
     * @param buildProcess
     * @throws BuildException
     */
    protected void checkProcessComplete(BuildProcess buildProcess) throws BuildException {
        // Check hybrid structure. If a manifest file and a binary file are not found, this is a failure.
        try {
            Builder builder = buildProcess.getBuilder( HybridAppCLIBuilder.RESOURCE_LAYER_NAME );
            if ( builder != null ) {
                HybridAppCLIBuilder hybridBuilder = (HybridAppCLIBuilder) builder;
                if ( ! hybridBuilder.isManifestFound() ) {
                    throw new IOException( HybridAppCLIBuilder.ERROR_MANIFEST_NOT_FOUND );
                }
                if ( ! hybridBuilder.isBinaryFound() ) {
                    throw new IOException( HybridAppCLIBuilder.ERROR_BINARY_NOT_FOUND );
                }
            }
        } catch( IOException e ) {
            throw new BuildException( e );
        }
        
        getPrompter().notify( MessageFormat.format( SUCCEED_MSG, this.output ) );
    }
    
    private void initExcludes(CommandLine cmdLine) {
        this.excludes.clear();

        for (String exclude : DEFAULT_EXCLUDES) {
            this.excludes.add(exclude);
        }

        if (cmdLine.hasOption(OPT_NAME_EXCLUDE)) {
            String[] excludes = cmdLine.getOptionValues(OPT_NAME_EXCLUDE);
            for (String exclude : excludes) {
                this.excludes.add(exclude);
            }
        }

        if (isExcludeUIFW(cmdLine)) {
            this.excludes.add(WebConstant.TIZEN_WEB_UI_FRAMWORK_LIBRARY_DIRECTORY+File.separator+"*");
        }
    }

    /**
     * Generate builders to {@link BuildProcess}.
     * @param cmdLine
     * @param buildProcess
     * @param parentLayer
     * @throws IOException
     */
    public void generateAllBuilders(CommandLine cmdLine
                                            , BuildProcess buildProcess
                                            , ResourceLayer parentLayer) throws IOException {

        FileHandler fh = new VirtualFileHandler();
        fh.setCurrentWorkingDirectory(parentLayer.getFileHandler().getCurrentWorkingDirectory());

        if (isExcludeUIFW(cmdLine)) {
            addUIFWBuilder(true
                            , cmdLine.hasOption(OPT_NAME_EXCLUDE_UI_FRAMEWORK_USEMIN)
                            , buildProcess
                            , getLastResourceLayer(buildProcess, parentLayer)
                            , fh);
        }

        boolean hasReference = cmdLine.hasOption(OPT_NAME_REFERENCE_PROJECT);
        if (hasReference) {
            addHybridBuilder( cmdLine
                                , buildProcess
                                , getLastResourceLayer(buildProcess, parentLayer)
                                , fh);
        }

        if (cmdLine.hasOption(OPT_NAME_OPTIMIZE)) {
            addOptimizingBuilders(buildProcess
                                    , getLastResourceLayer(buildProcess, parentLayer)
                                    , fh);
        }

        addLastbuilder(buildProcess, getLastResourceLayer(buildProcess, parentLayer));
    }

    /**
     * Exclude Tizen web UI framework or not.
     * If excluding UI framework option is true, it return true. Or return false.
     * @param cmdLine
     * @return
     */
    private boolean isExcludeUIFW(CommandLine cmdLine) {
        if (cmdLine.hasOption(OPT_NAME_EXCLUDE_UI_FRAMEWORK)
                || cmdLine.hasOption(OPT_NAME_EXCLUDE_UI_FRAMEWORK_USEMIN)) {
            return true;
        }
        return false;
    }

    /**
     * Add hybrid builder to buildProcess. It will be used to build including reference project.
     * @param cmdLine
     * @param buildProcess
     * @param lastLayer
     * @throws IOException
     */
    private void addHybridBuilder(CommandLine cmdLine, BuildProcess buildProcess, ResourceLayer lastLayer, FileHandler fh) throws IOException {
        ResourceLayer layer = new ResourceLayer(HybridAppCLIBuilder.RESOURCE_LAYER_NAME
                                                    , lastLayer
                                                    , fh);
        
        String refPath = convertPath(cmdLine.getOptionValue(OPT_NAME_REFERENCE_PROJECT));
        
        HybridAppCLIBuilder hybridAppBuilder = new HybridAppCLIBuilder( layer, refPath );
        buildProcess.addBuilder(hybridAppBuilder);
    }

    /**
     * get last resource layer located in buildProcess.
     * @param buildProcess
     * @param defaultLayer
     * @return
     */
    private ResourceLayer getLastResourceLayer(BuildProcess buildProcess, ResourceLayer defaultLayer) {
        ResourceLayer lastLayer = buildProcess.getLastResourceLayer();
        return (lastLayer != null) ? lastLayer : defaultLayer;
    }

    /**
     * Add UI framework builder to buildProcess.
     * @param isExclude If set true, Tizen web UI framework library will be excluded output path.
     *                  Application will be used target's Tizen web UI framework library.
     * @param useMin If 'isExclude' parameter set true and this parameter set true,
     *                  application will be used target's minified Tizen web UI framework library.
     *                  Or, application will be used target's original Tizen web UI framework library.
     * @param buildProcess
     * @param parentLayer
     */
    private void addUIFWBuilder(boolean isExclude, boolean useMin, BuildProcess buildProcess,
            ResourceLayer parentLayer, FileHandler fh) {
        ResourceLayer layer = new ResourceLayer(UIFWBuilder.RESOURCE_LAYER_NAME
                                                    , parentLayer
                                                    , fh);
        UIFWBuilder uifwBuilder = new UIFWBuilder(isExclude, layer);
        uifwBuilder.setUseMin(useMin);

        buildProcess.addBuilder(uifwBuilder);
    }

    /**
     * Add builder that write resources to file system to buildProcess.
     * @param buildProcess
     * @param parentLayer
     */
    private void addLastbuilder(BuildProcess buildProcess, ResourceLayer parentLayer) {
        // add file output builder
        FileHandler fh = new StandardFileHandler();
        fh.setCurrentWorkingDirectory(parentLayer.getFileHandler().getCurrentWorkingDirectory());
        ResourceLayer toFSLayer = new ResourceLayer(CopyBuilder.RESOURCE_LAYER_NAME
                                                        , parentLayer
                                                        , fh);
        CopyBuilder toFSBuilder = new CopyBuilder(FilenameUtil.getRelativePath(cwd, output), toFSLayer);
        buildProcess.addBuilder(toFSBuilder);
    }

    /**
     * Get resource list for build.
     * @param cmdLine
     * @param layer parent layer
     * @param includes Include resource list. If you want to add all resources located in input path, set this parameter to null.
     *                  If you set this parameter, another resources will not be added to list.
     * @param excludes Exclude resource list. If you want to exclude resources, set this parameter using regular expression.
     *                  It should be set regular expression about file element.
     * @return
     * @throws IOException
     */
    private Resource[] getBuildResources(CommandLine cmdLine
                                        , ResourceLayer layer
                                        , String[] includes
                                        , String[] excludes) throws IOException {
        List<Resource> resources = new ArrayList<Resource>();
        setIncludes(includes);
        setExcludes(excludes);

        String cwd = layer.getFileHandler().getCurrentWorkingDirectory();
        resources.addAll(getResources(cwd
                                        , FilenameUtil.getRelativePath(cwd, cwd)
                                        , layer));

        if (cmdLine.hasOption(OPT_NAME_REFERENCE_PROJECT)) {
            FileHandler refFh = new StandardFileHandler();
            ResourceLayer referenceLayer = new ResourceLayer(HybridAppCLIBuilder.RESOURCE_LAYER_REFERENCE_NAME
                                                                , refFh);
            layer.setParent(referenceLayer);
            String refPath = convertPath(cmdLine.getOptionValue(OPT_NAME_REFERENCE_PROJECT));
            refFh.setCurrentWorkingDirectory(refPath);

            resources.addAll(getResources(refPath
                                            , FilenameUtil.getRelativePath(refPath, refPath)
                                            , referenceLayer));
        }

        return resources.toArray(new Resource[resources.size()]);
    }

    /**
     * get resource list
     * @param cwd root path located resources. It means working path. means current path.
     * @param relativeSrcDir relative directory including resources.
     * @param layer {@link ResourceLayer} about resource
     * @throws IOException
     */
    private List<Resource> getResources(String cwd, String relativeSrcDir, ResourceLayer layer) throws IOException {
        List<Resource> resources = new ArrayList<Resource>();
        List<File> files = null;
        String absoluteDir = cwd+ File.separator + relativeSrcDir;
        files = FileUtil.findFiles(new File(absoluteDir), ".*", true);

        for (File file : files) {
            String fileRelativePath = FilenameUtil.getRelativePath(cwd, file.getCanonicalPath());

            if (!FilenameUtil.equals(file.getCanonicalPath(), this.cwd)
                    && !filter.accept(cwd, fileRelativePath)) {
                logger.debug("Ignore {}", fileRelativePath);
                continue;
            }

            logger.debug("resource path : {}", fileRelativePath);
            Resource resource = new Resource(layer, fileRelativePath);

            resources.add(resource);
        }
        return resources;
    }

    /**
     * Set including resource list
     * @param includes
     */
    private void setIncludes(String[] includes) {
        filter.clearIncludes();
        if (null != includes) {
            for (final String include : includes) {
                filter.addIncludes(filterFactory.create(include));
            }
        }
    }

    /**
     * Set excluding resource list
     * @param excludes
     */
    private void setExcludes(String[] excludes) {
        filter.clearExcludes();
        if (null != excludes) {
            for (final String exclude : excludes) {
                filter.addExcludes(filterFactory.create(exclude));
            }
        }
    }

    /**
     * Add optimizing builders to buildProcess.
     * @param buildProcess
     * @param parentLayer
     */
    private void addOptimizingBuilders(BuildProcess buildProcess, ResourceLayer parentLayer, FileHandler fh) {
        if (buildProcess == null) {
            return;
        }

        // add js minify builder
        ResourceLayer jsMinLayer = new ResourceLayer(JSMinifyBuilder.RESOURCE_LAYER_NAME
                                                        , parentLayer
                                                        , fh);
        JSMinifyBuilder jsMin = new JSMinifyBuilder(jsMinLayer);
        buildProcess.addBuilder(jsMin);
        
        // add css minify builder
        ResourceLayer cssMinLayer = new ResourceLayer(CssMinifyBuilder.RESOURCE_LAYER_NAME
                                                        , jsMinLayer
                                                        , fh);
        CssMinifyBuilder cssMin = new CssMinifyBuilder(cssMinLayer);
        buildProcess.addBuilder(cssMin);
    }

    @Override
    protected Help createHelp() {
        return new Help(getOptions()) {
            @Override
            public StringBuffer getUsage(StringBuffer usageBuffer) {
                usageBuffer = super.getUsage(usageBuffer);

                String[] usageFrag = usageBuffer.toString().split("\\s");
                usageBuffer.delete(0, usageBuffer.length());

                for (String frag : usageFrag) {
                    usageBuffer.append(frag+" ");
                    if (frag.equals(getSyntax())) {
                        usageBuffer.append("<build target path> ");
                    }
                }
                return usageBuffer;
            }
        };
    }
}
