/*
* Native IDE - Command Line Interface
*
* Copyright (c) 2000 - 2013 Samsung Electronics Co., Ltd. All rights reserved.
*
* Contact:
* Ho Namkoong <ho.namkoong@samsung.com>
* BonYong Lee <bonyong.lee@samsung.com>
* Kangho Kim <kh5325.kim@samsung.com>
* 
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
* Contributors:
* - S-Core Co., Ltd
*
*/

package org.tizen.cli.exec.pack;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.text.MessageFormat;
import java.util.ArrayList;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.tizen.cli.exec.AbstractLauncher;
import org.tizen.common.core.application.InstallPathConfig;
import org.tizen.common.core.command.CommandCancelException;
import org.tizen.common.core.command.Prompter;
import org.tizen.common.util.FileUtil;
import org.tizen.common.util.IOUtil;
import org.tizen.common.util.OSChecker;
import org.tizen.common.util.StringUtil;

public class NativeMain extends AbstractLauncher{
    
    private CommandLine cmdLine;
    
    private static final String PATH_SBI = FileUtil.appendPath(InstallPathConfig.getSDKPath(), "tools/smart-build-interface/bin");
    private static final String CMD_CLI_NATIVE_PACK_LINUX = FileUtil.appendPath(PATH_SBI, "sbi_pack");
    private static final String CMD_CLI_NATIVE_PACK_WINDOWS = FileUtil.appendPath(PATH_SBI, "sbi-pack.bat");
    
    private static final String OPTION_PLATFORM_NAME = "name";
    private static final String OPTION_PLATFORM_VERSION = "version";
    private static final String OPTION_ARCHITECTURE = "arch";
    private static final String OPTION_TOOLCHAIN = "toolchain";
    private static final String OPTION_SIGN_AK = "sign-author-key";
    private static final String OPTION_SIGN_APWD = "sign-author-pwd";
    private static final String OPTION_SIGN_DK = "sign-dist2-key";
    private static final String OPTION_SIGN_DPWD = "sign-dist2-pwd";
    private static final String OPTION_SIGN_DCA = "sign-dist2-ca";
    private static final String OPTION_SIGN_DROOT = "sign-dist2-root";
    private static final String OPTION_REF_PROJECTS = "ref-prj";
    
    private static final String DESC_PLATFORM_NAME = Messages.getString("NativeMain.DESC_PLATFORM_NAME"); //$NON-NLS-1$
    private static final String DESC_PLATFORM_VERSION = Messages.getString("NativeMain.DESC_PLATFORM_VERSION"); //$NON-NLS-1$
    private static final String DESC_ARCHITECTURE = Messages.getString("NativeMain.DESC_ARCH"); //$NON-NLS-1$
    private static final String DESC_TOOLCHAIN = Messages.getString("NativeMain.DESC_TOOLCHAIN"); //$NON-NLS-1$
    private static final String DESC_SIGN_AK = Messages.getString("NativeMain.DESC_APK"); //$NON-NLS-1$
    private static final String DESC_SIGN_APWD = Messages.getString("NativeMain.DESC_APW"); //$NON-NLS-1$
    private static final String DESC_SIGN_DK = Messages.getString("NativeMain.DESC_DPK"); //$NON-NLS-1$
    private static final String DESC_SIGN_DPWD = Messages.getString("NativeMain.DESC_DPW"); //$NON-NLS-1$
    private static final String DESC_SIGN_DCA = Messages.getString("NativeMain.DESC_DCA"); //$NON-NLS-1$
    private static final String DESC_SIGN_DROOT = Messages.getString("NativeMain.DESC_DROOT"); //$NON-NLS-1$
    private static final String DESC_REF_PROJECTS = Messages.getString("NativeMain.DESC_RP"); //$NON-NLS-1$
    
    private static final String[] OPTION_REMAINING = {
        "sign_cert_dir", "sign_dist1_key", "sign_dist1_pwd", "sign_dist1_ca"
    };

    public static void main(String[] args) throws Exception {
        
        final NativeMain instance = new NativeMain();
        instance.run( args );
    }

    @Override
    protected void execute(CommandLine cmdLine) throws Exception {
        
        Prompter prompter = this.getPrompter();
        
        this.cmdLine = cmdLine;
        
        logger.trace("get user input");
        
        if(StringUtil.isEmpty(this.cmdLine.getOptionValue(OPTION_SIGN_AK)) ||
                StringUtil.isEmpty(this.cmdLine.getOptionValue(OPTION_SIGN_APWD))) {
            prompter.error(MessageFormat.format(Messages.getString("NativeMain.EXCEPTION_MANDATORY_MISSING"), OPTION_SIGN_AK, OPTION_SIGN_APWD)); //$NON-NLS-1$
            return;
        }
        
        String rowRefPrjs = this.cmdLine.getOptionValue(OPTION_REF_PROJECTS);
        
        if(rowRefPrjs != null) {
                String[] refPrjPaths = rowRefPrjs.split(",");
                CLIMultiPackager packager = new CLIMultiPackager("../", refPrjPaths, getExecutor());
            try {
                packager.doMultiPreProcess();
            }
            catch (CommandCancelException e) {
                packager.removeMutilAppResources();
                prompter.error(e.getMessage());
                prompter.error(Messages.getString("NativeMain.CANCELD")); //$NON-NLS-1$
                return;
            }
        }
        
        ArrayList<String> cmdList = new ArrayList<String>();
        
        if(OSChecker.isWindows()) {
            cmdList.add(CMD_CLI_NATIVE_PACK_WINDOWS);
        }
        else {
            cmdList.add(CMD_CLI_NATIVE_PACK_LINUX);
        }
        
        appendOption(cmdList, OPTION_PLATFORM_NAME);
        appendOption(cmdList, OPTION_PLATFORM_VERSION);
        appendOption(cmdList, OPTION_ARCHITECTURE);
        appendOption(cmdList, OPTION_TOOLCHAIN);
        appendOption(cmdList, OPTION_SIGN_AK);
        appendOption(cmdList, OPTION_SIGN_APWD);
        appendOption(cmdList, OPTION_SIGN_DK);
        appendOption(cmdList, OPTION_SIGN_DPWD);
        appendOption(cmdList, OPTION_SIGN_DCA);
        appendOption(cmdList, OPTION_SIGN_DROOT);
        
        appendRemaningOptions(cmdList);
        
        Process cmdP = Runtime.getRuntime().exec(cmdList.toArray(new String[cmdList.size()]));
        
        String line;
        
        InputStream is = null;
        InputStream eis = null;
        try {
            is = cmdP.getInputStream();
            BufferedReader outReader = new BufferedReader(new InputStreamReader(is));
            while((line = outReader.readLine()) != null ) {
                prompter.notify(line);
            }
            
            eis = cmdP.getErrorStream();
            BufferedReader errReader = new BufferedReader(new InputStreamReader(eis));
            while((line = errReader.readLine()) != null) {
                prompter.error(line);
            }
        }
        finally {
            IOUtil.tryClose(is, eis);
        }
    
    }

    private void appendRemaningOptions(ArrayList<String> cmdList) {
        for(String remaningOption: OPTION_REMAINING) {
            cmdList.add("--" + convertToShellOption(remaningOption));
            cmdList.add(" ");
        }
    }

    private void appendOption(ArrayList<String> cmdList, String optionTarget) {
        String optionValue = cmdLine.getOptionValue(optionTarget);
        if(!StringUtil.isEmpty(optionValue)) {
            cmdList.add("--" + convertToShellOption(optionTarget));
            cmdList.add(optionValue);
        }
    }
    
    private String convertToShellOption(String origin) {
        return origin.replace('-', '_');
    }
    
    @SuppressWarnings("static-access")
    @Override
    protected Options getOptions() {
        Options result = super.getOptions();
        result.addOption(OptionBuilder.hasArg().withLongOpt(OPTION_SIGN_AK).withDescription(DESC_SIGN_AK).isRequired().create("ak"));
        result.addOption(OptionBuilder.hasArg().withLongOpt(OPTION_SIGN_APWD).withDescription(DESC_SIGN_APWD).isRequired().create("ap"));
        result.addOption(OptionBuilder.hasArg().withLongOpt(OPTION_PLATFORM_NAME).withDescription(DESC_PLATFORM_NAME).create("n"));
        result.addOption(OptionBuilder.hasArg().withLongOpt(OPTION_PLATFORM_VERSION).withDescription(DESC_PLATFORM_VERSION).create("v"));
        result.addOption(OptionBuilder.hasArg().withLongOpt(OPTION_ARCHITECTURE).withDescription(DESC_ARCHITECTURE).create("a"));
        result.addOption(OptionBuilder.hasArg().withLongOpt(OPTION_TOOLCHAIN).withDescription(DESC_TOOLCHAIN).create("t"));
        result.addOption(OptionBuilder.hasArg().withLongOpt(OPTION_SIGN_DK).withDescription(DESC_SIGN_DK).create("dk"));
        result.addOption(OptionBuilder.hasArg().withLongOpt(OPTION_SIGN_DPWD).withDescription(DESC_SIGN_DPWD).create("dp"));
        result.addOption(OptionBuilder.hasArg().withLongOpt(OPTION_SIGN_DCA).withDescription(DESC_SIGN_DCA).create("dc"));
        result.addOption(OptionBuilder.hasArg().withLongOpt(OPTION_SIGN_DROOT).withDescription(DESC_SIGN_DROOT).create("dr"));
        result.addOption(OptionBuilder.hasArg().withLongOpt(OPTION_REF_PROJECTS).withDescription(DESC_REF_PROJECTS).create("rp"));
        
        return result;
    }
    
    @Override
    protected boolean isPrintHelp() {
        return true;
    }
    
}
