/*
 * Web IDE - Command Line Interface
 *
 * Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact: 
 * BonYong Lee <bonyong.lee@samsung.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
 *
 */
package org.tizen.cli.exec.wgt;

import static org.tizen.cli.exec.LaunchOptionConstants.DESC_NOCHECK;
import static org.tizen.cli.exec.LaunchOptionConstants.DESC_OVERWRITE;
import static org.tizen.cli.exec.LaunchOptionConstants.OPT_NOCHECK;
import static org.tizen.cli.exec.LaunchOptionConstants.OPT_OVERWRITE;
import static org.tizen.common.util.FilenameUtil.getRelativePath;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.tizen.cli.exec.AbstractLauncher;
import org.tizen.cli.exec.Help;
import org.tizen.common.FactoryWithArgument;
import org.tizen.common.builder.Resource;
import org.tizen.common.builder.ResourceLayer;
import org.tizen.common.core.command.Executor;
import org.tizen.common.core.command.Prompter;
import org.tizen.common.core.command.prompter.ChoiceOption;
import org.tizen.common.core.command.prompter.Option;
import org.tizen.common.core.command.zip.ZipCommand;
import org.tizen.common.file.FileHandler;
import org.tizen.common.file.FileHandler.Attribute;
import org.tizen.common.file.Filter;
import org.tizen.common.file.SimpleFileFilter;
import org.tizen.common.file.VirtualFileHandler;
import org.tizen.common.file.filter.WildCardFilterFactory;
import org.tizen.common.util.CollectionUtil;
import org.tizen.common.util.FileUtil;
import org.tizen.common.util.FilenameUtil;
import org.tizen.common.util.IOUtil;
import org.tizen.web.common.WebConstant;

/**
 * Command Line Interface for creation of wgt file
 * 
 * @author BonYong Lee{@literal <bonyong.lee@samsung.com>} (S-Core)
 */
public class
Main
extends AbstractLauncher
{
    protected SimpleFileFilter filter = new SimpleFileFilter(true);
    protected FactoryWithArgument<Filter, String> filterFactory = new WildCardFilterFactory();

    protected static final String RESOURCE_LAYER_START = "start";

    /**
     * Entry point for cli main
     * 
     * @param args user input parameter
     * 
     * @throws Exception If unhandled exception occur
     */
    public static
    void
    main(
            final String[] args
            )
                    throws Exception
                    {
        final Main instance = new Main();
        instance.run( args );
                    }

    /* (non-Javadoc)
     * @see org.tizen.cli.exec.AbstractLauncher#getOptions()
     */
    @Override
    @SuppressWarnings("static-access")
    protected
    Options
    getOptions()
    {
        final Options opts = super.getOptions();

        opts.addOption( OptionBuilder.withLongOpt( OPT_OVERWRITE ).withDescription( DESC_OVERWRITE ).create( OPT_OVERWRITE.substring( 0, 1 ) ) );
        opts.addOption( OptionBuilder.withLongOpt( OPT_NOCHECK ).withDescription( DESC_NOCHECK ).create( OPT_NOCHECK.substring( 0, 1 ) ) );
//        opts.addOption( OptionBuilder.hasArg().withLongOpt( OPT_INCLUDE ).withDescription( DESC_INCLUDE ).create( OPT_INCLUDE.substring( 0, 1 ) ) );
//        opts.addOption( OptionBuilder.hasArg().withLongOpt( OPT_EXCLUDE ).withDescription( DESC_EXCLUDE ).create( OPT_EXCLUDE.substring( 0, 1 ) ) );

        return opts;
    }

    /* (non-Javadoc)
     * @see org.tizen.cli.exec.AbstractLauncher#execute(org.apache.commons.cli.CommandLine)
     */
    @Override
    @SuppressWarnings("unchecked")
    protected
    void
    execute(
            final CommandLine cmdLine
            )
                    throws Exception
                    {
        final List<String> args = cmdLine.getArgList();
        logger.trace( "arguments :{}", args );

        int nArgs = args.size();

        final File cwd = new File( "." ).getCanonicalFile();

        String wgtPath = cwd.getName() + "." + WebConstant.WIDGET_FILE_EXTENSION;
        if ( 0 < nArgs )
        {
            wgtPath = convertPath( args.get( 0 ) );
        }
        logger.debug( "Widget path :{}", wgtPath );

        String baseDir = convertPath( "." );
        if ( 1 < nArgs )
        {
            baseDir = convertPath( args.get( 1 ) );
        }
        logger.debug( "Base directory :{}", baseDir );

//        final String[] includes = cmdLine.getOptionValues( OPT_INCLUDE );
//        logger.debug( "Includes :{}", CollectionUtil.toString( includes ) );
//
//        String[] excludes = cmdLine.getOptionValues( OPT_EXCLUDE );
//        if ( null == includes && null == excludes )
//        {
//            excludes = getDefaultExcludes();
//        }

        String[] excludes = getDefaultExcludes();
        logger.debug( "Excludes :{}", CollectionUtil.toString( excludes ) );

        if ( !cmdLine.hasOption( OPT_NOCHECK ) )
        {
            logger.trace( "Check input" );
            if ( !exists( baseDir ) )
            {
                logger.trace( "Base directory does exist" );
                final Prompter prompter = getPrompter();

                prompter.notify( "Base directory path is wrong: " + baseDir );
                return ;

            }
            else if ( !isValidRoot( baseDir ) )
            {
                logger.trace( "Base directory is not valid root" );
                final ChoiceOption yes = new ChoiceOption( "Yes", true );
                final ChoiceOption no = new ChoiceOption( "No" );

                final Prompter prompter = getPrompter();

                final Option option = prompter.interact(
                        MessageFormat.format( ".project file doesn''t exist in ''{0}''\nDo you want to continue [Y/n]? ", baseDir ),
                        yes,
                        no
                        );

                if ( no.equals( option ) )
                {
                    prompter.notify( "Process is canceled." );
                    return ;
                }
            }
        }

        if ( cmdLine.hasOption( OPT_OVERWRITE ) )
        {
            if (exists(wgtPath)) {
                getFileHandler().removeFile( wgtPath );
                logger.info( "Old wgt file deleted" );
            } else {
                logger.info( "Old wgt file does not exist" );
            }
        } else if (exists(wgtPath)) {
            getPrompter().notify("Widget already exist.");
            return;
        }

        ResourceLayer startLayer = new ResourceLayer("start", new VirtualFileHandler());
        Resource[] resources = getResources(baseDir, startLayer, null, excludes);
        zipFiles(wgtPath, baseDir, resources);
    }

    private Resource[] getResources(String baseDir, ResourceLayer resourceLayer, String[] includes, String[] excludes) throws IOException {
        List<File> files = FileUtil.findFiles(new File(baseDir), ".*", true);
        List<Resource> resources = new ArrayList<Resource>();
        FileHandler fh = resourceLayer.getFileHandler();
        
        setIncludes(includes);
        setExcludes(excludes);

        for (File file : files) {
            String filePath = file.getPath();
            if ( !FilenameUtil.equals(filePath, baseDir)
                    && !filter.accept(baseDir, getRelativePath(baseDir, filePath))) {
                logger.debug("Ignore {}", filePath);
                continue;
            }

            Resource resource = new Resource(resourceLayer, filePath);

            String dir = null;
            if (file.isFile()) {
                dir = file.getParent();
            } else {
                dir = filePath;
            }

            if (!fh.is(dir, Attribute.EXISTS)) {
                fh.makeDirectory(dir, true);
            }

            InputStream is = null;
            try {
                is = new FileInputStream(file);
                resource.setContents(is);
                resources.add(resource);
            } finally {
                IOUtil.tryClose(is);
            }
        }
        return resources.toArray(new Resource[resources.size()]);
    }

    /**
     * Set including file name patterns
     * 
     * @param includes including file name patterns
     */
    public
    void
    setIncludes(
        final String[] includes
    )
    {
        filter.clearIncludes();
        if ( null != includes )
        {
            for ( final String include : includes )
            {
                filter.addIncludes( filterFactory.create( include ) );
            }
        }
    }
    
    /**
     * Set excluding file name patterns
     * 
     * @param excludes excluding file name patterns
     */
    public
    void
    setExcludes(
        final String[] excludes
    )
    {
        filter.clearExcludes();
        if ( null != excludes )
        {
            for ( final String exclude : excludes )
            {
                filter.addExcludes( filterFactory.create( exclude ) );
            }
        }
    }

    /**
     * Archive <code>baseDir</code> to <code>wgtPath</code> including
     * <code>includes</code> and excluding <code>excludes</code>
     * 
     * @param wgtPath wgt file path
     * @param baseDir directory to root
     * @param includes includes file pattern
     * @param excludes excludes file pattern
     */
    private void zipFiles(final String wgtPath, final String baseDir, final Resource[] resources) {
        final ZipCommand command = new ZipCommand(baseDir, resources, wgtPath);

        final Executor executor = getExecutor();
        executor.execute(command);
    }

    /**
     * Check if <code>path</code> exists
     * 
     * @param path file path
     * 
     * @return <code>true</code> if <code>path</code> exists
     * 
     * @throws IOException If <code>path</code> can't be accessed
     */
    protected
    boolean
    exists(
            final String path
            )
                    throws IOException
                    {
        return getFileHandler().is( path, Attribute.EXISTS );
                    }

    /**
     * Check if <code>path</code> is Tizen web project root
     * 
     * @param path directory path to check
     * 
     * @return <code>true</code> if <code>path</code> is Tize web project root
     * @throws IOException If <code>path</code>'s sub files can't be accessed
     */
    protected
    boolean
    isValidRoot(
            final String path
            )
                    throws IOException
                    {
        final FileHandler fileHandler = getFileHandler();
        final String projectFilePath = FilenameUtil.addTailingPath( path, ".project" );
        return fileHandler.is( projectFilePath, Attribute.EXISTS );
                    }

    /**
     * Return default excludes file patterns
     * 
     * @return default excludes file patterns
     */
    protected
    String[] getDefaultExcludes()
    {
        return new String[] { ".*", "*.wgt" };
    }

    /**
     * Archive <code>baseDir</code> to <code>wgtPath</code>
     * 
     * including <code>includes</code> and excluding <code>excludes</code>
     * 
     * @param wgtPath wgt file path
     * @param baseDir directory to root
     * @param includes includes file pattern
     * @param excludes excludes file pattern
     */
    protected
    void
    zipFiles(
            final String wgtPath,
            final String baseDir
            )
    {
        zipFiles(wgtPath, baseDir, null);
    }

    /* (non-Javadoc)
     * @see org.tizen.cli.exec.AbstractLauncher#createHelp()
     */
    @Override
    protected Help createHelp() {
        Help help = super.createHelp();

        help.setSyntax( help.getSyntax() + getSyntax() );
        return help;
    }

    /* (non-Javadoc)
     * @see org.tizen.cli.exec.AbstractLauncher#getSyntax()
     */
    @Override
    protected
    String
    getSyntax()
    {
        return " [options]";
    }
}
