/*
 * CLI - Command Line Interface
 *
 * Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact: 
 * BonYong Lee <bonyong.lee@samsung.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
 *
 */
package org.tizen.cli.exec;

import static org.tizen.common.core.command.PrompterConstants.SUPPORT_CONSOLE;
import static org.tizen.common.util.CollectionUtil.isEmpty;
import static org.tizen.common.util.StringUtil.isEmpty;
import static org.tizen.common.util.StringUtil.trim;

import java.io.BufferedReader;
import java.io.Console;
import java.io.IOException;
import java.io.PrintStream;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Map;

import org.tizen.common.core.command.Prompter;
import org.tizen.common.core.command.UserField;
import org.tizen.common.core.command.prompter.AbstractPrompter;
import org.tizen.common.core.command.prompter.ChoiceOption;
import org.tizen.common.core.command.prompter.GenericOption;
import org.tizen.common.core.command.prompter.Option;
import org.tizen.common.util.Assert;

/**
 * <p>
 * ConsolePrompter.
 * 
 * {@link Prompter} for command line interface
 * 
 * Print out using {@link PrintStream} and Read input from console( including standard input )
 * </p>
 * 
 * @author BonYong Lee{@literal <bonyong.lee@samsung.com>} (S-Core)
 */
public class
ConsolePrompter
extends AbstractPrompter
implements Prompter
{
	/**
	 * {@link PrintStream} for output
	 */
	protected final PrintStream out;
	
	/**
	 * {@link BufferedReader} for input
	 */
	protected final BufferedReader reader;
	
	/**
	 * Constructor with {@link PrintStream} and {@link Reader}
	 * 
	 * @param out {@link PrintStream} for output
	 * @param reader {@link Reader} for input
	 */
	public
	ConsolePrompter(
		final PrintStream out,
		final Reader reader
	)
	{
		Assert.notNull( out );
		this.out = out;
		this.reader = new BufferedReader( reader );
	}

	/* (non-Javadoc)
	 * @see org.tizen.common.core.command.Prompter#interact(java.lang.String, int, org.tizen.common.core.command.prompter.Option[])
	 */
	@Override
	public
	Option
	interact(
		final String question,
		final Option... options
	)
	{
		final HashSet<Option> reducedOptions = new LinkedHashSet<Option>();
		Option defaultOption = null;
		final ArrayList<String> optionNames = new ArrayList<String>();
		for ( final Option option : options )
		{
			if ( reducedOptions.contains( option ) )
			{
				throw new IllegalArgumentException( "Question can't have duplicated choice(s) :" + option );
			}
			optionNames.add( option.getName() );
			reducedOptions.add( option );
			if ( option.isDefault() )
			{
				if ( null != defaultOption )
				{
					throw new IllegalArgumentException(
						"Question must have only one default choice"
					);
				}
					
				defaultOption = option;
			}
		}
		
		String choice = null;
		try
		{
			while ( true )
			{
				printQuestion( question, options );
				
				choice = reader.readLine();
				if ( isEmpty( choice ) )
				{
					return defaultOption;
				}
				for ( final Option option : options )
				{
					if ( option.isMatch( choice ) )
					{
						return option;
					}
				}
			}
		} catch ( final IOException e )
		{
			throw new IllegalStateException( e );
		}
	}
	
	/**
	 * Print out question and options for choice
	 * 
	 * @param question message to print
	 * @param options options to suggest
	 */
	protected
	void
	printQuestion(
		final String question,
		final Option... options
	)
	{
		out.print( getFullQuestion(question, options) );
	}
	
	private char[] getFullQuestion(String question, Option[] options) {
		
		StringBuffer buffer = new StringBuffer();
		buffer.append(question + "\n");
		
		int len = options.length - 1;
		for(int i = 0; i< len; i++) {
			   if(options[i] instanceof ChoiceOption) {
					ChoiceOption choiceOption = (ChoiceOption) options[i];
					buffer.append(String.format("%s: (%s), ", choiceOption.getName(), choiceOption.getShortName()));
				}
		}
		
		if(options[len] instanceof ChoiceOption) {
			ChoiceOption choiceOption = (ChoiceOption) options[len];
			buffer.append(String.format("%s: (%s) ", choiceOption.getName(), choiceOption.getShortName()));
		}
		
		buffer.append("?");
		
		return buffer.toString().toCharArray();
	}

	/* (non-Javadoc)
	 * @see org.tizen.common.core.command.Prompter#notify(java.lang.String, java.lang.Object[])
	 */
	@Override
	public
	void
	notify(
		final String message
	)
	{
		out.println( message );
	}

	/* (non-Javadoc)
	 * @see org.tizen.common.core.command.Prompter#cancel()
	 */
	@Override
	public void cancel()
	{
	}

	/* (non-Javadoc)
	 * @see org.tizen.common.core.command.Prompter#password(java.lang.String)
	 */
	@Override
	public
	Object password(
		final String message
	)
	{
		out.print( trim( message ) );
		out.print( " " );
		Console console = System.console();
		return console.readPassword();
	}

	/* (non-Javadoc)
	 * @see org.tizen.common.core.command.Prompter#error(java.lang.String, java.lang.Object[])
	 */
	@Override
	public
	void
	error(
		final String message
	)
	{
		out.println( message );
	}

	/* (non-Javadoc)
	 * @see org.tizen.common.core.command.Prompter#batch(java.util.Collection)
	 */
	@Override
	public
	void
	batch(
		final Collection<UserField> userFields,
		final Map<String, Object> options
	)
	{
		batch( 0, userFields );
	}
	
	/**
	 * <p>
	 * Process <code>fields</code> for <code>depth</code> level
	 * 
	 * process result are stored in <code>answer</code>
	 * </p>
	 * @param depth process depth
	 * @param fields user input specifications
	 * @param answer user answer
	 */
	protected
	void
	batch(
		final int depth,
		final Collection<UserField> fields
	)
	{
		for ( final UserField child : fields )
		{
			batch( depth, child );
		}
	}
	
	/**
	 * <p>
	 * Process <code>field</code> for <code>depth</code> level
	 * 
	 * process result are stored in <code>answer</code>
	 * </p>
	 * @param depth process depth
	 * @param field user input specification
	 * @param answer user answer
	 */
	protected
	void
	batch(
		final int depth,
		final UserField field
	)
	{
		final String msg = field.getMessage();
		
		final Collection<Object> supports = field.getSupports();
		if ( null != supports && !supports.contains( SUPPORT_CONSOLE ) )
		{
			logger.warn( "{} is not supported in {}", field, this );
			return ;
		}
		
		final Class<?> type = field.getType();
		
		boolean bChild = true;
		
		if ( field.canModify() )
		{
			if ( String.class.equals( type ) )
			{
				final GenericOption option = new GenericOption();
				interact( indent( depth, msg ), option );
				if ( field.getValue() == null || !isEmpty( option.getAnswer() ) )
				{
					field.setValue( option.getAnswer() );
				}
			}
			else if ( char[].class.equals( type ) || Character[].class.equals( type ) )
			{
				field.setValue( password( indent( depth, msg ) ) );
			}
			else if ( boolean.class.equals( type ) || Boolean.class.equals( type ) )
			{
				final boolean initValue = (null == field.getValue())?true:((Boolean) field.getValue());
				final ChoiceOption yes = new ChoiceOption( "Yes", initValue );
				final ChoiceOption no = new ChoiceOption( "No", !initValue );
				final Object opt = interact( indent( depth, msg ), yes, no );
				bChild = yes.equals( opt );
				field.setValue( bChild );
			}
		}
		
		final Collection<UserField> children = field.getChildren();
		if ( bChild && !isEmpty( children ) )
		{
			notify( indent( depth, msg ) );
			batch( depth + 1, children );
		}
	}

	/**
	 * Make indented message
	 * 
	 * @param depth message depth
	 * @param msg message
	 * 
	 * @return created message
	 */
	protected
	String
	indent(
		final int depth,
		final String msg
	)
	{
		final StringBuilder buffer = new StringBuilder();
		
		for ( int i = 0 ; i<depth ; ++i )
		{
			buffer.append( " " );
		}
		buffer.append( msg );
		return buffer.toString();
	}

}
