<?xml version="1.0" encoding="utf-8"?>
<xsl:stylesheet version="1.0"
	xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

	<xsl:variable name="newline">
		<xsl:text>
</xsl:text>
	</xsl:variable>
	<xsl:variable name="tab">
		<xsl:text disable-output-escaping="yes">	</xsl:text>
	</xsl:variable>

	<xsl:template match="/">
		<!-- Apply following template only once if contacts type datasource is found -->
		<xsl:variable name="remote_datasource_count"
			select="count(//dataBinding//dataSource[@modelType='remote'])" />
		<xsl:if test="$remote_datasource_count>0">
			<xsl:apply-templates select="//dataBinding" />
		</xsl:if>
	</xsl:template>

	<xsl:template match="dataBinding">

		<xsl:text disable-output-escaping="yes">
<![CDATA[/*******************************************************************************
 * This file was generated by UI Builder.
 * This file will be auto-generated each and everytime you save your project.
 * Do not hand edit this file.
 ********************************************************************************/

#include "uib_datasource_handler_remote.h"
#include <json-glib/json-glib.h>
#include <json-glib/json-builder.h>
#include <json-glib/json-generator.h>

static uib_datasource_interface g_uib_remote_datasource;
static xmlDocPtr* remoteDatasourceRootNode = NULL;

static bool is_remote_datasource_update_databinding_cb_registered = false;
static bool isRefreshCallbackRegistered[MAX_ALLOWED_SOURCES];
static remote_ds_s *current_ds = NULL;
static remote_ds_s *registered_sources[MAX_ALLOWED_SOURCES];
static int registered_source_count = 0;

static size_t curl_write_cb(void *, size_t, size_t, void *);
static void (*refresh_databinding_cb)();
static JsonNode* prepare_datasource_tree();
static void refresh_remote_datasource(xmlDocPtr, remote_ds_s*);
static size_t curl_header_cb(void *, size_t, size_t, void *);
int curl_progress_cb(void *, double, double, double, double);
static Eina_Bool remotesource_call_job(void *);
bool parse_json_data(remote_ds_s*, xmlDocPtr*);
bool parse_xml_data(remote_ds_s*, xmlDocPtr*);

static bool populate_remote_records() {
	dlog_print(DLOG_DEBUG, LOG_TAG, "Entering %s - ", __func__);
	if (remoteDatasourceRootNode == NULL) {
		remoteDatasourceRootNode = (xmlDocPtr*) malloc(
				sizeof(xmlDocPtr) * MAX_ALLOWED_SOURCES);
		for (int i = 0; i < MAX_ALLOWED_SOURCES; ++i) {
			remoteDatasourceRootNode[i] = NULL;
		}
	}

	JsonNode * datasource_root_node = prepare_datasource_tree();
	remoteDatasourceRootNode[current_ds->registry_id] = convert_json_to_xml(
			datasource_root_node);
	json_node_free(datasource_root_node);
	dlog_print(DLOG_INFO, LOG_TAG, "%s - remote_source name %s rootnode %d",
			__func__, current_ds->source_name,
			remoteDatasourceRootNode[current_ds->registry_id]);
	dlog_print(DLOG_DEBUG, LOG_TAG, "Exiting %s - ", __func__);
	return true;
}

static void initialize_remote_datasource() {
	dlog_print(DLOG_DEBUG, LOG_TAG, "Entering %s - %s", __func__,
			current_ds->source_name);
	populate_remote_records();

	if (!isRefreshCallbackRegistered[current_ds->registry_id]) {
		isRefreshCallbackRegistered[current_ds->registry_id] = true;
		//Register callback functions to be invoked when a record changes.
		//Poller is not enabled if polling_interval is set to -1
		if (current_ds->polling_interval != -1) {
			current_ds->poller = ecore_poller_add(ECORE_POLLER_CORE,
					(current_ds->polling_interval / 1000),
					remotesource_call_job, current_ds);	//Or use ecore_timer for accurate timer. Though this one takes care of power conservation as well
			ecore_poller_poll_interval_set(ECORE_POLLER_CORE, 1);
			if (current_ds->poller == NULL) {
				dlog_print(DLOG_ERROR, LOG_TAG,
						"%s poller creation failed for remote datasource - %s",
						__func__, current_ds->source_name);
			} else {
				dlog_print(DLOG_INFO, LOG_TAG,
						"%s poller creation succeeded for remote datasource - %s",
						__func__, current_ds->source_name);
			}
		}
	}
}

void curl_cleanup(remote_ds_s *ds) {
	if (ds == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "cleanup(): ad is NULL");
		return;
	}

	if (ds->cleanup_done)
		return;

	if (ds->curl != NULL) {
		curl_easy_cleanup(ds->curl);
		dlog_print(DLOG_DEBUG, LOG_TAG, "curl_easy_cleanup(curl_handler)");
	} else
		dlog_print(DLOG_DEBUG, LOG_TAG, "cleanup(): curl_handler is NULL");

	ds->cleanup_done = true;
}

static void download_thread(void *data, Ecore_Thread *thread) {
	dlog_print(DLOG_DEBUG, LOG_TAG, "Enter %s - ", __func__);
	CURL *curl;
	remote_ds_s *this_ds = (remote_ds_s*) data;
	dlog_print(DLOG_DEBUG, LOG_TAG, "%s - curl url%s", __func__, this_ds->url);

	// Start a libcurl easy session
	curl = curl_easy_init();
	this_ds->curl = curl;
	dlog_print(DLOG_DEBUG, LOG_TAG, "curl_easy_init()");
	this_ds->downloading = true;
	this_ds->thread_running = true;
	this_ds->cleanup_done = false;
	//Reset response_data
	strcpy(this_ds->response_data, "");
	// Downloading header
	if (curl) {
		// Set options for a curl easy handle
		CURLcode error_code = curl_easy_setopt(curl, CURLOPT_URL, this_ds->url);
		dlog_print(DLOG_DEBUG, LOG_TAG,
				"curl_easy_setopt(curl, CURLOPT_URL, this_ds->url): %s (%d)",
				curl_easy_strerror(error_code), error_code);

		error_code = curl_easy_setopt(curl, CURLOPT_FILETIME, 1L);
		dlog_print(DLOG_DEBUG, LOG_TAG,
				"curl_easy_setopt(curl, CURLOPT_FILETIME, 1L): %s (%d)",
				curl_easy_strerror(error_code), error_code);

		error_code = curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1L);
		dlog_print(DLOG_DEBUG, LOG_TAG,
				"curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1L): %s (%d)",
				curl_easy_strerror(error_code), error_code);

		error_code = curl_easy_setopt(curl, CURLOPT_HEADERFUNCTION,
				curl_header_cb);
		dlog_print(DLOG_DEBUG, LOG_TAG,
				"curl_easy_setopt(curl, CURLOPT_HEADERFUNCTION, header_cb): %s (%d)",
				curl_easy_strerror(error_code), error_code);

		// Enable the built-in progress meter
		error_code = curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 0L);
		dlog_print(DLOG_DEBUG, LOG_TAG,
				"curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 0L): %s (%d)",
				curl_easy_strerror(error_code), error_code);

		// Set the progress callback
		error_code = curl_easy_setopt(curl, CURLOPT_PROGRESSFUNCTION,
				curl_progress_cb);
		dlog_print(DLOG_DEBUG, LOG_TAG,
				"curl_easy_setopt(curl, CURLOPT_PROGRESSFUNCTION, progress_cb): %s (%d)",
				curl_easy_strerror(error_code), error_code);

		// Set progress callback data
		error_code = curl_easy_setopt(curl, CURLOPT_PROGRESSDATA, this_ds);
		dlog_print(DLOG_DEBUG, LOG_TAG,
				"curl_easy_setopt(curl, CURLOPT_PROGRESSDATA, this_ds): %s (%d)",
				curl_easy_strerror(error_code), error_code);

		// Set options for a curl easy handle
		error_code = curl_easy_setopt(curl, CURLOPT_WRITEDATA, this_ds);
		dlog_print(DLOG_DEBUG, LOG_TAG,
				"curl_easy_setopt(curl, CURLOPT_WRITEDATA, this_ds): %s (%d)",
				curl_easy_strerror(error_code), error_code);

		error_code = curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION,
				curl_write_cb);
		dlog_print(DLOG_DEBUG, LOG_TAG,
				"curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, write_cb): %s (%d)",
				curl_easy_strerror(error_code), error_code);

		// Set timeout option for curl easy handle
		error_code = curl_easy_setopt(curl, CURLOPT_TIMEOUT_MS,
				this_ds->timeout_ms);
		dlog_print(DLOG_DEBUG, LOG_TAG,
				"curl_easy_setopt(curl, CURLOPT_TIMEOUT_MS, this_ds): %s (%d)",
				curl_easy_strerror(error_code), error_code);

		// Disable SSL verify for curl call
		error_code = curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, FALSE);
		dlog_print(DLOG_DEBUG, LOG_TAG,
				"curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, FALSE): %s (%d)",
				curl_easy_strerror(error_code), error_code);

		// Perform a blocking file transfer
		error_code = curl_easy_perform(curl);
		dlog_print(DLOG_DEBUG, LOG_TAG, "curl_easy_perform(curl): %s (%d)",
				curl_easy_strerror(error_code), error_code);
		if (error_code == CURLE_ABORTED_BY_CALLBACK) {
			curl_cleanup(this_ds);
			this_ds->downloading = false;
			return;
		} else if (error_code != CURLE_OK) {
			curl_cleanup(this_ds);
			this_ds->downloading = false;
			dlog_print(DLOG_DEBUG, LOG_TAG, "curl error: %s (%d)\n",
					curl_easy_strerror(error_code), error_code);
			return;
		}
	} else {
		dlog_print(DLOG_ERROR, LOG_TAG, "curl initialization failed.");
	}
	dlog_print(DLOG_DEBUG, LOG_TAG, "Exit %s - %s", __func__);
}

static void thread_end_cleanup(remote_ds_s *ds) {
	ds->downloading = false;
	ds->thread_running = false;
	curl_cleanup(ds);
	dlog_print(DLOG_DEBUG, LOG_TAG, "thread_end_cleanup(): freeing lock");
	eina_lock_release(&ds->mutex);
}

static void download_thread_cancel_cb(void *data, Ecore_Thread *thread) {
	dlog_print(DLOG_DEBUG, LOG_TAG, "Entering %s - ", __func__);

	if (data == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "data is NULL");
		return;
	}

	remote_ds_s *ds = (remote_ds_s *) data;
	thread_end_cleanup(ds);
	dlog_print(DLOG_DEBUG, LOG_TAG, "Exiting %s - ", __func__);
}

bool parse_json_data(remote_ds_s* ds, xmlDocPtr* rootNode) {
	JsonParser* parser = json_parser_new();
	bool is_data_parsing_successfull = false;
	GError* error = NULL;
	if (!json_parser_load_from_data(parser, ds->response_data,
			strlen(ds->response_data), &error)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "JSON parser error:%s", error->message);
		g_error_free(error);
	} else {
		JsonNode* response_root_node = json_node_copy(
				json_parser_get_root(parser));
		*rootNode = convert_json_to_xml(response_root_node);
		json_node_free(response_root_node);
		g_object_unref(parser);
		is_data_parsing_successfull = true;
	}
	return is_data_parsing_successfull;
}

bool parse_xml_data(remote_ds_s* ds, xmlDocPtr* rootNode) {
	bool is_data_parsing_successfull = false;
	xmlDocPtr parsedXMLDoc = xmlReadMemory(ds->response_data,
			strlen(ds->response_data), NULL, NULL, 0);
	if (parsedXMLDoc != NULL) {
		is_data_parsing_successfull = true;
		*rootNode = parsedXMLDoc;
	} else {
		dlog_print(DLOG_ERROR, LOG_TAG, "XML parser error:");
	}
	return is_data_parsing_successfull;
}

static void download_thread_end_cb(void *data, Ecore_Thread *thread) {
	remote_ds_s *ds = (remote_ds_s *) data;
	dlog_print(DLOG_DEBUG, LOG_TAG, "Entering %s - for ", __func__,
			ds->source_name);

	if (data == NULL) {
		dlog_print(DLOG_ERROR, LOG_TAG, "%s - received curl data is NULL",
				__func__);
		return;
	}
	dlog_print(DLOG_DEBUG, LOG_TAG, "%s data received - %s", __func__,
			ds->response_data);
	if (strcmp(ds->response_data, ds->prev_response_data) != 0) {
		dlog_print(DLOG_INFO, LOG_TAG, "%s New response data received for %s",
				__func__, ds->source_name);
		xmlDocPtr rootNode = NULL;
		bool is_data_parsed = false;
		if (ds->content_type == UIB_JSON) {
			is_data_parsed = parse_json_data(ds, &rootNode);
		} else if (ds->content_type == UIB_XML) {
			is_data_parsed = parse_xml_data(ds, &rootNode);
		}
		if (is_data_parsed) {
			//Copy this response for comparison
			ds->prev_response_data = realloc(ds->prev_response_data,
					strlen(ds->response_data));
			if (ds->prev_response_data == NULL) {
				/* out of memory! */
				dlog_print(DLOG_ERROR, LOG_TAG,
						"%s not enough memory (realloc returned NULL) for response caching\n");

			} else {
				memcpy(ds->prev_response_data, ds->response_data,
						strlen(ds->response_data) * sizeof(char));
			}
			//get_json_data_from_node(rootNode);
			refresh_remote_datasource(rootNode, ds);
		}
	} else {
		dlog_print(DLOG_INFO, LOG_TAG,
				"%s Received same response. Skipping databinding refresh for %s",
				__func__, ds->source_name);
	}
	ds->downloading = false;
	thread_end_cleanup(ds);
	dlog_print(DLOG_DEBUG, LOG_TAG, "Exiting %s - for ", __func__,
			ds->source_name);
}

static void download_feedback_cb(void *data, Ecore_Thread *thread,
		void *msg_data) {
	dlog_print(DLOG_DEBUG, LOG_TAG, "feedback received");
}

static Eina_Bool remotesource_call_job(void * datasource) {
	dlog_print(DLOG_DEBUG, LOG_TAG, "%s - for datasource %s", __func__,
			((remote_ds_s*) datasource)->source_name);
	remote_ds_s * this_ds = (remote_ds_s *) datasource;
	if (this_ds->thread_running == false) {
		dlog_print(DLOG_DEBUG, LOG_TAG, "%s: taking lock", __func__);
		eina_lock_take(&this_ds->mutex);
		dlog_print(DLOG_DEBUG, LOG_TAG, "%s:: lock taken", __func__);
		this_ds->thread = ecore_thread_feedback_run(download_thread,
				download_feedback_cb, download_thread_end_cb,
				download_thread_cancel_cb, this_ds, EINA_FALSE);
		if (this_ds->thread != NULL) {
			dlog_print(DLOG_DEBUG, LOG_TAG,
					"%s - new curl thread started for %s", __func__,
					this_ds->source_name);
		} else {
			dlog_print(DLOG_ERROR, LOG_TAG,
					"%s - thread creation failed for %s", __func__,
					this_ds->source_name);
		}

	} else {
		dlog_print(DLOG_INFO, LOG_TAG,
				"%s - another curl thread running for %s", __func__,
				this_ds->source_name);
	}

	return ECORE_CALLBACK_RENEW;
}

static JsonNode* prepare_datasource_tree() {

	remotesource_call_job(current_ds);

	return json_node_new(JSON_NODE_OBJECT);
}

static size_t curl_write_cb(void *buffer, size_t size, size_t nitems,
		void *data) {
	dlog_print(DLOG_DEBUG, LOG_TAG, "Entering %s - ", __func__);
	if (!data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "%s received curl data is NULL",
				__func__);
		return 0;
	}
	remote_ds_s * this_ds = (remote_ds_s *) data;
	size_t realsize = size * nitems;
	int existing_len = strlen(this_ds->response_data);
	size_t existing_size = existing_len * sizeof(char);

	char * received_response = (char*) malloc(existing_size + 1);
	strcpy(received_response, this_ds->response_data);

	this_ds->response_data = realloc(this_ds->response_data,
			existing_size + realsize + 1);
	if (this_ds->response_data == NULL) {
		/* out of memory! */
		dlog_print(DLOG_ERROR, LOG_TAG,
				"not enough memory (realloc returned NULL)\n");
		return 0;
	}
	strcpy(this_ds->response_data, received_response);
	free(received_response);
	memcpy(&(this_ds->response_data[existing_len]), buffer, realsize);
	this_ds->response_data[existing_size + realsize] = '\0';

	dlog_print(DLOG_DEBUG, LOG_TAG, "new chunk");

	this_ds->downloading = false;

	dlog_print(DLOG_DEBUG, LOG_TAG, "Exiting %s - ", __func__);

	return realsize;
}

int curl_progress_cb(void *data, double dltotal, double dlnow, double ultotal,
		double ulnow) {
	if (!data) {
		dlog_print(DLOG_ERROR, LOG_TAG, "data is NULL");
		return 0;
	}

	double progress = dlnow / dltotal;
	char str_buffer[BUFFER_SIZE];

	dlog_print(DLOG_DEBUG, LOG_TAG, "%s progress: dlnow - %d dltotal - %d",
			__func__, (int) dlnow, (int) dltotal);
// Check whether the progress is 100%
	if (dlnow == dltotal) {
		//this_ds->downloading = false;
	} else {
		snprintf(str_buffer, BUFFER_SIZE, "Download progress: %d%%",
				(int) (progress * 100));
	}
// Continue downloading
	return 0;
}

static size_t curl_header_cb(void *ptr, size_t size, size_t nitems, void *data) {
	char buffer[(size * nitems) + 1];   // +1 for NULL-termination

	snprintf(buffer, sizeof(buffer), "%s", (char *) ptr);
	dlog_print(DLOG_DEBUG, LOG_TAG, "Header: %s", buffer);

	return size * nitems;
}

void set_remote_datasource(remote_ds_s *this_ds) {
	dlog_print(DLOG_DEBUG, LOG_TAG, "Entering %s - for datasource %s", __func__,
			this_ds->source_name);
	current_ds = this_ds;
	bool already_registered = false;
	for (int i = 0; i < registered_source_count; ++i) {
		if (!strcmp(this_ds->source_name, registered_sources[i]->source_name)) {
			already_registered = true;
			break;
		}
	}
	if (!already_registered && registered_source_count < MAX_ALLOWED_SOURCES) {
		this_ds->registry_id = registered_source_count;
		registered_sources[registered_source_count++] = this_ds;
	}
	dlog_print(DLOG_DEBUG, LOG_TAG,
			"Exiting %s - for datasource %s with registry id : %d", __func__,
			this_ds->source_name, this_ds->registry_id);
}

static void cleanup_datasource() {

	is_remote_datasource_update_databinding_cb_registered = false;

    //Delete pollers and cleanup datasource
	for (int i = 0; i < registered_source_count; ++i) {
		isRefreshCallbackRegistered[i] = false;
		ecore_poller_del(registered_sources[i]->poller);
		curl_cleanup(registered_sources[i]);

		if (remoteDatasourceRootNode != NULL
				&& remoteDatasourceRootNode[i] != NULL) {
			xmlFreeDoc(remoteDatasourceRootNode[i]);
		}
	}
	free(remoteDatasourceRootNode);
	remoteDatasourceRootNode = NULL;
	curl_global_cleanup();
}

static void register_refresh_callback(void (*func_ptr)()) {
	dlog_print(DLOG_DEBUG, LOG_TAG, "%s - ", __func__);
	if (!is_remote_datasource_update_databinding_cb_registered) {
		refresh_databinding_cb = func_ptr;
		is_remote_datasource_update_databinding_cb_registered = true;
	}
}

static void** get_datasource_root_element() {
	dlog_print(DLOG_DEBUG, LOG_TAG, "Entering %s - for current datasource %s",
			__func__, current_ds->source_name);
	if ((remoteDatasourceRootNode == NULL)
			|| (remoteDatasourceRootNode[current_ds->registry_id] == NULL)) {
		initialize_remote_datasource();
	}
	dlog_print(DLOG_DEBUG, LOG_TAG, "Exiting %s - for current datasource %s",
			__func__, current_ds->source_name);
	return &remoteDatasourceRootNode[current_ds->registry_id];
}

static void refresh_remote_datasource(xmlDocPtr rootNode, remote_ds_s * ds) {
	dlog_print(DLOG_DEBUG, LOG_TAG, "Entering %s - for %s", __func__,
			ds->source_name);
	xmlDocPtr oldRemoteRootNode = remoteDatasourceRootNode[ds->registry_id];
	remoteDatasourceRootNode[ds->registry_id] = rootNode;
	if ((refresh_databinding_cb != NULL)
			&& (is_remote_datasource_update_databinding_cb_registered)) {
		(*refresh_databinding_cb)();
	}
	if (oldRemoteRootNode != NULL) {
		xmlFreeDoc(oldRemoteRootNode);
	}
	dlog_print(DLOG_DEBUG, LOG_TAG, "Exiting %s - for %s", __func__,
			ds->source_name);
}

uib_datasource_interface* uib_remote_datasource_get_instance() {
	dlog_print(DLOG_DEBUG, LOG_TAG, "%s - ", __func__);
	if (!g_uib_remote_datasource._is_init) {
		g_uib_remote_datasource._is_init = true;
		g_uib_remote_datasource.ds_type = remote_API;
		g_uib_remote_datasource.get_datasource_root_element =
				&get_datasource_root_element;
		g_uib_remote_datasource.register_datasource_update_callback =
				&register_refresh_callback;
		g_uib_remote_datasource.cleanup_datasource = &cleanup_datasource;
		CURLcode error_code = curl_global_init(CURL_GLOBAL_ALL);
		if (CURLE_OK != error_code) {
			dlog_print(DLOG_ERROR, LOG_TAG, "%s cURL initialization failed: %s",
					__func__, curl_easy_strerror(error_code));
		}
	}
	return &g_uib_remote_datasource;
}
]]>
</xsl:text>

	</xsl:template>
</xsl:stylesheet>
