=begin 
  
 SBSTool/controller.rb 
 
Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved. 
 
Contact: 
Taejun Ha <taejun.ha@samsung.com> 
Jiil Hyoun <jiil.hyoun@samsung.com> 
Donghyuk Yang <donghyuk.yang@samsung.com> 
DongHee Yang <donghee.yang@samsung.com> 
 
Licensed under the Apache License, Version 2.0 (the "License"); 
you may not use this file except in compliance with the License. 
You may obtain a copy of the License at 
 
http://www.apache.org/licenses/LICENSE-2.0 
 
Unless required by applicable law or agreed to in writing, software 
distributed under the License is distributed on an "AS IS" BASIS, 
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
See the License for the specific language governing permissions and 
limitations under the License. 
 
Contributors: 
- S-Core Co., Ltd 
=end 

require 'fileutils'

class RootstrapController
	@@installed = false
	@@apt_get_initiated = false 
    attr_accessor :type

	@rootstrap = nil

	# install SBS itself
	def initialize
        @type = File.basename(File.dirname(__FILE__))
		# if SBS is not install, install it first
		if not @@installed then
		    $log.info "Start to install SBS..."
			install_SBS
			@@installed = true
		else
		    $log.info "Skip to install SBS..."
		end
		
		# update environment variables 
       	$log.info "export PATH=#{$HOME}/sbs-install/bin:$PATH"
       	system "export PATH=#{$HOME}/sbs-install/bin:$PATH"
	end


	# install SBS
	def install_SBS
		# clean up
   		FileUtils.rm_rf "#{$HOME}/sbs-install" 
   		FileUtils.rm_rf "#{$HOME}/sbs" 
		
		# get git repository path
		git_path = get_value_from_file(File.dirname(__FILE__) + "/sbs_config", "GIT_REPOSITORY")
		if git_path.empty? then
			raise "Invalid GIT repository path or \".sbs_config\" not found"	
		end

		#  make temp directory
   		FileUtils.rm_rf $WORKING_DIR + "/.temp" 
   		FileUtils.mkdir $WORKING_DIR + "/.temp"

		# git clone
   		FileUtils.cd $WORKING_DIR + "/.temp"
        $log.info "git clone #{git_path}"
   		system "git clone #{git_path}"

		# install SBS
   		#system "git reset --hard origin"
        FileUtils.cd "sbs"
   		system "./install-sbs"

        FileUtils.cd $WORKING_DIR
	end


	# create new rootstrap
	def create_new_rootstrap( rootstrap )
		@rootstrap = rootstrap

		# create SBS default target
	    $log.info "#{$HOME}/sbs-install/bin/sbs -c"
		system "#{$HOME}/sbs-install/bin/sbs -c"
		
		# check if base target image exists 
		if not File.exist?("#{$HOME}/sbs/tools") then
			raise IOError, "raw tools directory does not exist" 
		end 
		
		# extract base image into rootstrap path
		path = @rootstrap.path
        $log.info "Copying #{$HOME}/sbs/tools to #{path}"
		FileUtils.rm_rf path
       	system "cp -rf #{$HOME}/sbs/tools #{path}"

		# create new SBOX2 target
		create_new_sbox2_target
	end


	# install packages
	def install_packages
		# install new packages
		pkg_list = @rootstrap.base_pkg_list
		pkg_list.each do |pkg|
            case pkg.protocol
            when "deb" then
				if not install_using_apt_get pkg then return false end
            when "scp" then
				if not install_using_dpkg pkg then return false end
            when "cp" then
				if not install_using_dpkg pkg then return false end
            else $log.warn "pkg protocol is unknown #{pkg.protocol}"
			end
		end
        #remove cache
        system "rm -rf #{@rootstrap.path}/var/cache/apt/archives/*.deb"
		# TODO: return installed package list
		return true
	end


	# remove packages
	def remove_packages
		@rootstrap.remove_pkg_list.each do |pkg|
			command("dpkg -r #{pkg.name}")
		end
	end


    def reload ( rootstrap )
        @rootstrap = rootstrap
        create_new_sbox2_target
    end 

	# get all package list installed
	def get_all_installed_packages
		result = []
		command_and_return("dpkg -l").each_line do |line|
            if line.start_with? "ii" then
				result.push line.strip.split[1] 
            end 
        end

		return result 
	end


	# get all file list in package
	def get_all_package_files(pkg_name)
		result = []
		command_and_return("dpkg -L #{pkg_name}").each_line do |line|
			file_path = line.strip!
			result.push file_path
		end
        
        return result
	end


	# get all dependent packages
	def get_all_dependent_packages(pkg_name)
		result = []
        command_and_return("apt-cache show #{pkg_name}").each_line do |line|
            if line.start_with? "Depends:" then 
                #remove "Depends: (xxx) , "
                line.gsub(/\([^)]*\)/,"").tr(',','').split(':')[1].split.each do |package|
					result.push package
				end
			end
        end 

        return result
	end


    # write self information to file that will save
    def write_info ( file_path )
        if not File.exist? "#{$HOME}/sbs-install/versions" then  
            $log.error "sbs is not installed "
            exit 1
        end 

        line = []
        File.open( "#{$HOME}/sbs-install/versions", "r") do |r|
            r.each_line do |info|
                line.push info.strip
            end 
        end 

        File.open( file_path, "w") do |w|
            line.each do |info|
                w.puts info
            end 
        end 
    end 


    # compare saved information file with self
    def same_as? ( file_path )
        if not File.exist? "#{$HOME}/sbs-install/versions" then  
            $log.error "sbs is not installed "
            exit 1
        end 

        o = []
        File.open( "#{$HOME}/sbs-install/versions", "r") do |origin|
            origin.each_line do |l|
                o.push l.strip
            end 
        end 

        s = []
        File.open( file_path, "r") do |saved|
            saved.each_line do |l|
                s.push l.strip 
            end 
        end 

        return o.sort == s.sort
    end 


	# PRIVATE METHODS/VARIABLES
	private

	# create new SBOX2 target
	def create_new_sbox2_target
		# remove old target if exists
		name = @rootstrap.name
		sbox2_target_config_path = "#{$HOME}/.scratchbox2/#{name}"
		if File.exist?(sbox2_target_config_path) then
			FileUtils.rm_rf sbox2_target_config_path
		end
		# get CPU transparency command of architecure
        transp = ""
		arch = @rootstrap.arch

		# create/initialze SBOX2 target
		path = @rootstrap.path
		FileUtils.cd path
		$log.info "#{$HOME}/sbs-install/bin/sb2-init -N -n -m slp2 #{transp} -t /bin -A #{arch} #{name} /bin/false"
		system "#{$HOME}/sbs-install/bin/sb2-init -N -n -m slp2 #{transp} -t /bin -A #{arch} #{name} /bin/false"
		FileUtils.cd "#{$WORKING_DIR}"

        #link .ssh folder
        if File.exist? "#{$HOME}/.ssh" then
            FileUtils.mkdir_p "#{name}/root"
            FileUtils.rm_rf "#{name}/root/.ssh"
            FileUtils.ln_sf "#{$HOME}/.ssh", "#{name}/root"
        end
	end


	# apt-get install ...
	def install_using_apt_get pkg
		# prepare for installation if first entered
		if not @@apt_get_initiated  then
			initiated_deb_installation(pkg.protocol, @rootstrap.url_list)
			@@apt_get_initiated = true
		end

		# execute apt-get command
		name = pkg.name	
        $log.info "apt-get install #{pkg.name}"
		return command "apt-get install --force-yes -y #{pkg.name}"
	end


	# dpkg -i ...
	def install_using_dpkg pkg
		# execute apt-get command
		file =  download_file_from_url pkg.protocol, pkg.url
		if not file.empty? then 
            $log.info "dpkg -i #{file}"
			return command "dpkg -i #{file}"
		end
		return true
	end



	# step for preparation
	def initiated_deb_installation (protocol, url_list)

	end


	# command to SBOX2 target
	def command(cmd)
		name = @rootstrap.name
		$log.info "#{$HOME}/sbs-install/bin/sb2 -R -t #{name} -m emulate -- #{cmd}"
		system "export LC_ALL=\"C\"; #{$HOME}/sbs-install/bin/sb2 -R -t #{name} -m emulate -- #{cmd}"
	end


	# return result of command
    def command_and_return(cmd)
		name = @rootstrap.name
        return `export LC_ALL=\"C\"; #{$HOME}/sbs-install/bin/sb2 -R -t #{name} -m emulate -- #{cmd}`
    end
end

