/*
 * context-manager
 *
 * Copyright (c) 2014 Samsung Electronics Co., Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#ifndef __CONTEXT_UTIL_JSON_H__
#define __CONTEXT_UTIL_JSON_H__

#include <sys/types.h>
#include <glib.h>

#ifdef DUMP_INTERNAL_JSON_MESSAGE
#   define dlog_json_message(jnode) \
	do { \
		char *__json_str = ctx::json::to_str((jnode)); \
		SECURE_LOGD("json: %s", __json_str); \
		free(__json_str); \
	} while(0)
#else
#   define dlog_json_message(jobj)
#endif


#define JSON_KEY_CLIENT		"Client"
#define JSON_KEY_REQ_ID		"ReqId"
#define JSON_KEY_CTX_ID		"CtxID"
#define JSON_KEY_METHOD		"Method"
#define JSON_KEY_RESULT		"Result"
#define JSON_KEY_OPTION		"Option"
#define JSON_KEY_DATA		"CtxData"
#define JSON_KEY_TIME		"Time"
#define JSON_KEY_TIME_ARRAY	"TimeArr"


typedef struct _JsonNode json_t;

namespace ctx { namespace json {

	// Allocation, conversion, deletion, and manipulation
	json_t* create(void);
	json_t* parse(const char* str);
	json_t* array_create(void);
	json_t* clone(json_t* jnode);

	char* to_str(json_t* jnode);
	void release(json_t** jnode);
	bool remove(json_t* jnode, const char* path, const char* key);

	bool set_node(json_t* jnode, const char* path, const char* key, json_t* jval);
	json_t* get_node(json_t* jnode, const char* path, const char* key);
	GList* get_keys(json_t* jnode, const char* path);

	// Primitive Types
	bool set(json_t* jnode, const char* path, const char* key, int val);
	bool set(json_t* jnode, const char* path, const char* key, int64_t val);
	bool set(json_t* jnode, const char* path, const char* key, double val);
	bool set(json_t* jnode, const char* path, const char* key, const char* val);

	bool get(json_t* jnode, const char* path, const char* key, int* val);
	bool get(json_t* jnode, const char* path, const char* key, int64_t* val);
	bool get(json_t* jnode, const char* path, const char* key, double* val);
	char* get_str(json_t* jnode, const char* path, const char* key);

	// Array Types
	bool array_add(json_t* jnode, const char* path, const char* key, int val);
	bool array_add(json_t* jnode, const char* path, const char* key, int64_t val);
	bool array_add(json_t* jnode, const char* path, const char* key, double val);

	bool set_array(json_t* jnode, const char* path, const char* key, int val[], int size);
	bool set_array(json_t* jnode, const char* path, const char* key, int64_t val[], int size);
	bool set_array(json_t* jnode, const char* path, const char* key, double val[], int size);

	bool get_array(json_t* jnode, const char* path, const char* key, int** val, int* size);
	bool get_array(json_t* jnode, const char* path, const char* key, int64_t** val, int* size);
	bool get_array(json_t* jnode, const char* path, const char* key, double** val, int* size);

} }	/* namespace ctx::json */

#endif /* __CONTEXT_UTIL_JSON_H__ */
