#ifndef __DALI_STAGE_H__
#define __DALI_STAGE_H__

/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

// INTERNAL INCLUDES
#include <dali/public-api/object/base-handle.h>
#include <dali/public-api/signals/dali-signal.h>

namespace Dali
{
/**
 * @addtogroup dali_core_common
 * @{
 */

namespace Internal DALI_INTERNAL
{
class Stage;
}

class Actor;
class Layer;
class ObjectRegistry;
class RenderTaskList;
struct Vector2;
struct Vector3;
struct Vector4;
struct KeyEvent;
struct TouchEvent;
struct WheelEvent;

/**
 * @brief The Stage is a top-level object used for displaying a tree of Actors.
 *
 * Multiple stage/window support is not currently provided.
 *
 * @since_tizen 2.4
 */
class DALI_IMPORT_API Stage : public BaseHandle
{
public:

  typedef Signal< void (const KeyEvent&)> KeyEventSignalType;     ///< Key event signal type @since_tizen 2.4
  typedef Signal< void () > EventProcessingFinishedSignalType;    ///< Event Processing finished signal type @since_tizen 2.4
  typedef Signal< void (const TouchEvent&)> TouchedSignalType;    ///< Touched signal type @since_tizen 2.4
  typedef Signal< void (const WheelEvent&)> WheelEventSignalType; ///< Touched signal type @since_tizen 2.4
  typedef Signal< void () > ContextStatusSignal;                  ///< Context status signal type @since_tizen 2.4
  typedef Signal< void () > SceneCreatedSignalType;               ///< Scene created signal type @since_tizen 2.4

  static const Vector4 DEFAULT_BACKGROUND_COLOR; ///< Default black background.
  static const Vector4 DEBUG_BACKGROUND_COLOR;   ///< Green background, useful when debugging.

  /**
   * @brief Allows the creation of an empty stage handle.
   *
   * To retrieve the current stage, this handle can be set using Stage::GetCurrent().
   * @since_tizen 2.4
   */
  Stage();

  /**
   * @brief Get the current Stage.
   *
   * @since_tizen 2.4
   * @return The current stage or an empty handle if the internal core has not been created or has been already destroyed.
   */
  static Stage GetCurrent();

  /**
   * @brief Query whether the Stage exists; this should only return false during or after destruction of Dali core.
   *
   * @since_tizen 2.4
   * @return True when it's safe to call Stage::GetCurrent().
   */
  static bool IsInstalled();

  /**
   * @brief Destructor
   *
   * This is non-virtual since derived Handle types must not contain data or virtual methods.
   * @since_tizen 2.4
   */
  ~Stage();

  /**
   * @brief This copy constructor is required for (smart) pointer semantics.
   *
   * @since_tizen 2.4
   * @param [in] handle A reference to the copied handle
   */
  Stage(const Stage& handle);

  /**
   * @brief This assignment operator is required for (smart) pointer semantics.
   *
   * @since_tizen 2.4
   * @param [in] rhs  A reference to the copied handle
   * @return A reference to this
   */
  Stage& operator=(const Stage& rhs);

  // Containment

  /**
   * @brief Adds a child Actor to the Stage.
   *
   * The child will be referenced.
   * @since_tizen 2.4
   * @param [in] actor The child.
   * @pre The actor has been initialized.
   * @pre The actor does not have a parent.
   */
  void Add(Actor& actor);

  /**
   * @brief Removes a child Actor from the Stage.
   *
   * The child will be unreferenced.
   * @since_tizen 2.4
   * @param [in] actor The child.
   * @pre The actor has been added to the stage.
   */
  void Remove(Actor& actor);

  /**
   * @brief Returns the size of the Stage in pixels as a Vector.
   *
   * The x component will be the width of the Stage in pixels
   * The y component will be the height of the Stage in pixels
   * The z component will be the distance between far and near planes
   * @since_tizen 2.4
   * @return The size of the Stage as a Vector.
   */
  Vector2 GetSize() const;

  // Render Tasks

  /**
   * @brief Retrieve the list of render-tasks.
   *
   * @since_tizen 2.4
   * @return A valid handle to a RenderTaskList.
   */
  RenderTaskList GetRenderTaskList() const;

  // Layers

  /**
   * @brief Query the number of on-stage layers.
   *
   * Note that a default layer is always provided (count >= 1).
   * @since_tizen 2.4
   * @return The number of layers.
   */
  unsigned int GetLayerCount() const;

  /**
   * @brief Retrieve the layer at a specified depth.
   *
   * @since_tizen 2.4
   * @param[in] depth The depth.
   * @return The layer found at the given depth.
   * @pre Depth is less than layer count; see GetLayerCount().
   */
  Layer GetLayer(unsigned int depth) const;

  /**
   * @brief Returns the Stage's Root Layer.
   *
   * @since_tizen 2.4
   * @return The root layer.
   */
  Layer GetRootLayer() const;

  // Background color

  /**
   * @brief Set the background color of the stage.
   *
   * @since_tizen 2.4
   * @param[in] color The new background color.
   */
  void SetBackgroundColor(Vector4 color);

  /**
   * @brief Retrieve the background color of the stage.
   *
   * @since_tizen 2.4
   * @return The background color.
   */
  Vector4 GetBackgroundColor() const;

  /**
   * @brief Retrieve the DPI of the display device to which the stage is connected.
   *
   * @since_tizen 2.4
   * @return The horizontal and vertical DPI
   */
  Vector2 GetDpi() const;

  /**
   * @brief Get the Object registry.
   *
   * @since_tizen 2.4
   * @return The object registry.
   */
  ObjectRegistry GetObjectRegistry() const;

  // Rendering

  /**
   * @brief Keep rendering for at least the given amount of time.
   *
   * By default Dali will stop rendering when no Actor positions are being set, and when no animations are running etc.
   * This method is useful to force screen refreshes e.g. when updating a NativeImage.
   * @since_tizen 2.4
   * @param[i] durationSeconds Time to keep rendering, 0 means render at least one more frame
   */
  void KeepRendering( float durationSeconds );

  // Signals

  /**
   * @brief This signal is emitted when key event is received.
   *
   * A callback of the following type may be connected:
   * @code
   *   void YourCallbackName(const KeyEvent& event);
   * @endcode
   * @since_tizen 2.4
   * @return The signal to connect to.
   */
  KeyEventSignalType& KeyEventSignal();

  /**
   * @brief This signal is emitted just after the event processing is finished.
   *
   * @since_tizen 2.4
   * @return The signal to connect to.
   */
  EventProcessingFinishedSignalType& EventProcessingFinishedSignal();

  /**
   * @brief This signal is emitted when the screen is touched and when the touch ends
   * (i.e. the down & up touch events only).
   *
   * If there are multiple touch points, then this will be emitted when the first touch occurs and
   * then when the last finger is lifted.
   * An interrupted event will also be emitted.
   * A callback of the following type may be connected:
   * @code
   *   void YourCallbackName(const TouchEvent& event);
   * @endcode
   *
   * @since_tizen 2.4
   * @return The touch signal to connect to.
   * @note Motion events are not emitted.
   */
  TouchedSignalType& TouchedSignal();

  /**
   * @brief This signal is emitted when wheel event is received.
   *
   * A callback of the following type may be connected:
   * @code
   *   void YourCallbackName(const WheelEvent& event);
   * @endcode
   * @since_tizen 2.4
   * @return The signal to connect to.
   */
  WheelEventSignalType& WheelEventSignal();

  /**
   * @brief This signal is emitted when the GL context is lost (Platform specific behaviour).
   *
   * If the application is responsible for handling context loss, it should listen to
   * this signal and tear down UI components when recieved.
   * @since_tizen 2.4
   * @return The context lost signal to connect to.
   */
  ContextStatusSignal& ContextLostSignal();

  /**
   * @brief This signal is emitted when the GL context is regained (Platform specific
   * behaviour).
   *
   * If the application is responsible for handling context loss, it should listen to
   * this signal and rebuild UI components on receipt.
   * @since_tizen 2.4
   * @return The context regained signal to connect to.
   */
  ContextStatusSignal& ContextRegainedSignal();

  /**
   * @brief This signal is emitted after the initial scene is created.
   *
   * It will be triggered after the
   * application init signal.
   *
   * A callback of the following type may be connected:
   * @code
   *   void YourCallbackName();
   * @endcode
   * @since_tizen 2.4
   * @return The signal to connect to.
   */
  SceneCreatedSignalType& SceneCreatedSignal();

public: // Not intended for application developers

  /**
   * @brief This constructor is used by Stage::GetCurrent() methods.
   *
   * @since_tizen 2.4
   * @param [in] stage A pointer to a Dali resource
   */
  explicit DALI_INTERNAL Stage(Internal::Stage* stage);
};

/**
 * @}
 */

} // namespace Dali

#endif // __DALI_STAGE_H__
