#ifndef __DALI_PROPERTY_H__
#define __DALI_PROPERTY_H__

/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

// EXTERNAL INCLUDES
#include <string>
#include <utility>

// INTERNAL INCLUDES
#include <dali/public-api/common/vector-wrapper.h>
#include <dali/public-api/common/dali-common.h>
#include <dali/public-api/common/dali-vector.h>

namespace Dali
{
/**
 * @addtogroup dali_core_object
 * @{
 */

class Handle;

/**
 * @brief An object + property pair.
 * @since_tizen 2.4
 */
struct DALI_IMPORT_API Property
{
  /**
   * @brief A valid property index is zero or greater.
   * @since_tizen 2.4
   */
  typedef int Index;

  static const int INVALID_INDEX; ///< -1 is not a valid property index
  static const int INVALID_COMPONENT_INDEX; ///< -1 is not a valid property index

  typedef Dali::Vector< Index > IndexContainer; ///< A vector of property indices @since_tizen 2.4

  /**
   * @brief A value-type representing a property value.
   */
  class Value;

  /**
   * @brief A Map of property values.
   */
  class Map;

  /**
   * @brief An Array of property values.
   */
  class Array;

  /**
   * @brief The property types supported.
   * @since_tizen 2.4
   */
  enum Type
  {
    NONE,            ///< No type @since_tizen 2.4

    BOOLEAN,         ///< A boolean type @since_tizen 2.4
    FLOAT,           ///< A float type @since_tizen 2.4
    INTEGER,         ///< An integer type @since_tizen 2.4
    VECTOR2,         ///< a vector array of size=2 with float precision @since_tizen 2.4
    VECTOR3,         ///< a vector array of size=3 with float precision @since_tizen 2.4
    VECTOR4,         ///< a vector array of size=4 with float precision @since_tizen 2.4
    MATRIX3,         ///< a 3x3 matrix @since_tizen 2.4
    MATRIX,          ///< a 4x4 matrix @since_tizen 2.4
    RECTANGLE,       ///< an integer array of size=4 @since_tizen 2.4
    ROTATION,        ///< either a quaternion or an axis angle rotation @since_tizen 2.4
    STRING,          ///< A string type @since_tizen 2.4
    ARRAY,           ///< an array of Property::Value @since_tizen 2.4
    MAP              ///< a string key to Property:value mapping @since_tizen 2.4
  };

  /**
   * @brief The access mode for custom properties
   * @since_tizen 2.4
   */
  enum AccessMode
  {
    READ_ONLY,          ///< if the property is read-only @since_tizen 2.4
    READ_WRITE,         ///< If the property is read/writeable @since_tizen 2.4
    ANIMATABLE,         ///< If the property can be animated or constrained @since_tizen 2.4
    ACCESS_MODE_COUNT   ///< The number of access modes @since_tizen 2.4
  };


  /**
   * @brief Create a Property instance.
   *
   * @since_tizen 2.4
   * @param [in] object A valid handle to the target object.
   * @param [in] propertyIndex The index of a property.
   */
  Property( Handle& object, Property::Index propertyIndex );


  /**
   * @brief Create a Property instance.
   *
   * @since_tizen 2.4
   * @param [in] object A valid handle to the target object.
   * @param [in] propertyIndex The index of a property.
   * @param [in] componentIndex Index to a sub component of a property, for use with Vector2, Vector3 and Vector4. -1 for main property (default is -1)
   */
  Property( Handle& object, Property::Index propertyIndex, int componentIndex );

  /**
   * @brief Create a Property instance.
   *
   * @since_tizen 2.4
   * @param [in] object A valid handle to the target object.
   * @param [in] propertyName The property name.
   * @note This performs a property index query and is therefore slower than
   * constructing a Property directly with the index.
   */
  Property( Handle& object, const std::string& propertyName );

  /**
   * @brief Create a Property instance.
   *
   * @since_tizen 2.4
   * @param [in] object A valid handle to the target object.
   * @param [in] propertyName The property name.
   * @param [in] componentIndex Index to a sub component of a property, for use with Vector2, Vector3 and Vector4. -1 for main property (default is -1)
   * @note This performs a property index query and is therefore slower than
   * constructing a Property directly with the index.
   */
  Property( Handle& object, const std::string& propertyName, int componentIndex );

  /**
   * @brief Non-virtual destructor; Property is not intended as a base class.
   * @since_tizen 2.4
   */
  ~Property();

  Handle& object; ///< A valid handle to the target object.

  Index propertyIndex; ///< The index of a property provided by object.

  int componentIndex; ///< Index of a property sub component, for use with Vector2, Vector3 and Vector4, -1 if using main property
};

/**
 * @}
 */
} // namespace Dali

#endif // __DALI_PROPERTY_H__
